#!/usr/bin/python
# -*- coding: utf-8 -*-

# This is a very basic library to use a Yocto-Color-V2 with
# eight 8x8 RGB LED panels to make a 64x8 full-color LED display.
# it  uses a small Yoctopuce bitmap font file to render
# any string using ISO-latin-1 character set

import os,  struct, collections

from yocto_api import YAPI, YRefParam
from yocto_colorledcluster import YColorLedCluster



class BitmapFont:
    # Constructor: load and parse a yfm font file
    def __init__(self, yfm_filename):
        self.height = 0
        if not os.path.exists(yfm_filename):
            print("Font file not found:", yfm_filename)
            return
        with open(yfm_filename, 'rb') as file:
            yfm = file.read()
        fields = 'sig ver bpp w h baseline first last'
        YFSHeader = collections.namedtuple('YFSHeader', fields)
        header = YFSHeader._make(struct.unpack("<2sBBHBBBB", yfm[:10]))
        if header.sig != b'YF':
            print("Invalid font file, Yoctopuce yfm file format expected")
            return
        firstc = int(header.first)
        nchar = int(header.last) - firstc + 1
        lsize = int(header.w/8)
        bpos = 10 + 2 * nchar
        fpos = struct.unpack("<"+str(nchar)+"H", yfm[10:bpos])
        # build character-indexed column bitmaps
        self.height = int(header.h)
        self.cwidth = [0] * 256
        self.cbitmap = [[]] * 256
        cidx = 0
        curr_bitmap = [[] for i in range(self.height)]
        for col in range(lsize):
            for bit in range(8):
                ofs = bpos + col
                for y in range(self.height-1, -1, -1):
                    if ((yfm[ofs] << bit) & 0x80) != 0:
                        curr_bitmap[y].append(1)
                    else:
                        curr_bitmap[y].append(0)
                    ofs += lsize
                if 8*col + bit == fpos[cidx]:
                    ch = firstc + cidx
                    self.cwidth[ch] = len(curr_bitmap[0])
                    self.cbitmap[ch] = curr_bitmap
                    curr_bitmap = [[] for i in range(self.height)]
                    cidx = cidx + 1
                    if cidx >= len(fpos): return

    # Tell if a font file was properly loaded
    def isValid(self):
        return self.height > 0

    # Compute the pixel height of a given string
    def stringHeight(self, text):
        return self.height

    # Compute the pixel width of a given string
    def stringWidth(self, text):
        res = 0
        for i in range(len(text)):
            res = res + self.cwidth[ord(text[i])]
        return res

    # Draw a string in a given bi-dimensional RGB buffer
    # Buffer is made of vertical arrays of pixels

    def drawString(self, text, buffer, x, y, maxWidth, color):
        for idx in range(len(text)):
            c = ord(text[idx])
            cw = self.cwidth[c]
            bitmap = self.cbitmap[c]
            for rx in range(cw):
                for ry in range(self.height):
                    if bitmap[ry][rx] != 0:
                      if x<maxWidth :   buffer[x][y+ry] = color
                x += 1


class YoctoColorDisplay:
  ledCluster = None
  font         = None
  displayBuffer =[]
  displayWidth = 0
  displayHeight = 8
  def __init__(self,ledCluster):
     self.ledCluster =  ledCluster
     self.font = BitmapFont ("fonts/Small.yfm")
     self.displayWidth=  1+  int((self.ledCluster.get_activeLedCount() -1 ) /self.displayHeight)
     self.clearScr()

  def clearBuffer(self):
    self.displayBuffer = [[0] * self.displayHeight for i in range(self.displayWidth)]

  def clearScr(self):
    self.clearBuffer()
    self.__showBitmap( self.displayBuffer, 0)

  # Draw a bi-dimensional RGB array to a colorLedCluster
  # with a shift offset, including clipping if needed
  def __showBitmap(self, buffer, x = 0, y = 0):

    RGB = [0] * (self.displayWidth * self.displayHeight)
    xstart = ystart = 0
    xstop = len(buffer)
    ystop = len(buffer[0])
    if x < 0:
        xstart = -x
        x = 0
    if y < 0:
        ystart = -y
        y = 0
    if x + xstop - xstart >= self.displayWidth:
        xstop = self.displayWidth - x + xstart
    if y + ystop - ystart >= self.displayHeight:
        ystop = self.displayHeight - y + ystart
    ofs = x * self.displayHeight + y
    dy = ystop - ystart
    for rx in range(xstart, xstop):
        RGB[ofs:ofs+dy] = buffer[rx][ystart:ystop]
        ofs += self.displayHeight
    self.ledCluster.set_rgbColorArray(0, RGB)

  def drawText(self,text,xpos,color,updateDisplay=True):
    strHeight = self.font.stringHeight(text)
    strWidth = self.font.stringWidth(text)
    self.font.drawString(text, self.displayBuffer, xpos, 0, self.displayWidth,color)
    if updateDisplay : self.__showBitmap( self.displayBuffer, 0)


