<?php
include_once("datamatrix.php");

define("BROTHER_PRINTER_PORT","9100");
define("BROTHER_LEFT_MARGIN",44);
define("BROTHER_PRINT_AREA",1200);
define("BROTHER_RIGHT_MARGIN",52);

// Get status of Brother 103mm label printer
function BrotherGetStatus($printerIP)
{
    // Make sure the printer is turned on using an UDP packet
    $socket = socket_create(AF_INET, SOCK_DGRAM, SOL_UDP);
    if (!$socket) {
        return "Failed to create UDP socket";
    }
    socket_set_nonblock($socket);
    $tftpreq = "\0\1ping.txt\0octet\0";
    socket_sendto($socket, $tftpreq, strlen($tftpreq), 0 , $printerIP, 69);
    $read = array($socket); $write = NULL; $except = NULL;
    if(socket_select($read, $write, $except, 3) == 0) {
        return "Brother printer did not reply to TFTP ping";
    }
    socket_close($socket);
    // Request status using Microsoft Web Service API
    $url = 'http://'.$printerIP.':80/WebServices/PrinterService';
    $xml = '
<soap:Envelope xmlns:soap="http://www.w3.org/2003/05/soap-envelope" xmlns:pri="http://schemas.microsoft.com/windows/2006/08/wdp/print" xmlns:wsa="http://schemas.xmlsoap.org/ws/2004/08/addressing">
   <soap:Header>
      <wsa:To>'.$url.'</wsa:To>
      <wsa:Action>http://schemas.microsoft.com/windows/2006/08/wdp/print/GetPrinterElements</wsa:Action>
      <wsa:MessageID>urn:uuid:871a75b9-eec8-48f2-a089-8e553fbbfffb</wsa:MessageID>
      <wsa:ReplyTo>
         <wsa:Address>http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous</wsa:Address>
      </wsa:ReplyTo>
      <wsa:From>
         <wsa:Address>urn:uuid:303d8c17-91a6-40ee-a78c-1bb1f2a55971</wsa:Address>
      </wsa:From>
   </soap:Header>
   <soap:Body>
      <pri:GetPrinterElementsRequest>
         <pri:RequestedElements>
            <pri:Name>pri:PrinterStatus</pri:Name>
         </pri:RequestedElements>
      </pri:GetPrinterElementsRequest>
   </soap:Body>
</soap:Envelope>
';
    $options = [
        'http' => [
            'header'  => "Content-type: application/xml\r\n",
            'method'  => 'POST',
            'content' => utf8_encode($xml)
        ]
    ];
    $context  = stream_context_create($options);
    $contents = file_get_contents($url, false, $context);
    if ($contents === FALSE) { return "Brother printer not responding, check power !"; }
    $namespaces = Array('soap:','xmlns:','xsi:','wsa:','SOAP-ENV:','wprt:');
    $result = simplexml_load_string(str_replace($namespaces,'',$contents));
    if($result === false) {
        $contents .= "\n\n==> Failed loading XML:\n";
        foreach(libxml_get_errors() as $error) {
            $contents .= $error->message."\n";
        }
        return "unable to parse Brother status";
    }
    if(!$result->Body->GetPrinterElementsResponse ||
        !$result->Body->GetPrinterElementsResponse->PrinterElements ||
        !$result->Body->GetPrinterElementsResponse->PrinterElements->ElementData) {
        return "Unexpected Brother status reply";
    }
    $status = $result->Body->GetPrinterElementsResponse->PrinterElements->ElementData->PrinterStatus;
    if(utf8_decode($status->PrinterState) == 'Idle') return "OK";
    $res = 'Printer '.utf8_decode($status->PrinterState).': '.utf8_decode($status->PrinterPrimaryStateReason);
    return $res;
}

// Setup communication with Brother printer
function BrotherOpen($printerIP, &$errmsg)
{
    $fp = @fsockopen($printerIP, BROTHER_PRINTER_PORT, $errno, $errstr, 5);
    if (!$fp) {
        $errmsg="cannot connect to Brother printer at $printerIP ($errstr)";
        return false;
    }
    stream_set_timeout ( $fp, 25);
    if (feof($fp)) {
        $errmsg="socket is closed" ;
        return false;
    }
    // Prepare to print bitmap on continuous 103.7mm paper
    // On each raw, will send will 58 white + 1200 pixels + 38 white = 1296 bits
    $width = BROTHER_LEFT_MARGIN+BROTHER_PRINT_AREA+BROTHER_RIGHT_MARGIN;
    $esc = chr(27);
    $reset = "{$esc}@{$esc}ia\1";
    $disableNotification = "{$esc}i!\1";
    $setupMedia = "{$esc}iz\306\010\150".chr($width&0xff).chr($width>>8)."\0\0\0\0";
    $setupCut = "{$esc}iM{$esc}iA\1@{$esc}iK\10";
    $setupMargin = "{$esc}id\0\0";
    $enableCompression = "M\2";
    fwrite($fp, "{$reset}{$disableNotification}{$setupMedia}{$setupCut}{$setupMargin}{$enableCompression}");
    if (feof($fp)) {
        $errmsg="socket is closed";
        return false;
    }
    return $fp;
}

function BrotherSendRow($fp, $data, &$errmsg)
{
    if(strlen($data) == 6) {
        // Shorten big horizontal line in (R) Etranger labels
        if(ord($data[0]) == 0xb6 && ord($data[1]) == 0x00 && ord($data[2]) == 0x00 &&
           ord($data[3]) == 0x20 && ord($data[4]) == 0xab && ord($data[5]) == 0x00) {
            // drop row
            return true;
        }
    }
    fwrite($fp, "g\0".chr(strlen($data)).$data);
    if (feof($fp)) {
        $errmsg="socket is closed";
        return false;
    }
    return true;
}

function BrotherFlushAndClose($printerIP, $fp, $printHeight, $pad_mm, &$errmsg)
{
    $disableCompression = "M\0";
    $pad = '';
    for($i = 1; $i < $pad_mm; $i++) $pad .= 'ZZZZZZZZZZZZ'; // 12 blank lines
    $printHeight += 12 * $pad_mm;
    $eof = chr(0x1a);
    fwrite($fp, "{$disableCompression}{$pad}{$eof}");
    fclose($fp);
    // Wait until print is complete before announcing final OK status
    while($printHeight > 0) {
        sleep(1);
        $printerStatus = BrotherGetStatus($printerIP);
        if($printerStatus != 'OK') {
            $errmsg = $printerStatus;
            return false;
        }
        $printHeight -= 200;
    }
    return true;
}

/*
 * Print a GD image to Brother thermal printer, via direct TCP connection
 */
function BrotherPrintImage($printerIP, $image, $pad_mm, &$errmsg)
{
    $printerStatus = BrotherGetStatus($printerIP);
    if($printerStatus != 'OK') {
        $errmsg = $printerStatus;
        return false;
    }
    if(!$image) {
        $errmsg = 'Argument is not a GD image';
        return false;
    }
    $width = imagesx($image);
    $height = imagesy($image);
    if(!$width || !$height || $width < 128 || $height < 128) {
        $errmsg = 'Image is too small, this is probably an error';
        return false;
    }

    // convert the image to monochrome bitmap data through wbmp format
    // perform full sanity checks to ensure we understand data properly
    imagefilter($image, IMG_FILTER_GRAYSCALE);
    imagefilter($image, IMG_FILTER_BRIGHTNESS, -100);
    imagefilter($image, IMG_FILTER_CONTRAST, -100);
    ob_start();
    imagewbmp($image);
    $buffer = ob_get_contents();
    ob_end_clean();
    if(!$buffer || strlen($buffer) < 2048) { // 128 x 128
        $errmsg = 'Bad WBMP buffer ! is GD compiled with WBMP support ?';
        return false;
    }
    if(ord($buffer[0]) != 0 || ord($buffer[1]) != 0) {
        $errmsg = sprintf('Bad WBMP header: %02x %02x ?!?', $buffer[0], $buffer[1]);
        return false;
    }
    $pos = 2;
    $chkwidth = ord($buffer[$pos++]) & 0x7f;
    while(ord($buffer[$pos-1]) & 0x80) {
        $chkwidth = ($chkwidth << 7) + (ord($buffer[$pos++]) & 0x7f);
    }
    $chkheight = ord($buffer[$pos++]) & 0x7f;
    while(ord($buffer[$pos-1]) & 0x80) {
        $chkheight = ($chkheight << 7) + (ord($buffer[$pos++]) & 0x7f);
    }
    if($chkwidth != $width || $chkheight != $height) {
        $errmsg = sprintf('WBMP size mismatch: %d x %d versus %d x %d ?!?', $chkwidth, $chkheight, $width, $height);
        return false;
    }

    // crop white lines at top and right of image
    $srcRowLen = intVal(($width+7)/8);
    //Print("Original width=$width height=$height<br>\n");
    while($height > 0) {
        for($i = 0; $i < $srcRowLen-1; $i++) {
            if(ord($buffer[$pos+$i]) != 0xff) break;
        }
        if($i < $srcRowLen-1) break;
        $pos += $srcRowLen;
        $height--;
    }
    while($width > 0) {
        $endline = $pos + (($width-8) >> 3);
        for($i = 0; $i < $height; $i++) {
            if(ord($buffer[$endline+$i*$srcRowLen]) != 0xff) break;
        }
        if($i < $height) break;
        $width -= 8;
    }
    //Print("White-cropped width=$width height=$height<br>\n");

    // truncate image height to 1200 pixels in any case (paper width)
    if($height >= BROTHER_PRINT_AREA) {
        $height = BROTHER_PRINT_AREA;
        $lmargin = BROTHER_LEFT_MARGIN;
        $rmargin = BROTHER_RIGHT_MARGIN;
    } else {
        $extra = BROTHER_PRINT_AREA - $height;
        $lmargin = BROTHER_LEFT_MARGIN + ($extra >> 1);
        $rmargin = BROTHER_RIGHT_MARGIN + (($extra+1) >> 1);
    }

    // process data vertically to generate landscape mode output
    $fp = BrotherOpen($printerIP, $errmsg);
    if(!$fp) {
        return false;
    }
    for($x = 0; $x < $width; $x++) {
        $srcBit = (0x80 >> ($x & 7));
        $row = '';
        $prevlitt = 0;
        $prevbyte = 0;
        $prevrep = ($lmargin >> 3);
        $dstBit = 0x80 >> ($lmargin & 7);
        $byte = 0;
        for($y = 0; $y < $height+$rmargin; $y++) {
            if($y < $height) {
                $ofs = $pos + ($y * $srcRowLen);
                if((ord($buffer[$ofs]) & $srcBit) == 0) $byte |= $dstBit;
            }
            $dstBit >>= 1;
            if($dstBit == 0) {
                if($prevbyte == $byte && $prevrep < 127) {
                    $prevrep++;
                } else {
                    if($prevrep == 1 && $prevlitt > 0 && $prevlitt < 127) {
                        $row[strlen($row)-1-$prevlitt] = chr($prevlitt);
                        $row .= chr($prevbyte);
                        $prevlitt++;
                    } else {
                        $prevlitt = ($prevrep == 1 ? 1 : 0);
                        $row .= chr(1 - $prevrep) . chr($prevbyte);
                    }
                    $prevbyte = $byte;
                    $prevrep = 1;
                }
                $dstBit = 0x80;
                $byte = 0;
            }
        }
        $row .= chr(1-$prevrep).chr($prevbyte);
        if(($x & 7) == 7) $pos++;
        //Print("row: ".bin2hex($row)."<br>\n");
        if(!BrotherSendRow($fp, $row, $errmsg)) {
            break;
        }
    }
    return BrotherFlushAndClose($printerIP, $fp, $height, $pad_mm, $errmsg);
}

// Return a newly allocated GD image object for the requested MH10.8 DataMatrix 2D barcode
function ImageDataMatrix($fields, $scale = 1)
{
    // Step 1: Pack fields into a MH10.8 string
    $GS = chr(29);
    $RS = chr(30);
    $EOT = chr(4);
    // Start with MH10.8 header
    $mh10 = '[)>'.$RS.'06';
    foreach($fields as $field) {
        // append each MH10.8 encoded field
        $mh10 .= $GS.$field['code'].$field['value'];
    }
    // Append MH10.8 trailer
    $mh10 .= $RS.$EOT;

    // 2. Use TCPDF DataMatrix support code to determine proper pixmap
    $datamatrix = new Datamatrix($mh10);
    $barcode = $datamatrix->getBarcodeArray();
    $width = $barcode['num_cols'];
    $height = $barcode['num_rows'];

    // Create a GD image with the 2D barcode
    $image = imagecreate($width * $scale, $height * $scale);
    $white = imagecolorallocate($image, 255, 255, 255);
    $black = imagecolorallocate($image, 0, 0, 0);
    for($y = 0; $y < $height; $y++) {
        $row = $barcode['bcode'][$y];
        $y1 = $y * $scale;
        $y2 = $y1 + $scale-1;
        for($x = 0; $x < $width; $x++) {
            if($row[$x]) {
                $x1 = $x * $scale;
                imagefilledrectangle($image, $x1, $y1, $x1+$scale-1, $y2, $black);
            }
        }
    }

    return $image;
}

function printDataMatrixLabel($printerip, $fields, $imagefile = '')
{
    $frameWidth = 1200;
    $frameHeight = 760;
    $label = imagecreate($frameWidth, $frameHeight+96);
    $white = imagecolorallocate($label, 255, 255, 255);
    $black = imagecolorallocate($label, 0, 0, 0);
    // If available, add the part image first in background
    if($imagefile) {
        $image = imagecreatefrompng($imagefile);
        $srcH = imagesy($image);
        $srcW = imagesx($image);
        Print("Using image {$imagefile} ({$srcW} x {$srcH})\n<br>\n");
        $dstH = 350;
        $dstW = round($dstH * $srcW / $srcH);
        imagecopyresampled($label, $image, $frameWidth - 10 - $dstW, $frameHeight - 10 - $dstH,
            0, 0, $dstW, $dstH, $srcW, $srcH);
    }
    // Add the DataMatrix 2D barcode
    $barcode = ImageDataMatrix($fields, 5);
    imagecopy($label, $barcode, 80, 80, 0, 0, imagesx($barcode), imagesy($barcode));
    imagedestroy($barcode);
    // Then add text
    $font = "ttfonts/DejaVuSans.ttf";
    $bold = "ttfonts/DejaVuSans-Bold.ttf";
    $title = '';
    imagesetthickness($label, 2);
    imagerectangle($label, 3, 3, $frameWidth-3, $frameHeight-3, $black);
    imagesetthickness($label, 1);
    $vofs = 160;
    $leftMargin = 300;
    $descWidth = 330;
    foreach($fields as $field) {
        imagettftext($label, 24, 0, $leftMargin, $vofs, $black, $font, '('.$field['code'].') '.$field['desc'].':');
        imagettftext($label, 24, 0, $leftMargin+$descWidth, $vofs, $black, $font, $field['value']);
        $vofs += 50;
        if($field['code'] == '1P') {
            $title = $field['value'];
        }
    }
    // Add title to the label
    imagettftext($label, 32, 0, $leftMargin, 90, $black, $bold, $title);
    // Rotate the image 90 degrees, as expected by the printing code
    $rotated = imagerotate($label, 90, 0);
    if(!BrotherPrintImage($printerip, $rotated, 0, $errmsg)) {
        Print('ERROR: '.$errmsg);
        return false;
    }
    imagedestroy($label);
}
