#!/usr/bin/python
# -*- coding: utf-8 -*-

# basic  library to drive a RD6006 power supply with a Yocto-Serial
# connected to the internal TTL port. You will need the Yoctopuce
# library for python, available on www.yoctopuce.com/EN/libraries.php

# basically this library just accesses the right modbus registers
# on the RD6006. The full register list is available here:
# github.com/Baldanos/rd6006/blob/master/registers.md
#
# For more details about interfacing a RD6006 with Yoctopuce devices,
# have a look here:
# www.yoctopuce.com/EN/article/rd6006-adding-a-wired-network-connection

import datetime,zlib
from yocto_serialport import *
from yocto_genericsensor import *
from yocto_files import *


class YRD6006:
    _serial     = None
    _slaveId    = 0
    _module     = None
    _genSensor1 = None
    _genSensor2 = None
    _serialMode= "115200,8N1"
    _serialProtocol = "Modbus-RTU"
    _serialVoltageLevel = YSerialPort.VOLTAGELEVEL_TTL3V
    _sensorLogFrequency = "5/s"
    _lastVoltageValue = float('nan')
    _lastCurrentValue = float('nan')
    _jobFileName ="RD6006.job"
    _jobFile = '{"task1":{"timeout":500,"interval":200,"script":[{"writeMODBUS":"0103000A0002"},{"expect":":010304($cV:WORD)($cI:WORD).*"},{"compute":"$1=$cV/100"},{"compute":"$2=$cI/1000"}]}}'
    _lastUpdate = datetime.datetime.now()

    # Constructor. ySerialPort is the instance the Serial port hosted on the Yocto-serial
    # which is  connected to  the RD6006 internal TTL serial port
    def __init__(self, ySerialPort, slaveId=1):
       self._serial = ySerialPort
       self._slaveId = slaveId;
       self._module = self._serial.get_module()
       serialNumber = self._module.get_serialNumber()
       self._genSensor1 =YGenericSensor.FindGenericSensor(serialNumber+".genericSensor1")
       self._genSensor1.registerTimedReportCallback(self._voltageChange)
       self._genSensor2 =YGenericSensor.FindGenericSensor(serialNumber+".genericSensor2")
       self._genSensor2.registerTimedReportCallback(self._currentChange)
       self._serial.reset()

    # internal, automatically called 5 times per seconds
    def _voltageChange(self,src,value):
       self._lastVoltageValue =  float(value.get_averageValue())
       self._lastUpdate = datetime.datetime.now()

    # internal, automatically called 5 times per seconds
    def _currentChange(self,src,value):
       self._lastCurrentValue =  float(value.get_averageValue())
       self._lastUpdate = datetime.datetime.now()

    # update the RD6006 Date and time with system time
    def setDateTime(self):
       now = datetime.datetime.now()
       self._serial.modbusWriteRegisters(self._slaveId, 48, [now.year,now.month,now.day,now.hour,now.minute,now.second ])

    # update the RD6006 Date and time with system time
    def isSerialPortConfigured(self):

        if self._serial.get_serialMode() !=  self._serialMode : return False
        if self._serial.get_protocol()   !=  self._serialProtocol : return False
        if self._serial.get_voltageLevel()!= self._serialVoltageLevel : return False
        if self._genSensor1.get_logFrequency() != self._sensorLogFrequency: return False
        if self._genSensor2.get_logFrequency() != self._sensorLogFrequency: return False
        if self._serial.get_currentJob() != self._jobFileName : return False
        # keep this test last.
        serialNumber = self._module.get_serialNumber()
        fileSystem =YFiles.FindFiles(serialNumber+".files")
        crc = zlib.crc32(self._jobFile.encode())& 0xffffffff
        flist = fileSystem.get_list("")
        for f in flist:
          if (f.get_name()==self._jobFileName) and (f.get_crc()==crc):  return True
        return False

    # Configure the serial port Parameter
    # one only need to run it once, as settings
    # will then be saved in the Yocto-Serial
    # device flash memory
    def configureSerialPort(self):
       self._serial.set_currentJob("")
       self._serial.set_protocol(self._serialProtocol)
       self._serial.set_serialMode( self._serialMode)
       self._serial.set_voltageLevel(self._serialVoltageLevel)
       self._genSensor1.set_logicalName("Voltage")
       self._genSensor1.set_unit("V")
       self._genSensor1.set_logFrequency("5/s")
       self._genSensor2.set_logicalName("Current")
       self._genSensor2.set_unit("A")
       self._genSensor2.set_logFrequency("5/s")
       self._serial.uploadJob(self._jobFileName, self._jobFile)
       self._serial.set_startupJob(self._jobFileName)
       self._serial.set_currentJob(self._jobFileName)
       self._module.saveToFlash()

    # sets the Voltage settings in volts
    def set_voltage(self,value):
       self._serial.modbusWriteRegister(self._slaveId, 8, int(value*100) )

    # returns the Voltage settings in volts
    def get_voltage(self):
       res =  self._serial.modbusReadRegisters(self._slaveId,  8, 1  )
       return res[0]/100

   # sets the Current settings in Amp
    def set_current(self,value):
       self._serial.modbusWriteRegister(self._slaveId, 9, int(value*1000) )

    # returns the Voltage settings in volts
    def get_current(self):
       res =  self._serial.modbusReadRegisters(self._slaveId,  9, 1  )
       return res[0]/1000

    # enable disable the output
    def enableOutput(self,OnOffValue):
        if OnOffValue :
            self._serial.modbusWriteRegister(self._slaveId, 18, 1 )
        else:
            self._serial.modbusWriteRegister(self._slaveId, 18, 0 )

    #  Return the voltage output  value, in volts
    # get_voltageOutput is implemented though timed callbacks
    # to avoid the temptation of flooding the Yocto-Serial with
    # get requests. Callbacks values are generated by the RD6006
    # job running on the Yocto-Serial (see configureSerialPort)
    def get_voltageOutput(self):
        delta =  datetime.datetime.now() -self._lastUpdate
        if (delta.total_seconds() >=1) : return float('nan')
        return self._lastVoltageValue

    # Returns the Current  output  value, in Amp
    # get_currentOutput is implemented though timed callbacks
    # to avoid the temptation of flooding the Yocto-Serial with
    # get requests. Callbacks values are generated by the RD6006
    # job running on the Yocto-Serial (see configureSerialPort)
    def get_currentOutput(self):
        delta = datetime.datetime.now() -self._lastUpdate
        if (delta.total_seconds() >=1): return float('nan')
        return self._lastCurrentValue





