import logging
import json
import re
import os
import signal
import subprocess
from datetime import *
from yocto_api import *
from yocto_servo import *
from yocto_motor import *
from yocto_anbutton import *
from yocto_colorled import *
from yocto_relay import *

Zones = [
    "RightFromMountain", "RightFromCableCar", "RightFromPassage", "RightToVillage", "RightAroundVillage",
    "LeftAroundVillage", "LeftFromVillage", "LeftToPassage", "LeftToCableCar", "LeftToMountain",
    "InTheTunnel"
]

IRSensors = {}

def setVolume(n):
    subprocess.Popen('osascript -e "set Volume '+str(n)+'"',
                     stdout=subprocess.PIPE, shell=True, preexec_fn=os.setsid)


def parse_timedelta(time_str):
    parts = re.match(r'((?P<hours>[\d.]+?)hr)?((?P<minutes>[\d.]+?)m)?((?P<seconds>[\d.]+?)s)?', time_str)
    if not parts:
        return
    parts = parts.groupdict()
    time_params = {}
    for (name, param) in parts.items():
        if param:
            time_params[name] = float(param)
    return timedelta(**time_params)

class StateMachine:
    def __init__(self, name, logging_level=logging.NOTSET):
        self._name = name
        self._states = {}
        self._timeouts = {}
        self._state = ''
        self._changed = datetime.datetime.now()
        self._next_timeout = timedelta(0)
        self._pending = []
        self._logger = logging.getLogger(name)
        if logging_level != logging.NOTSET:
            self._logger.setLevel(logging_level)

    #
    # Protected methods
    #
    def log_dbg(self, msg):
        self._logger.debug('%s %s %s', self._name, self._state, msg)

    def log_inf(self, msg):
        self._logger.info('%s %s %s', self._name, self._state, msg)

    def log_wrn(self, msg):
        self._logger.warning('%s %s %s', self._name, self._state, msg)

    def log_err(self, msg):
        self._logger.error('%s %s %s', self._name, self._state, msg)

    def fire_transition(self, event_info, transition):
        is_fun = re.match(r'\w+[(]([^)]*)[)]', transition)
        if is_fun:
            args = json.loads('['+is_fun.group(1)+']')
            next_state = getattr(self, transition[:is_fun.start(1)-1])(*args)
        else:
            next_state = transition
        if (next_state is not None) and (next_state != '') and (next_state in self._states):
            self.log_dbg('transition '+event_info+' => '+next_state)
            self._state = next_state
            self._changed = datetime.datetime.now()
            self._next_timeout = timedelta(0)
        elif (next_state is not None) and (next_state != ''):
            self.log_err('transition '+event_info+' => BAD STATE '+next_state)

    def add_state(self, state_name):
        self._states[state_name] = {}
        self._timeouts[state_name] = []
        if self._state == '':
            self._state = state_name
            self._changed = datetime.datetime.now()
            self._checked = False

    def add_trans(self, state_name, event_name, transition):
        self._states[state_name][event_name] = transition

    def add_timeout(self, state_name, timeout_str, transition):
        self._timeouts[state_name].append({ 'timeout': parse_timedelta(timeout_str), 'trans': transition })

    def add_global_trans(self, event_name, transition):
        for state_name in self._states:
            self._states[state_name][event_name] = transition

    #
    # Public methods
    #
    def get_state(self):
        return self._state

    def can_handle(self, event_name):
        return event_name in self._states[self._state]

    def trigger(self, event_name):
        return self._pending.append(event_name)

    def handle_events(self):
        for i in range(len(self._pending)):
            trans = self._states[self._state]
            event_name = self._pending[i]
            if event_name in trans:
                self.fire_transition(event_name, trans[event_name])
            else:
                self.log_wrn('Event '+event_name+' is not handled')
        self._pending = []
        delta_t = datetime.datetime.now() - self._changed
        timeout_idx = -1
        next_timeout = timedelta(365)
        if delta_t >= self._next_timeout:
            for i in range(len(self._timeouts[self._state])):
                timeout = self._timeouts[self._state][i]['timeout']
                if timeout >= self._next_timeout:
                    if (timeout_idx == -1) and (delta_t > timeout):
                        self.log_dbg("timeout hit: "+str(timeout.total_seconds()))
                        timeout_idx = i
                    else:
                        next_timeout = timeout
                        break
            if timeout_idx == -1:
                self._next_timeout = next_timeout
            else:
                timeout2 = self._timeouts[self._state][timeout_idx]['timeout']
                curr_state = self._state
                self._next_timeout = timeout2
                self.fire_transition('+'+str(timeout.total_seconds())+'s', self._timeouts[self._state][timeout_idx]['trans'])
                if (self._state == curr_state) and (self._next_timeout == timeout2):  # no state transition occured
                    self._next_timeout = next_timeout
                    #self.log_dbg("next timeout: "+str(next_timeout.total_seconds()))

#
# WindMill: handled by the Yocto-Motor-DC of the same name
#
class WindMill(StateMachine):
    def __init__(self, name, min_speed, max_speed, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._motor = YMotor.FindMotor(name)
        self._minSpeed = min_speed
        self._maxSpeed = max_speed
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("Halted")
        self.add_trans("Halted", "start", "startWind()")
        self.add_trans("Halted", "stop", "")
        self.add_state("Started")
        self.add_trans("Started", "stop", "Stopping")
        self.add_timeout("Started", "100s", "stopWind()")
        self.add_state("Stopping")
        self.add_trans("Stopping", "stop", "")
        self.add_timeout("Stopping", "3s", "stopWind()")

    def detect_state(self):
        if not self._motor.isOnline():
            return ''
        if self._motor.get_drivingForce() == 0:
            self._motor.resetStatus()
            return "Halted"
        self._motor.set_failSafeTimeout(120000)
        return "Started"

    def startWind(self):
        if not self._motor.isOnline():
            return "Startup"
        self._motor.resetStatus()
        self._motor.set_failSafeTimeout(120000)
        self._motor.set_drivingForce(random.randint(self._minSpeed, self._maxSpeed))
        return "Started"

    def stopWind(self):
        if not self._motor.isOnline():
            return "Startup"
        self._motor.drivingForceMove(0,3000)
        self._motor.set_failSafeTimeout(0)
        self._motor.resetStatus()
        return "Halted"

#
# LightHouse: handled by a Yocto-Motor-DC of the same name, and a Yocto-PowerLed
#
class LightHouse(StateMachine):
    def __init__(self, name, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._motor = YMotor.FindMotor(name)
        self._light = YColorLed.FindColorLed("Light"+name)
        self._speed = 55
        self._color = 0x303000
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("Halted")
        self.add_trans("Halted", "start", "startLight()")
        self.add_trans("Halted", "stop", "")
        self.add_state("Started")
        self.add_trans("Started", "stop", "stopLight()")
        self.add_timeout("Startup", "100s", "stopLight()")

    def detect_state(self):
        if not self._motor.isOnline():
            return ''
        if self._motor.get_drivingForce() == 0:
            self._motor.resetStatus()
            return "Halted"
        self._motor.set_failSafeTimeout(120000)
        return "Started"

    def startLight(self):
        if not self._motor.isOnline():
            return "Startup"
        self._motor.resetStatus()
        self._motor.set_failSafeTimeout(120000)
        self._motor.set_drivingForce(self._speed)
        self._light.rgbMove(self._color,1000)
        return "Started"

    def stopLight(self):
        if not self._motor.isOnline():
            return "Startup"
        self._motor.drivingForceMove(0,3000)
        self._motor.set_failSafeTimeout(0)
        self._light.rgbMove(0,3000)
        return "Halted"

#
# Baloon: handled by a Servo with the same name as the Ballon
#
class Balloon(StateMachine):
    def __init__(self, name, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._servo = YServo.FindServo(name)
        self._range = 30
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("Up")     # Hidden (0)
        self.add_trans("Up", "start", "move_high(-1)")
        self.add_trans("Up", "waitAndStart", "WaitAndStart")
        self.add_trans("Up", "stop", "")
        self.add_timeout("Up", "6s", "power_off()")
        self.add_state("WaitAndStart")
        self.add_trans("WaitAndStart", "start", "move_high(-1)")
        self.add_timeout("WaitAndStart", "4s", "move_high(-1)")
        self.add_state("High")   # Around 300
        self.add_trans("High", "stop", "go_up()")
        self.add_timeout("High", "5s", "move_high(-1)")
        self.add_timeout("High", "10s", "move_middle(600)")
        self.add_state("Middle") # Around 600
        self.add_trans("Middle", "stop", "go_up()")
        self.add_timeout("Middle", "5s", "move_middle(-1)")
        self.add_timeout("Middle", "10s", "move_high(300)")
        self.add_state("Low")    # Around 900
        self.add_trans("Low", "stop", "go_up()")
        self.add_timeout("Low", "5s", "move_low(-1)")
        self.add_timeout("Low", "10s", "move_middle(600)")

    def move_to(self, target_pos):
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(-target_pos, 5000)

    def detect_state(self):
        if not self._servo.isOnline():
            return ''
        pos = self._servo.get_position()
        self._servo.set_range(self._range)
        if pos == 0:
            return "Up"
        self.trigger("stop")
        if pos < 450:
            return "High"
        if pos < 750:
            return "Middle"
        return "Low"

    def go_up(self):
        if not self._servo.isOnline():
            return "Startup"
        self.move_to(0)
        return "Up"

    def move_high(self, pos):
        if not self._servo.isOnline():
            return "Startup"
        if(pos == -1):
            pos = random.randint(150,750)
        self.move_to(pos)
        if pos < 250:
            self.move_to(0)
            return "Up"
        if pos > 450:
            return "Middle"
        if self._state != "High":
            return "High"

    def move_middle(self, pos):
        if not self._servo.isOnline():
            return "Startup"
        if(pos == -1):
            pos = random.randint(0,1000)
        self.move_to(pos)
        if pos < 250:
            self.move_to(0)
            return "Up"
        if pos < 500:
            return "High"
        if pos > 700:
            return "Low"
        if self._state != "Middle":
            return "Middle"

    def move_low(self, pos):
        if not self._servo.isOnline():
            return "Startup"
        if(pos == -1):
            pos = random.randint(500,1000)
        self.move_to(pos)
        if pos < 800:
            return "Middle"
        if self._state != "Low":
            return "Low"

    def power_off(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_FALSE)

    def start(self):
        self.trigger("start")

#
# Cable car: handled by a Servo with the same name as the Cable Car
#
class CableCar(StateMachine):
    def __init__(self, name, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._servo = YServo.FindServo(name)
        self._range = 133
        self._duration = 16 # seconds
        self._top_pos = -900
        self._slow_pos = 750
        self._ground_pos = 1000
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("GroundStation")
        self.add_trans("GroundStation", "start", "slow_up()")
        self.add_timeout("GroundStation", "1s", "power_off()")
        self.add_state("Uphill")
        self.add_timeout("Uphill", "5s", "move_up()")
        self.add_timeout("Uphill", str(self._duration+5)+"s", "TopStation")
        self.add_state("TopStation")
        self.add_trans("TopStation", "start", "start_down()")
        self.add_timeout("TopStation", "1s", "power_off()")
        self.add_state("Downhill")
        self.add_timeout("Downhill", str(self._duration)+"s", "slow_down()")
        self.add_timeout("Downhill", str(self._duration+5)+"s", "GroundStation")
        self.add_global_trans("park", "start_down()")

    def move_to(self, target_pos):
        full_dist = abs(self._top_pos - self._slow_pos)
        distance = abs(target_pos - self._servo.get_position())
        duration = int(1000 * self._duration * distance / full_dist)
        #self.log_dbg("move servo to "+str(target_pos)+" in "+str(duration/1000)+"s")
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(target_pos, duration)

    def detect_state(self):
        if not self._servo.isOnline():
            return ''
        self._servo.set_range(self._range)
        if self._servo.get_position() == self._top_pos:
            return "TopStation"
        if self._servo.get_position() == self._ground_pos:
            return "GroundStation"
        self.move_to(self._top_pos)
        return "Uphill"

    def start_down(self):
        if not self._servo.isOnline():
            return "Startup"
        self.move_to(self._slow_pos)
        return "Downhill"

    def slow_down(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(self._ground_pos, 5000)

    def slow_up(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(self._slow_pos, 5000)
        return "Uphill"

    def move_up(self):
        if not self._servo.isOnline():
            return "Startup"
        self.move_to(self._top_pos)

    def power_off(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_FALSE)

    def start(self):
        self.trigger("start")

#
# Boat: handled by a Servo with the same name as the Boat
#
class Boat(StateMachine):
    def __init__(self, name, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._servo = YServo.FindServo(name)
        self._range = 190
        self._duration = 16 # seconds
        self._top_pos = -250
        self._slow_pos = 700
        self._ground_pos = 1000
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("Harbour")
        self.add_trans("Harbour", "start", "slow_up()")
        self.add_timeout("Harbour", "1s", "power_off()")
        self.add_state("Leaving")
        self.add_timeout("Leaving", "5s", "move_up()")
        self.add_timeout("Leaving", str(self._duration+5)+"s", "DeepSea")
        self.add_state("DeepSea")
        self.add_trans("DeepSea", "start", "start_down()")
        self.add_timeout("DeepSea", "1s", "power_off()")
        self.add_state("Arriving")
        self.add_timeout("Arriving", str(self._duration)+"s", "slow_down()")
        self.add_timeout("Arriving", str(self._duration+5)+"s", "Harbour")
        self.add_global_trans("park", "slow_up()")

    def move_to(self, target_pos):
        full_dist = abs(self._top_pos - self._slow_pos)
        distance = abs(target_pos - self._servo.get_position())
        duration = int(1000 * self._duration * distance / full_dist)
        #self.log_dbg("move servo to "+str(target_pos)+" in "+str(duration/1000)+"s")
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(target_pos, duration)

    def detect_state(self):
        if not self._servo.isOnline():
            return ''
        self._servo.set_range(self._range)
        if self._servo.get_position() == self._top_pos:
            return "DeepSea"
        if self._servo.get_position() == self._ground_pos:
            return "Harbour"
        self.move_to(self._top_pos)
        return "Leaving"

    def start_down(self):
        if not self._servo.isOnline():
            return "Startup"
        self.move_to(self._slow_pos)
        return "Arriving"

    def slow_down(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(self._ground_pos, 5000)

    def slow_up(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_TRUE)
        self._servo.move(self._slow_pos, 5000)
        return "Leaving"

    def move_up(self):
        if not self._servo.isOnline():
            return "Startup"
        self.move_to(self._top_pos)

    def power_off(self):
        if not self._servo.isOnline():
            return "Startup"
        self._servo.set_enabled(YServo.ENABLED_FALSE)

    def start(self):
        self.trigger("start")

#
# Train: handled by a Yocto-Motor-DC of the same name
#
class Train(StateMachine):
    def __init__(self, name, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._motor = YMotor.FindMotor(name)
        self._crossing = YRelay.FindRelay("RailroadCrossing")
        self._minimumSpeed = 25
        self._slowSpeed = 34
        self._cruiseSpeed = 38
        self._stopRequested = False
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("HaltedAtStation")
        self.add_trans("HaltedAtStation", "start", "start()")
        self.add_state("HaltedAtCableCar")
        self.add_trans("HaltedAtCableCar", "start", "start()")
        self.add_timeout("HaltedAtCableCar", "10s", "start()")
        self.add_state("EmergencyStop")
        self.add_trans("EmergencyStop", "start", "start()")
        for i in range(len(Zones)):
            zone = Zones[i]
            if i+1 >= len(Zones):
                next_zone = Zones[0]
                next2_zone = Zones[1]
            elif i+2 >= len(Zones):
                next_zone = Zones[i+1]
                next2_zone = Zones[0]
            else:
                next_zone = Zones[i+1]
                next2_zone = Zones[i+2]
            self.add_state(zone)
            self.add_trans(zone, next_zone, next_zone)
            self.add_trans(zone, next2_zone, next2_zone)
        self.add_global_trans("stop!", "emergency()")
        self.add_global_trans("stop", "goToStation()")
        self.add_timeout("RightFromMountain", "0.1s", "slow()")
        self.add_timeout("RightFromMountain", "0.2s", 'setLights("cableCar")')
        self.add_timeout("RightFromCableCar", "0.1s", 'setLights("village")')
        self.add_timeout("RightToVillage", "0.1s", "cruise()")
        self.add_timeout("LeftFromVillage", "0.1s", 'stopIfRequested()')
        self.add_timeout("LeftFromVillage", "0.2s", 'setLights("cableCar")')
        self.add_timeout("InTheTunnel", "0.1s", 'openCrossing()')
        self.add_timeout("InTheTunnel", "1s", 'setLights("windTurbines")')
#        for i in range(len(Zones)):
#            zone = Zones[i]
#            if zone == "InTheTunnel":
#                self.add_timeout(zone, "40s", "emergency()")
#            else:
#                self.add_timeout(zone, "15s", "emergency()")

    def clearTimings(self):
        for zone in IRSensors:
            irSensor = IRSensors[zone]
            irSensor.setUserData({"event":zone, "lastIn":False, "lastOut":False})

    def emergency(self):
        if not self._motor.isOnline():
            return
        self._motor.set_failSafeTimeout(0)
        self._motor.set_drivingForce(0)
        self.clearTimings()
        self.openCrossing()
        return "EmergencyStop"

    def goToStation(self):
        self._stopRequested = True

    def stopIfRequested(self):
        if(self._stopRequested):
            self._motor.set_failSafeTimeout(0)
            self._motor.drivingForceMove(0,3000)
            self.clearTimings()
            self.openCrossing()
            lights.trigger("stop")
            return "HaltedAtStation"

    def closeCrossing(self):
        self._crossing.set_state(YRelay.STATE_B)

    def openCrossing(self):
        self._crossing.set_state(YRelay.STATE_A)

    def keepALive(self):
        if not self._motor.isOnline():
            return
        self._motor.keepALive()

    def detect_state(self):
        if not self._motor.isOnline():
            return ''
        pos = "InTheTunnel"
        for zone in IRSensors:
            if (zone != pos) and (IRSensors[zone].get_calibratedValue() > 500):
                pos = zone
                break
        if self._motor.get_drivingForce() == 0:
            self._motor.resetStatus()
            if pos == "LeftFromVillage":
                pos = "HaltedAtStation"
            elif pos == "LeftToCableCar":
                pos = "HaltedAtCableCar"
            else:
                pos = "EmergencyStop"
        return pos

    def start(self):
        if not self._motor.isOnline():
            return "Startup"
        self._stopRequested = False
        self._motor.resetStatus()
        self._motor.set_drivingForce(self._minimumSpeed)
        self._motor.drivingForceMove(self._cruiseSpeed, 5000)
        self._motor.set_failSafeTimeout(120000)
        return self.detect_state()

    def slow(self):
        if not self._motor.isOnline():
            return "Startup"
        self._motor.drivingForceMove(self._slowSpeed, 1000)

    def cruise(self):
        if not self._motor.isOnline():
            return "Startup"
        self._motor.drivingForceMove(self._cruiseSpeed, 1000)

    def setLights(self, event_name):
        lights.trigger(event_name)

prevState = ''
prev2State = ''
def irSensorChanged(anButton, value):
    global prevState, prev2State
    trainState = train.get_state()
    userData = anButton.get_userData()
    upEvent = userData["event"]
    #train.log_dbg("EVENT: "+upEvent+"="+value)
    if int(value) == 1000:
        if upEvent == trainState or upEvent == prevState:
            return
        userData["lastIn"] = datetime.datetime.today()
        if (upEvent == "RightFromMountain") or (upEvent == "LeftToPassage"):
            train.closeCrossing()
        elif (upEvent == "RightFromPassage") or (upEvent== "LeftToMountain"):
            train.openCrossing()
    elif int(value) == 0:
        userData["lastOut"] = datetime.datetime.today()
        if (upEvent[-10:] == "ToMountain") and (train.get_state() == upEvent):
            upEvent = "InTheTunnel"
        else:
            return
    else:
        return
    if not train.can_handle(upEvent):
        return
    if re.match(r'(Left|Right).*', prev2State):
        prevSensor = YAnButton.FindAnButton(prev2State)
        userData = prevSensor.get_userData()
        if userData["lastOut"] and userData["lastIn"]:
            delta = userData["lastOut"] - userData["lastIn"]
            seconds = delta.total_seconds()
            if (seconds > 0.1) and (seconds < 1):
                train.log_wrn("short state duration for "+prev2State+": only "+str(delta))
                train.log_wrn("=> possible loss of wagons, emergency stop !")
    prev2State = prevState
    prevState = trainState
    train.keepALive()
    train.trigger(upEvent)

def enable_IRSensors():
    for zone in IRSensors:
        irSensor = IRSensors[zone]
        irSensor.setUserData({"event":zone, "lastIn":False, "lastOut":False})
        irSensor.registerValueCallback(irSensorChanged)

#
# Lights: handled by five Yocto-PowerLeds
#
class Lights(StateMachine):
    def __init__(self, name, logging_level=logging.NOTSET):
        super().__init__(name, logging_level)
        self._player = False
        self._volume = 1
        self._village = YColorLed.FindColorLed('LightVillage')
        self._cableCar = YColorLed.FindColorLed('LightCableCar')
        self._windTurbines = YColorLed.FindColorLed('LightWindTurbines')
        self._harbour = YColorLed.FindColorLed('LightHarbour')
        self._pressMe1 = YColorLed.FindColorLed('PressMe.colorLed1')
        self._pressMe2 = YColorLed.FindColorLed('PressMe.colorLed2')
        self._villageColor = 40 * 0x10000 + 0xff7f
        self._cableCarColor = 65 * 0x10000 + 0xff7f
        self._windTurbinesColor = 125 * 0x10000 + 0xff7f
        self._harbourColor = 177 * 0x10000 + 0xff7f
        self._moreRuns = 0
        self.add_state("Startup")
        self.add_timeout("Startup", "0.1s", "detect_state()")
        self.add_state("Shutdown")
        self.add_timeout("Shutdown", "0.1s", "fadeAll()")
        self.add_timeout("Shutdown", "0.2s", "soundFade(6)")
        self.add_timeout("Shutdown", "1.2s", "soundFade(4)")
        self.add_timeout("Shutdown", "2.2s", "soundFade(2)")
        self.add_timeout("Shutdown", "3.2s", "soundFade(1)")
        self.add_timeout("Shutdown", "4.2s", "soundFade(0)")
        self.add_timeout("Shutdown", "5.9s", "lightsBackOn()")
        self.add_timeout("Shutdown", "6s", "FullShutdown")
        self.add_state("FullShutdown")
        self.add_trans("FullShutdown", "start", "start()")
        self.add_timeout("FullShutdown", "0.1s", "fadeAll()")
        self.add_timeout("FullShutdown", "2.6s", "pulseMe()")
        self.add_timeout("FullShutdown", "3.1s", "fadeAll()")
        self.add_timeout("FullShutdown", "5.6s", "pulseMe()")
        self.add_timeout("FullShutdown", "6s", "FullShutdown")
        self.add_state("AboutToStart")
        self.add_timeout("AboutToStart", "5s", "startNow()")
        self.add_state("ShowVillage")
        self.add_timeout("ShowVillage", "0.1s", "showVillage()")
        self.add_timeout("ShowVillage", "5s", "shutdownIfNeeded()")
        self.add_timeout("ShowVillage", "25s", "ShowCableCar")
        self.add_trans("ShowVillage", "next", "ShowCableCar")
        self.add_state("ShowCableCar")
        self.add_timeout("ShowCableCar", "0.1s", "showCableCar()")
        self.add_timeout("ShowCableCar", "25s", "ShowMountain")
        self.add_trans("ShowCableCar", "next", "ShowMountain")
        self.add_state("ShowMountain")
        self.add_timeout("ShowMountain", "0.1s", "showMountain()")
        self.add_timeout("ShowMountain", "15s", "ShowWindTurbines")
        self.add_trans("ShowMountain", "next", "ShowWindTurbines")
        self.add_state("ShowWindTurbines")
        self.add_timeout("ShowWindTurbines", "0.1s", "showWindTurbines()")
        self.add_timeout("ShowWindTurbines", "3s", "ShowHarbour")
        self.add_trans("ShowWindTurbines", "next", "ShowHarbour")
        self.add_state("ShowHarbour")
        self.add_timeout("ShowHarbour", "0.1s", "showHarbour()")
        self.add_timeout("ShowHarbour", "30s", "ShowVillage")
        self.add_trans("ShowHarbour", "next", "ShowVillage")
        self.add_global_trans("stop", "Shutdown")
        self.add_global_trans("village", "ShowVillage")
        self.add_global_trans("cableCar", "ShowCableCar")
        self.add_global_trans("mountain", "ShowMountain")
        self.add_global_trans("windTurbines", "ShowWindTurbines")
        self.add_global_trans("harbour", "ShowHarbour")

    def detect_state(self):
        if self._village.isOnline():
            if (self._village.get_hslColor() & 0xff) > 64:
                return "ShowVillage"
        if self._cableCar.isOnline():
            if (self._cableCar.get_hslColor() & 0xff) > 64:
                return "ShowCableCar"
        if self._windTurbines.isOnline():
            if (self._windTurbines.get_hslColor() & 0xff) > 64:
                return "ShowWindTurbines"
        if self._harbour.isOnline():
            if (self._harbour.get_hslColor() & 0xff) > 64:
                return "ShowHarbour"
        return "FullShutdown"

    def start(self):
        if self._pressMe1.isOnline():
            self._pressMe1.rgbMove(0,3000)
            self._pressMe2.rgbMove(0,3000)
        if(self._player):
            self._player.terminate()
        path = "/yoctofriday/fenetre/musique"
        files = os.listdir(path)
        idx = random.randint(0,len(files)-1)
        while files[idx][0] == '.':
            idx = random.randint(0,len(files)-1)
        self.log_inf("Volume: "+str(5))
        self._volume = 10
        setVolume(10)
        self._player = subprocess.Popen('afplay "'+path+'/'+files[idx]+'"',
                                        stdout=subprocess.PIPE, shell=True, preexec_fn=os.setsid)
        if lightSwitch.isOnline():
            lightSwitch.set_state(YRelay.STATE_A)
        self._moreRuns = 2
        return "AboutToStart"

    def startNow(self):
        train.trigger("start")
        return "ShowVillage"

    def fadeVillage(self):
        self._village.hslMove(self._villageColor & 0xffff00, 3000)
        windMill.trigger("stop")

    def fadeCableCar(self):
        self._cableCar.hslMove(self._cableCarColor & 0xffff00, 3000)

    def fadeWindTurbines(self):
        self._windTurbines.hslMove(self._windTurbinesColor & 0xffff00, 3000)
        windTurbines.trigger("stop")

    def fadeHarbour(self):
        self._harbour.hslMove(self._harbourColor & 0xffff00, 3000)
        lightHouse.trigger("stop")

    def fadeAll(self):
        self.fadeVillage()
        self.fadeCableCar()
        self.fadeWindTurbines()
        self.fadeHarbour()
        balloon.trigger("stop")
        if train.get_state() != "HaltedAtStation":
            train.trigger("stop")
        if self._pressMe1.isOnline():
            self._pressMe1.rgbMove(0x00ff80,500)
            self._pressMe2.rgbMove(0x00ff80,500)

    def soundFade(self, vol):
        if(vol >= self._volume):
            return
        self.log_inf("Volume: "+str(vol))
        self._volume = vol
        setVolume(vol)

    def lightsBackOn(self):
        if lightSwitch.isOnline():
            lightSwitch.set_state(YRelay.STATE_B)
        boat.trigger("park")
        cableCar.trigger("park")
        if(self._player):
            self._player.terminate()
            self._player = False
        self._volume = -1
        setVolume(1)

    def shutdownIfNeeded(self):
        if self._moreRuns == 0:
            return "Shutdown"
        else:
            self._moreRuns = self._moreRuns - 1

    def pulseMe(self):
        if self._pressMe1.isOnline():
            self._pressMe1.rgbMove(0,400)
            self._pressMe2.rgbMove(0,400)

    def showVillage(self):
        self.fadeWindTurbines()
        self.fadeHarbour()
        if self._moreRuns == 0:
            self.log_inf("Enough played, go to station")
            if train.get_state() != "HaltedAtStation":
                train.trigger("stop")
        else:
            self.log_inf(str(self._moreRuns)+" more run(s) to go")
        windMill.trigger("start")
        balloon.trigger("waitAndStart")
        self._village.hslMove(self._villageColor + 0x10000*random.randint(-10,10), 3000)

    def showCableCar(self):
        self.fadeHarbour()
        cableCar.trigger("start")
        self._cableCar.hslMove(self._cableCarColor + 0x10000*random.randint(-10,10), 3000)

    def showMountain(self):
        self.fadeHarbour()

    def showWindTurbines(self):
        self.fadeVillage()
        windTurbines.trigger("start")
        self._windTurbines.hslMove(self._windTurbinesColor + 0x10000*random.randint(-10,10), 3000)

    def showHarbour(self):
        self.fadeVillage()
        self.fadeCableCar()
        lightHouse.trigger("start")
        boat.trigger("start")
        self._harbour.hslMove(self._harbourColor + 0x10000*random.randint(-10,10), 3000)

def pressMeChanged(anButton, value):
    lights.log_inf("pressMe: ["+value+"]")
    if (lights.get_state() == "FullShutdown") and (value == "0"):
        lights.trigger("start")

def stopChanged(anButton, value):
    lights.log_inf("stop: ["+value+"]")
    if value == "0":
        lights.trigger("stop")

def yapiLog(msg):
    logging.getLogger("YAPI").info("%s", msg)

#
# Complete scenery
#
logging.basicConfig(level=logging.DEBUG)
YAPI.RegisterLogFunction(yapiLog)

YAPI.RegisterHub("192.168.1.150") # YoctoHub LightSwitch
YAPI.RegisterHub("192.168.1.151") # YoctoHub Village
YAPI.RegisterHub("192.168.1.152") # YoctoHub See
YAPI.UpdateDeviceList()
#YAPI.DisableExceptions()
for i in range(len(Zones)):
    IRSensors[Zones[i]] = YAnButton.FindAnButton(Zones[i])

lightHouse = LightHouse("LightHouse")
windMill = WindMill("WindMill", 20, 30)
balloon = Balloon("HotAirBalloon")
cableCar = CableCar("CableCar")
windTurbines = WindMill("WindTurbines", 72, 80)
boat = Boat("Ferry")
train = Train("Train")
lights = Lights("Lights")

enable_IRSensors()

startButton = YAnButton.FindAnButton("PressMe")
startButton.registerValueCallback(pressMeChanged)
stopButton = YAnButton.FindAnButton("Stop")
stopButton.registerValueCallback(stopChanged)
lightSwitch = YRelay.FindRelay("LightSwitch")

random.seed()
while True:
    train.handle_events()
    lights.handle_events()
    cableCar.handle_events()
    balloon.handle_events()
    windMill.handle_events()
    windTurbines.handle_events()
    boat.handle_events()
    lightHouse.handle_events()
    YAPI.Sleep(50)
