#!/usr/bin/python
# -*- coding: UTF-8 -*-
# Control script for the automatic heater from
# www.yoctopuce.com/EN/article/building-an-oven-and-itsregulation

import os, sys, time

from yocto_api import *
from yocto_temperature import *
from yocto_relay import *
from yocto_anbutton import *
from yocto_display import *

addr = "127.0.0.1" # local virtualHub
target = 50        # default temperature target is 50°C
interval = 1000    # one PID cycle  every second

# some state global variables
PID_Data = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
PID_Ptr = 0
btstate = [False, False, False, False, False]
shutdown = False
running = False
mustRefresh = True  

# This function is automatically called when the user 
# presses a button. It will update some control global
# variables depending on which button is pressed.
def buttonChange(src, value):
    global btstate
    global target
    global running
    global mustRefresh
    global PID_Ptr
    global shutdown

    index = src.get_userData()
    # print(str(value)+" on "+ str(index))
    newstate = int(value) < 500
    if newstate and newstate != btstate[index]:
        if index == 0:
            target = target + 1
        if index == 1:
            target = target - 1
        if index == 2:
            running = True
            print("Run")
        if index == 3:
            print("Pause")
            running = False
            PID_Ptr = 0
        if index == 4:
            print("ShutDown")
            shutdown = True

        mustRefresh = True
    btstate[index] = newstate

# this function refresh the display. it will show
# the current temperature, the target temperature
# and the appliance state.
def refresh(temp):
    global t
    global L1, L2
    global d
    global shutdown
    maintxt = ""
    L1.clear()
    if shutdown:
        maintxt = 'SHUTDOWN'
    else:
        maintxt = ("%.1f" % temp) + "/" + ("%.0f" % target) + "°C"
        L1.selectFont('Small.yfm')
        if running:
            txt = "RUNNING"
        else:
            txt = "PAUSED"
        L1.drawText(64, 64, YDisplayLayer.ALIGN.BOTTOM_CENTER, txt)

    L1.selectFont('Medium.yfm')
    L1.drawText(64, 32, YDisplayLayer.ALIGN.CENTER, maintxt)
    d.swapLayerContent(1, 2)

# the PID code
def PID(target, current):
    global PID_Data
    global PID_Ptr
    # PID constants, empirically chosen
    PID_A = 300.0
    PID_B = 100.0
    PID_C = -50.0

    err = target - current
    if PID_Ptr < len(PID_Data):
        PID_Data[PID_Ptr] = err
        PID_Ptr = PID_Ptr + 1
    else:
        del PID_Data[0]
        PID_Data.append(err)

    if PID_Ptr > 2:
        P = PID_Data[PID_Ptr - 1]
        I = 0
        for n in range(0, PID_Ptr):
            I = I + PID_Data[n]
        I = I / PID_Ptr
        D = (PID_Data[PID_Ptr - 1] - PID_Data[0]) / PID_Ptr
        return PID_A * P + PID_B * I + PID_C * D
    return 0

# API init
print(YAPI.GetAPIVersion())
errmsg = YRefParam()
# Waitind for the virtual hub to be up
while YAPI.TestHub(addr, 10000, errmsg) != YAPI.SUCCESS:
    res = YAPI.Sleep(1000, errmsg)
print("VirtualHub found")

if YAPI.RegisterHub(addr, errmsg) != YAPI.SUCCESS:
    sys.exit("init error" + errmsg.value)
    
# looking for the yoctopuce devices
t = YTemperature.FindTemperature('Box-Temp')
r = YRelay.FindRelay('Box-Lamp')
d = YDisplay.FindDisplay('Temp-Display')

if not t.isOnline():
    sys.exit("init error: no Box-Temp temperature sensor")
if not r.isOnline():
    sys.exit("init error: no Box-Lamp relay")
if not d.isOnline():
    sys.exit("init error: no Temp-Display")

# bind the callback function to all buttons    
for i in range(0, 5):
    bt = YAnButton.FindAnButton(d.get_module().get_serialNumber() + '.anButton' + str(i + 1))
    if not bt.isOnline():
        sys.exit("init error: anButton" + str(i) + ' found');
    bt.set_userData(i)
    bt.registerValueCallback(buttonChange)

# display init
d.resetAll()
L1 = d.get_displayLayer(1)
L2 = d.get_displayLayer(2)
L1.hide()

n = 0
while not shutdown:

    if n == 0:
        tick = YAPI.GetTickCount()
        temp = t.get_currentValue()
        mustRefresh = True
        # the relay control is based on the pulse
        # function which is much secure than trying
        # to control both switches ON and OFF
        if running and r.isOnline():
            delay = int(round(PID(target, temp)))
            if delay < 0:
                delay = 0
            if delay > interval:
                delay = int(interval * 2)
            r.pulse(delay) 

    if mustRefresh:
        refresh(temp)
        mustRefresh = False

    # we want the display to be reactive, so the  main
    # loop run 10 times faster than the PID, and we try 
    # to make sure there is one PID cycle every *interval* 
    # ms hence "the time to wait" computing  
    n = (n + 1) % 10
    dt = YAPI.GetTickCount() - tick
    delta = int(n * interval / 10 - + (dt.seconds * 1000 + dt.microseconds / 1000))
    if delta > 0:
        YAPI.Sleep(delta, errmsg)      
    else:
        YAPI.HandleEvents(errmsg)
      
# loop exit, we stop the heating, refresh the
# display and force the raspberry PI to shutdown      
r.set_state(YRelay.STATE_A)
refresh(temp)
print("shutting down now")
os.system("sudo halt -p")
