' ********************************************************************
'
'  $Id: svn_id $
'
'  Implements yFindVoltage(), the high-level API for Voltage functions
'
'  - - - - - - - - - License information: - - - - - - - - -
'
'  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
'
'  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
'  non-exclusive license to use, modify, copy and integrate this
'  file into your software for the sole purpose of interfacing
'  with Yoctopuce products.
'
'  You may reproduce and distribute copies of this file in
'  source or object form, as long as the sole purpose of this
'  code is to interface with Yoctopuce products. You must retain
'  this notice in the distributed source file.
'
'  You should refer to Yoctopuce General Terms and Conditions
'  for additional information regarding your rights and
'  obligations.
'
'  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
'  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
'  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
'  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
'  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
'  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
'  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
'  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
'  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
'  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
'  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
'  WARRANTY, OR OTHERWISE.
'
' *********************************************************************


Imports YDEV_DESCR = System.Int32
Imports YFUN_DESCR = System.Int32
Imports System.Runtime.InteropServices
Imports System.Text

Module yocto_voltage

    REM --- (YVoltage return codes)
    REM --- (end of YVoltage return codes)
    REM --- (YVoltage dlldef)
    REM --- (end of YVoltage dlldef)
   REM --- (YVoltage yapiwrapper)
   REM --- (end of YVoltage yapiwrapper)
  REM --- (YVoltage globals)

  Public Const Y_ENABLED_FALSE As Integer = 0
  Public Const Y_ENABLED_TRUE As Integer = 1
  Public Const Y_ENABLED_INVALID As Integer = -1
  Public Const Y_SIGNALBIAS_INVALID As Double = YAPI.INVALID_DOUBLE
  Public Delegate Sub YVoltageValueCallback(ByVal func As YVoltage, ByVal value As String)
  Public Delegate Sub YVoltageTimedReportCallback(ByVal func As YVoltage, ByVal measure As YMeasure)
  REM --- (end of YVoltage globals)

  REM --- (YVoltage class start)

  '''*
  ''' <summary>
  '''   The <c>YVoltage</c> class allows you to read and configure Yoctopuce voltage sensors.
  ''' <para>
  '''   It inherits from <c>YSensor</c> class the core functions to read measurements,
  '''   to register callback functions, and to access the autonomous datalogger.
  ''' </para>
  ''' </summary>
  '''/
  Public Class YVoltage
    Inherits YSensor
    REM --- (end of YVoltage class start)

    REM --- (YVoltage definitions)
    Public Const ENABLED_FALSE As Integer = 0
    Public Const ENABLED_TRUE As Integer = 1
    Public Const ENABLED_INVALID As Integer = -1
    Public Const SIGNALBIAS_INVALID As Double = YAPI.INVALID_DOUBLE
    REM --- (end of YVoltage definitions)

    REM --- (YVoltage attributes declaration)
    Protected _enabled As Integer
    Protected _signalBias As Double
    Protected _valueCallbackVoltage As YVoltageValueCallback
    Protected _timedReportCallbackVoltage As YVoltageTimedReportCallback
    REM --- (end of YVoltage attributes declaration)

    Public Sub New(ByVal func As String)
      MyBase.New(func)
      _classname = "Voltage"
      REM --- (YVoltage attributes initialization)
      _enabled = ENABLED_INVALID
      _signalBias = SIGNALBIAS_INVALID
      _valueCallbackVoltage = Nothing
      _timedReportCallbackVoltage = Nothing
      REM --- (end of YVoltage attributes initialization)
    End Sub

    REM --- (YVoltage private methods declaration)

    Protected Overrides Function _parseAttr(ByRef json_val As YJSONObject) As Integer
      If json_val.has("enabled") Then
        If (json_val.getInt("enabled") > 0) Then _enabled = 1 Else _enabled = 0
      End If
      If json_val.has("signalBias") Then
        _signalBias = Math.Round(json_val.getDouble("signalBias") / 65.536) / 1000.0
      End If
      Return MyBase._parseAttr(json_val)
    End Function

    REM --- (end of YVoltage private methods declaration)

    REM --- (YVoltage public methods declaration)
    '''*
    ''' <summary>
    '''   Returns the activation state of this input.
    ''' <para>
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <returns>
    '''   either <c>YVoltage.ENABLED_FALSE</c> or <c>YVoltage.ENABLED_TRUE</c>, according to the activation
    '''   state of this input
    ''' </returns>
    ''' <para>
    '''   On failure, throws an exception or returns <c>YVoltage.ENABLED_INVALID</c>.
    ''' </para>
    '''/
    Public Function get_enabled() As Integer
      Dim res As Integer
      If (Me._cacheExpiration <= YAPI.GetTickCount()) Then
        If (Me.load(YAPI._yapiContext.GetCacheValidity()) <> YAPI.SUCCESS) Then
          Return ENABLED_INVALID
        End If
      End If
      res = Me._enabled
      Return res
    End Function


    '''*
    ''' <summary>
    '''   Changes the activation state of this voltage input.
    ''' <para>
    '''   When AC measurements are disabled,
    '''   the device will always assume a DC signal, and vice-versa. When both AC and DC measurements
    '''   are active, the device switches between AC and DC mode based on the relative amplitude
    '''   of variations compared to the average value.
    '''   Remember to call the <c>saveToFlash()</c>
    '''   method of the module if the modification must be kept.
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <param name="newval">
    '''   either <c>YVoltage.ENABLED_FALSE</c> or <c>YVoltage.ENABLED_TRUE</c>, according to the activation
    '''   state of this voltage input
    ''' </param>
    ''' <para>
    ''' </para>
    ''' <returns>
    '''   <c>YAPI.SUCCESS</c> if the call succeeds.
    ''' </returns>
    ''' <para>
    '''   On failure, throws an exception or returns a negative error code.
    ''' </para>
    '''/
    Public Function set_enabled(ByVal newval As Integer) As Integer
      Dim rest_val As String
      If (newval > 0) Then rest_val = "1" Else rest_val = "0"
      Return _setAttr("enabled", rest_val)
    End Function

    '''*
    ''' <summary>
    '''   Changes the DC bias configured for zero shift adjustment.
    ''' <para>
    '''   If your DC current reads positive when it should be zero, set up
    '''   a positive signalBias of the same value to fix the zero shift.
    '''   Remember to call the <c>saveToFlash()</c>
    '''   method of the module if the modification must be kept.
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <param name="newval">
    '''   a floating point number corresponding to the DC bias configured for zero shift adjustment
    ''' </param>
    ''' <para>
    ''' </para>
    ''' <returns>
    '''   <c>YAPI.SUCCESS</c> if the call succeeds.
    ''' </returns>
    ''' <para>
    '''   On failure, throws an exception or returns a negative error code.
    ''' </para>
    '''/
    Public Function set_signalBias(ByVal newval As Double) As Integer
      Dim rest_val As String
      rest_val = Ltrim(Str(Math.Round(newval * 65536.0)))
      Return _setAttr("signalBias", rest_val)
    End Function
    '''*
    ''' <summary>
    '''   Returns the DC bias configured for zero shift adjustment.
    ''' <para>
    '''   A positive bias value is used to correct a positive DC bias,
    '''   while a negative bias value is used to correct a negative DC bias.
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <returns>
    '''   a floating point number corresponding to the DC bias configured for zero shift adjustment
    ''' </returns>
    ''' <para>
    '''   On failure, throws an exception or returns <c>YVoltage.SIGNALBIAS_INVALID</c>.
    ''' </para>
    '''/
    Public Function get_signalBias() As Double
      Dim res As Double = 0
      If (Me._cacheExpiration <= YAPI.GetTickCount()) Then
        If (Me.load(YAPI._yapiContext.GetCacheValidity()) <> YAPI.SUCCESS) Then
          Return SIGNALBIAS_INVALID
        End If
      End If
      res = Me._signalBias
      Return res
    End Function

    '''*
    ''' <summary>
    '''   Retrieves a voltage sensor for a given identifier.
    ''' <para>
    '''   The identifier can be specified using several formats:
    ''' </para>
    ''' <para>
    ''' </para>
    ''' <para>
    '''   - FunctionLogicalName
    ''' </para>
    ''' <para>
    '''   - ModuleSerialNumber.FunctionIdentifier
    ''' </para>
    ''' <para>
    '''   - ModuleSerialNumber.FunctionLogicalName
    ''' </para>
    ''' <para>
    '''   - ModuleLogicalName.FunctionIdentifier
    ''' </para>
    ''' <para>
    '''   - ModuleLogicalName.FunctionLogicalName
    ''' </para>
    ''' <para>
    ''' </para>
    ''' <para>
    '''   This function does not require that the voltage sensor is online at the time
    '''   it is invoked. The returned object is nevertheless valid.
    '''   Use the method <c>YVoltage.isOnline()</c> to test if the voltage sensor is
    '''   indeed online at a given time. In case of ambiguity when looking for
    '''   a voltage sensor by logical name, no error is notified: the first instance
    '''   found is returned. The search is performed first by hardware name,
    '''   then by logical name.
    ''' </para>
    ''' <para>
    '''   If a call to this object's is_online() method returns FALSE although
    '''   you are certain that the matching device is plugged, make sure that you did
    '''   call registerHub() at application initialization time.
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <param name="func">
    '''   a string that uniquely characterizes the voltage sensor, for instance
    '''   <c>MOTORCTL.voltage</c>.
    ''' </param>
    ''' <returns>
    '''   a <c>YVoltage</c> object allowing you to drive the voltage sensor.
    ''' </returns>
    '''/
    Public Shared Function FindVoltage(func As String) As YVoltage
      Dim obj As YVoltage
      obj = CType(YFunction._FindFromCache("Voltage", func), YVoltage)
      If ((obj Is Nothing)) Then
        obj = New YVoltage(func)
        YFunction._AddToCache("Voltage", func, obj)
      End If
      Return obj
    End Function

    '''*
    ''' <summary>
    '''   Registers the callback function that is invoked on every change of advertised value.
    ''' <para>
    '''   The callback is invoked only during the execution of <c>ySleep</c> or <c>yHandleEvents</c>.
    '''   This provides control over the time when the callback is triggered. For good responsiveness, remember to call
    '''   one of these two functions periodically. To unregister a callback, pass a Nothing pointer as argument.
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <param name="callback">
    '''   the callback function to call, or a Nothing pointer. The callback function should take two
    '''   arguments: the function object of which the value has changed, and the character string describing
    '''   the new advertised value.
    ''' @noreturn
    ''' </param>
    '''/
    Public Overloads Function registerValueCallback(callback As YVoltageValueCallback) As Integer
      Dim val As String
      If (Not (callback Is Nothing)) Then
        YFunction._UpdateValueCallbackList(Me, True)
      Else
        YFunction._UpdateValueCallbackList(Me, False)
      End If
      Me._valueCallbackVoltage = callback
      REM // Immediately invoke value callback with current value
      If (Not (callback Is Nothing) AndAlso Me.isOnline()) Then
        val = Me._advertisedValue
        If (Not (val = "")) Then
          Me._invokeValueCallback(val)
        End If
      End If
      Return 0
    End Function

    Public Overrides Function _invokeValueCallback(value As String) As Integer
      If (Not (Me._valueCallbackVoltage Is Nothing)) Then
        Me._valueCallbackVoltage(Me, value)
      Else
        MyBase._invokeValueCallback(value)
      End If
      Return 0
    End Function

    '''*
    ''' <summary>
    '''   Registers the callback function that is invoked on every periodic timed notification.
    ''' <para>
    '''   The callback is invoked only during the execution of <c>ySleep</c> or <c>yHandleEvents</c>.
    '''   This provides control over the time when the callback is triggered. For good responsiveness, remember to call
    '''   one of these two functions periodically. To unregister a callback, pass a Nothing pointer as argument.
    ''' </para>
    ''' <para>
    ''' </para>
    ''' </summary>
    ''' <param name="callback">
    '''   the callback function to call, or a Nothing pointer. The callback function should take two
    '''   arguments: the function object of which the value has changed, and an <c>YMeasure</c> object describing
    '''   the new advertised value.
    ''' @noreturn
    ''' </param>
    '''/
    Public Overloads Function registerTimedReportCallback(callback As YVoltageTimedReportCallback) As Integer
      Dim sensor As YSensor
      sensor = Me
      If (Not (callback Is Nothing)) Then
        YFunction._UpdateTimedReportCallbackList(sensor, True)
      Else
        YFunction._UpdateTimedReportCallbackList(sensor, False)
      End If
      Me._timedReportCallbackVoltage = callback
      Return 0
    End Function

    Public Overrides Function _invokeTimedReportCallback(value As YMeasure) As Integer
      If (Not (Me._timedReportCallbackVoltage Is Nothing)) Then
        Me._timedReportCallbackVoltage(Me, value)
      Else
        MyBase._invokeTimedReportCallback(value)
      End If
      Return 0
    End Function

    '''*
    ''' <summary>
    '''   Calibrate the device by adjusting <c>signalBias</c> so that the current
    '''   input voltage is precisely seen as zero.
    ''' <para>
    '''   Before calling this method, make
    '''   sure to short the power source inputs as close as possible to the connector, and
    '''   to disconnect the load to ensure the wires don't capture radiated noise.
    '''   Remember to call the <c>saveToFlash()</c>
    '''   method of the module if the modification must be kept.
    ''' </para>
    ''' </summary>
    ''' <returns>
    '''   <c>YAPI.SUCCESS</c> if the call succeeds.
    ''' </returns>
    ''' <para>
    '''   On failure, throws an exception or returns a negative error code.
    ''' </para>
    '''/
    Public Overridable Function zeroAdjust() As Integer
      Dim currSignal As Double = 0
      Dim bias As Double = 0
      currSignal = Me.get_currentRawValue()
      bias = Me.get_signalBias() + currSignal
      If Not((bias > -0.5) AndAlso (bias < 0.5)) Then
        me._throw(YAPI.INVALID_ARGUMENT, "suspicious zeroAdjust, please ensure that the power source inputs are shorted")
        return YAPI.INVALID_ARGUMENT
      end if
      Return Me.set_signalBias(bias)
    End Function


    '''*
    ''' <summary>
    '''   Continues the enumeration of voltage sensors started using <c>yFirstVoltage()</c>.
    ''' <para>
    '''   Caution: You can't make any assumption about the returned voltage sensors order.
    '''   If you want to find a specific a voltage sensor, use <c>Voltage.findVoltage()</c>
    '''   and a hardwareID or a logical name.
    ''' </para>
    ''' </summary>
    ''' <returns>
    '''   a pointer to a <c>YVoltage</c> object, corresponding to
    '''   a voltage sensor currently online, or a <c>Nothing</c> pointer
    '''   if there are no more voltage sensors to enumerate.
    ''' </returns>
    '''/
    Public Function nextVoltage() As YVoltage
      Dim hwid As String = ""
      If (YISERR(_nextFunction(hwid))) Then
        Return Nothing
      End If
      If (hwid = "") Then
        Return Nothing
      End If
      Return YVoltage.FindVoltage(hwid)
    End Function

    '''*
    ''' <summary>
    '''   Starts the enumeration of voltage sensors currently accessible.
    ''' <para>
    '''   Use the method <c>YVoltage.nextVoltage()</c> to iterate on
    '''   next voltage sensors.
    ''' </para>
    ''' </summary>
    ''' <returns>
    '''   a pointer to a <c>YVoltage</c> object, corresponding to
    '''   the first voltage sensor currently online, or a <c>Nothing</c> pointer
    '''   if there are none.
    ''' </returns>
    '''/
    Public Shared Function FirstVoltage() As YVoltage
      Dim v_fundescr(1) As YFUN_DESCR
      Dim dev As YDEV_DESCR
      Dim neededsize, err As Integer
      Dim serial, funcId, funcName, funcVal As String
      Dim errmsg As String = ""
      Dim size As Integer = Marshal.SizeOf(v_fundescr(0))
      Dim p As IntPtr = Marshal.AllocHGlobal(Marshal.SizeOf(v_fundescr(0)))

      err = yapiGetFunctionsByClass("Voltage", 0, p, size, neededsize, errmsg)
      Marshal.Copy(p, v_fundescr, 0, 1)
      Marshal.FreeHGlobal(p)

      If (YISERR(err) Or (neededsize = 0)) Then
        Return Nothing
      End If
      serial = ""
      funcId = ""
      funcName = ""
      funcVal = ""
      errmsg = ""
      If (YISERR(yapiGetFunctionInfo(v_fundescr(0), dev, serial, funcId, funcName, funcVal, errmsg))) Then
        Return Nothing
      End If
      Return YVoltage.FindVoltage(serial + "." + funcId)
    End Function

    REM --- (end of YVoltage public methods declaration)

  End Class

  REM --- (YVoltage functions)

  '''*
  ''' <summary>
  '''   Retrieves a voltage sensor for a given identifier.
  ''' <para>
  '''   The identifier can be specified using several formats:
  ''' </para>
  ''' <para>
  ''' </para>
  ''' <para>
  '''   - FunctionLogicalName
  ''' </para>
  ''' <para>
  '''   - ModuleSerialNumber.FunctionIdentifier
  ''' </para>
  ''' <para>
  '''   - ModuleSerialNumber.FunctionLogicalName
  ''' </para>
  ''' <para>
  '''   - ModuleLogicalName.FunctionIdentifier
  ''' </para>
  ''' <para>
  '''   - ModuleLogicalName.FunctionLogicalName
  ''' </para>
  ''' <para>
  ''' </para>
  ''' <para>
  '''   This function does not require that the voltage sensor is online at the time
  '''   it is invoked. The returned object is nevertheless valid.
  '''   Use the method <c>YVoltage.isOnline()</c> to test if the voltage sensor is
  '''   indeed online at a given time. In case of ambiguity when looking for
  '''   a voltage sensor by logical name, no error is notified: the first instance
  '''   found is returned. The search is performed first by hardware name,
  '''   then by logical name.
  ''' </para>
  ''' <para>
  '''   If a call to this object's is_online() method returns FALSE although
  '''   you are certain that the matching device is plugged, make sure that you did
  '''   call registerHub() at application initialization time.
  ''' </para>
  ''' <para>
  ''' </para>
  ''' </summary>
  ''' <param name="func">
  '''   a string that uniquely characterizes the voltage sensor, for instance
  '''   <c>MOTORCTL.voltage</c>.
  ''' </param>
  ''' <returns>
  '''   a <c>YVoltage</c> object allowing you to drive the voltage sensor.
  ''' </returns>
  '''/
  Public Function yFindVoltage(ByVal func As String) As YVoltage
    Return YVoltage.FindVoltage(func)
  End Function

  '''*
  ''' <summary>
  '''   Starts the enumeration of voltage sensors currently accessible.
  ''' <para>
  '''   Use the method <c>YVoltage.nextVoltage()</c> to iterate on
  '''   next voltage sensors.
  ''' </para>
  ''' </summary>
  ''' <returns>
  '''   a pointer to a <c>YVoltage</c> object, corresponding to
  '''   the first voltage sensor currently online, or a <c>Nothing</c> pointer
  '''   if there are none.
  ''' </returns>
  '''/
  Public Function yFirstVoltage() As YVoltage
    Return YVoltage.FirstVoltage()
  End Function


  REM --- (end of YVoltage functions)

End Module
