/*********************************************************************
 *
 *  $Id: YQuadratureDecoder.cs 50689 2022-08-17 14:37:15Z mvuilleu $
 *
 *  Implements FindQuadratureDecoder(), the high-level API for QuadratureDecoder functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
namespace com.yoctopuce.YoctoAPI
{

//--- (YQuadratureDecoder return codes)
//--- (end of YQuadratureDecoder return codes)
//--- (YQuadratureDecoder class start)
/**
 * <summary>
 *   YQuadratureDecoder Class: quadrature decoder control interface, available for instance in the
 *   Yocto-MaxiKnob or the Yocto-PWM-Rx
 * <para>
 *   The <c>YQuadratureDecoder</c> class allows you to read and configure Yoctopuce quadrature decoders.
 *   It inherits from <c>YSensor</c> class the core functions to read measurements,
 *   to register callback functions, and to access the autonomous datalogger.
 * </para>
 * </summary>
 */
public class YQuadratureDecoder : YSensor
{
//--- (end of YQuadratureDecoder class start)
//--- (YQuadratureDecoder definitions)
    /**
     * <summary>
     *   invalid speed value
     * </summary>
     */
    public const  double SPEED_INVALID = YAPI.INVALID_DOUBLE;
    /**
     * <summary>
     *   invalid decoding value
     * </summary>
     */
    public const int DECODING_OFF = 0;
    public const int DECODING_ON = 1;
    public const int DECODING_INVALID = -1;
    /**
     * <summary>
     *   invalid edgesPerCycle value
     * </summary>
     */
    public const  int EDGESPERCYCLE_INVALID = YAPI.INVALID_UINT;
    protected double _speed = SPEED_INVALID;
    protected int _decoding = DECODING_INVALID;
    protected int _edgesPerCycle = EDGESPERCYCLE_INVALID;
    protected ValueCallback _valueCallbackQuadratureDecoder = null;
    protected TimedReportCallback _timedReportCallbackQuadratureDecoder = null;

    public new delegate Task ValueCallback(YQuadratureDecoder func, string value);
    public new delegate Task TimedReportCallback(YQuadratureDecoder func, YMeasure measure);
    //--- (end of YQuadratureDecoder definitions)


    /**
     * <summary>
     * </summary>
     * <param name="func">
     *   functionid
     * </param>
     */
    protected YQuadratureDecoder(YAPIContext ctx, string func)
        : base(ctx, func, "QuadratureDecoder")
    {
        //--- (YQuadratureDecoder attributes initialization)
        //--- (end of YQuadratureDecoder attributes initialization)
    }

    /**
     * <summary>
     * </summary>
     * <param name="func">
     *   functionid
     * </param>
     */
    protected YQuadratureDecoder(string func)
        : this(YAPI.imm_GetYCtx(), func)
    {
    }

    //--- (YQuadratureDecoder implementation)
#pragma warning disable 1998
    internal override void imm_parseAttr(YJSONObject json_val)
    {
        if (json_val.has("speed")) {
            _speed = Math.Round(json_val.getDouble("speed") / 65.536) / 1000.0;
        }
        if (json_val.has("decoding")) {
            _decoding = json_val.getInt("decoding") > 0 ? 1 : 0;
        }
        if (json_val.has("edgesPerCycle")) {
            _edgesPerCycle = json_val.getInt("edgesPerCycle");
        }
        base.imm_parseAttr(json_val);
    }

    /**
     * <summary>
     *   Changes the current expected position of the quadrature decoder.
     * <para>
     *   Invoking this function implicitly activates the quadrature decoder.
     * </para>
     * <para>
     * </para>
     * </summary>
     * <param name="newval">
     *   a floating point number corresponding to the current expected position of the quadrature decoder
     * </param>
     * <para>
     * </para>
     * <returns>
     *   <c>YAPI.SUCCESS</c> if the call succeeds.
     * </returns>
     * <para>
     *   On failure, throws an exception or returns a negative error code.
     * </para>
     */
    public async Task<int> set_currentValue(double  newval)
    {
        string rest_val;
        rest_val = Math.Round(newval * 65536.0).ToString();
        await _setAttr("currentValue",rest_val);
        return YAPI.SUCCESS;
    }

    /**
     * <summary>
     *   Returns the cycle frequency, in Hz.
     * <para>
     * </para>
     * <para>
     * </para>
     * </summary>
     * <returns>
     *   a floating point number corresponding to the cycle frequency, in Hz
     * </returns>
     * <para>
     *   On failure, throws an exception or returns <c>YQuadratureDecoder.SPEED_INVALID</c>.
     * </para>
     */
    public async Task<double> get_speed()
    {
        double res;
        if (_cacheExpiration <= YAPIContext.GetTickCount()) {
            if (await this.load(await _yapi.GetCacheValidity()) != YAPI.SUCCESS) {
                return SPEED_INVALID;
            }
        }
        res = _speed;
        return res;
    }


    /**
     * <summary>
     *   Returns the current activation state of the quadrature decoder.
     * <para>
     * </para>
     * <para>
     * </para>
     * </summary>
     * <returns>
     *   either <c>YQuadratureDecoder.DECODING_OFF</c> or <c>YQuadratureDecoder.DECODING_ON</c>, according
     *   to the current activation state of the quadrature decoder
     * </returns>
     * <para>
     *   On failure, throws an exception or returns <c>YQuadratureDecoder.DECODING_INVALID</c>.
     * </para>
     */
    public async Task<int> get_decoding()
    {
        int res;
        if (_cacheExpiration <= YAPIContext.GetTickCount()) {
            if (await this.load(await _yapi.GetCacheValidity()) != YAPI.SUCCESS) {
                return DECODING_INVALID;
            }
        }
        res = _decoding;
        return res;
    }


    /**
     * <summary>
     *   Changes the activation state of the quadrature decoder.
     * <para>
     *   Remember to call the <c>saveToFlash()</c>
     *   method of the module if the modification must be kept.
     * </para>
     * <para>
     * </para>
     * </summary>
     * <param name="newval">
     *   either <c>YQuadratureDecoder.DECODING_OFF</c> or <c>YQuadratureDecoder.DECODING_ON</c>, according
     *   to the activation state of the quadrature decoder
     * </param>
     * <para>
     * </para>
     * <returns>
     *   <c>YAPI.SUCCESS</c> if the call succeeds.
     * </returns>
     * <para>
     *   On failure, throws an exception or returns a negative error code.
     * </para>
     */
    public async Task<int> set_decoding(int  newval)
    {
        string rest_val;
        rest_val = (newval > 0 ? "1" : "0");
        await _setAttr("decoding",rest_val);
        return YAPI.SUCCESS;
    }

    /**
     * <summary>
     *   Returns the edge count per full cycle configuration setting.
     * <para>
     * </para>
     * <para>
     * </para>
     * </summary>
     * <returns>
     *   an integer corresponding to the edge count per full cycle configuration setting
     * </returns>
     * <para>
     *   On failure, throws an exception or returns <c>YQuadratureDecoder.EDGESPERCYCLE_INVALID</c>.
     * </para>
     */
    public async Task<int> get_edgesPerCycle()
    {
        int res;
        if (_cacheExpiration <= YAPIContext.GetTickCount()) {
            if (await this.load(await _yapi.GetCacheValidity()) != YAPI.SUCCESS) {
                return EDGESPERCYCLE_INVALID;
            }
        }
        res = _edgesPerCycle;
        return res;
    }


    /**
     * <summary>
     *   Changes the edge count per full cycle configuration setting.
     * <para>
     *   Remember to call the <c>saveToFlash()</c>
     *   method of the module if the modification must be kept.
     * </para>
     * <para>
     * </para>
     * </summary>
     * <param name="newval">
     *   an integer corresponding to the edge count per full cycle configuration setting
     * </param>
     * <para>
     * </para>
     * <returns>
     *   <c>YAPI.SUCCESS</c> if the call succeeds.
     * </returns>
     * <para>
     *   On failure, throws an exception or returns a negative error code.
     * </para>
     */
    public async Task<int> set_edgesPerCycle(int  newval)
    {
        string rest_val;
        rest_val = (newval).ToString();
        await _setAttr("edgesPerCycle",rest_val);
        return YAPI.SUCCESS;
    }

    /**
     * <summary>
     *   Retrieves a quadrature decoder for a given identifier.
     * <para>
     *   The identifier can be specified using several formats:
     * </para>
     * <para>
     * </para>
     * <para>
     *   - FunctionLogicalName
     * </para>
     * <para>
     *   - ModuleSerialNumber.FunctionIdentifier
     * </para>
     * <para>
     *   - ModuleSerialNumber.FunctionLogicalName
     * </para>
     * <para>
     *   - ModuleLogicalName.FunctionIdentifier
     * </para>
     * <para>
     *   - ModuleLogicalName.FunctionLogicalName
     * </para>
     * <para>
     * </para>
     * <para>
     *   This function does not require that the quadrature decoder is online at the time
     *   it is invoked. The returned object is nevertheless valid.
     *   Use the method <c>YQuadratureDecoder.isOnline()</c> to test if the quadrature decoder is
     *   indeed online at a given time. In case of ambiguity when looking for
     *   a quadrature decoder by logical name, no error is notified: the first instance
     *   found is returned. The search is performed first by hardware name,
     *   then by logical name.
     * </para>
     * <para>
     *   If a call to this object's is_online() method returns FALSE although
     *   you are certain that the matching device is plugged, make sure that you did
     *   call registerHub() at application initialization time.
     * </para>
     * <para>
     * </para>
     * </summary>
     * <param name="func">
     *   a string that uniquely characterizes the quadrature decoder, for instance
     *   <c>YMXBTN01.quadratureDecoder1</c>.
     * </param>
     * <returns>
     *   a <c>YQuadratureDecoder</c> object allowing you to drive the quadrature decoder.
     * </returns>
     */
    public static YQuadratureDecoder FindQuadratureDecoder(string func)
    {
        YQuadratureDecoder obj;
        obj = (YQuadratureDecoder) YFunction._FindFromCache("QuadratureDecoder", func);
        if (obj == null) {
            obj = new YQuadratureDecoder(func);
            YFunction._AddToCache("QuadratureDecoder",  func, obj);
        }
        return obj;
    }

    /**
     * <summary>
     *   Retrieves a quadrature decoder for a given identifier in a YAPI context.
     * <para>
     *   The identifier can be specified using several formats:
     * </para>
     * <para>
     * </para>
     * <para>
     *   - FunctionLogicalName
     * </para>
     * <para>
     *   - ModuleSerialNumber.FunctionIdentifier
     * </para>
     * <para>
     *   - ModuleSerialNumber.FunctionLogicalName
     * </para>
     * <para>
     *   - ModuleLogicalName.FunctionIdentifier
     * </para>
     * <para>
     *   - ModuleLogicalName.FunctionLogicalName
     * </para>
     * <para>
     * </para>
     * <para>
     *   This function does not require that the quadrature decoder is online at the time
     *   it is invoked. The returned object is nevertheless valid.
     *   Use the method <c>YQuadratureDecoder.isOnline()</c> to test if the quadrature decoder is
     *   indeed online at a given time. In case of ambiguity when looking for
     *   a quadrature decoder by logical name, no error is notified: the first instance
     *   found is returned. The search is performed first by hardware name,
     *   then by logical name.
     * </para>
     * </summary>
     * <param name="yctx">
     *   a YAPI context
     * </param>
     * <param name="func">
     *   a string that uniquely characterizes the quadrature decoder, for instance
     *   <c>YMXBTN01.quadratureDecoder1</c>.
     * </param>
     * <returns>
     *   a <c>YQuadratureDecoder</c> object allowing you to drive the quadrature decoder.
     * </returns>
     */
    public static YQuadratureDecoder FindQuadratureDecoderInContext(YAPIContext yctx,string func)
    {
        YQuadratureDecoder obj;
        obj = (YQuadratureDecoder) YFunction._FindFromCacheInContext(yctx,  "QuadratureDecoder", func);
        if (obj == null) {
            obj = new YQuadratureDecoder(yctx, func);
            YFunction._AddToCache("QuadratureDecoder",  func, obj);
        }
        return obj;
    }

    /**
     * <summary>
     *   Registers the callback function that is invoked on every change of advertised value.
     * <para>
     *   The callback is invoked only during the execution of <c>ySleep</c> or <c>yHandleEvents</c>.
     *   This provides control over the time when the callback is triggered. For good responsiveness, remember to call
     *   one of these two functions periodically. To unregister a callback, pass a null pointer as argument.
     * </para>
     * <para>
     * </para>
     * </summary>
     * <param name="callback">
     *   the callback function to call, or a null pointer. The callback function should take two
     *   arguments: the function object of which the value has changed, and the character string describing
     *   the new advertised value.
     * @noreturn
     * </param>
     */
    public async Task<int> registerValueCallback(ValueCallback callback)
    {
        string val;
        if (callback != null) {
            await YFunction._UpdateValueCallbackList(this, true);
        } else {
            await YFunction._UpdateValueCallbackList(this, false);
        }
        _valueCallbackQuadratureDecoder = callback;
        // Immediately invoke value callback with current value
        if (callback != null && await this.isOnline()) {
            val = _advertisedValue;
            if (!(val == "")) {
                await this._invokeValueCallback(val);
            }
        }
        return 0;
    }

    public override async Task<int> _invokeValueCallback(string value)
    {
        if (_valueCallbackQuadratureDecoder != null) {
            await _valueCallbackQuadratureDecoder(this, value);
        } else {
            await base._invokeValueCallback(value);
        }
        return 0;
    }

    /**
     * <summary>
     *   Registers the callback function that is invoked on every periodic timed notification.
     * <para>
     *   The callback is invoked only during the execution of <c>ySleep</c> or <c>yHandleEvents</c>.
     *   This provides control over the time when the callback is triggered. For good responsiveness, remember to call
     *   one of these two functions periodically. To unregister a callback, pass a null pointer as argument.
     * </para>
     * <para>
     * </para>
     * </summary>
     * <param name="callback">
     *   the callback function to call, or a null pointer. The callback function should take two
     *   arguments: the function object of which the value has changed, and an <c>YMeasure</c> object describing
     *   the new advertised value.
     * @noreturn
     * </param>
     */
    public async Task<int> registerTimedReportCallback(TimedReportCallback callback)
    {
        YSensor sensor;
        sensor = this;
        if (callback != null) {
            await YFunction._UpdateTimedReportCallbackList(sensor, true);
        } else {
            await YFunction._UpdateTimedReportCallbackList(sensor, false);
        }
        _timedReportCallbackQuadratureDecoder = callback;
        return 0;
    }

    public override async Task<int> _invokeTimedReportCallback(YMeasure value)
    {
        if (_timedReportCallbackQuadratureDecoder != null) {
            await _timedReportCallbackQuadratureDecoder(this, value);
        } else {
            await base._invokeTimedReportCallback(value);
        }
        return 0;
    }

    /**
     * <summary>
     *   Continues the enumeration of quadrature decoders started using <c>yFirstQuadratureDecoder()</c>.
     * <para>
     *   Caution: You can't make any assumption about the returned quadrature decoders order.
     *   If you want to find a specific a quadrature decoder, use <c>QuadratureDecoder.findQuadratureDecoder()</c>
     *   and a hardwareID or a logical name.
     * </para>
     * </summary>
     * <returns>
     *   a pointer to a <c>YQuadratureDecoder</c> object, corresponding to
     *   a quadrature decoder currently online, or a <c>null</c> pointer
     *   if there are no more quadrature decoders to enumerate.
     * </returns>
     */
    public YQuadratureDecoder nextQuadratureDecoder()
    {
        string next_hwid;
        try {
            string hwid = _yapi._yHash.imm_resolveHwID(_className, _func);
            next_hwid = _yapi._yHash.imm_getNextHardwareId(_className, hwid);
        } catch (YAPI_Exception) {
            next_hwid = null;
        }
        if(next_hwid == null) return null;
        return FindQuadratureDecoderInContext(_yapi, next_hwid);
    }

    /**
     * <summary>
     *   Starts the enumeration of quadrature decoders currently accessible.
     * <para>
     *   Use the method <c>YQuadratureDecoder.nextQuadratureDecoder()</c> to iterate on
     *   next quadrature decoders.
     * </para>
     * </summary>
     * <returns>
     *   a pointer to a <c>YQuadratureDecoder</c> object, corresponding to
     *   the first quadrature decoder currently online, or a <c>null</c> pointer
     *   if there are none.
     * </returns>
     */
    public static YQuadratureDecoder FirstQuadratureDecoder()
    {
        YAPIContext yctx = YAPI.imm_GetYCtx();
        string next_hwid = yctx._yHash.imm_getFirstHardwareId("QuadratureDecoder");
        if (next_hwid == null)  return null;
        return FindQuadratureDecoderInContext(yctx, next_hwid);
    }

    /**
     * <summary>
     *   Starts the enumeration of quadrature decoders currently accessible.
     * <para>
     *   Use the method <c>YQuadratureDecoder.nextQuadratureDecoder()</c> to iterate on
     *   next quadrature decoders.
     * </para>
     * </summary>
     * <param name="yctx">
     *   a YAPI context.
     * </param>
     * <returns>
     *   a pointer to a <c>YQuadratureDecoder</c> object, corresponding to
     *   the first quadrature decoder currently online, or a <c>null</c> pointer
     *   if there are none.
     * </returns>
     */
    public static YQuadratureDecoder FirstQuadratureDecoderInContext(YAPIContext yctx)
    {
        string next_hwid = yctx._yHash.imm_getFirstHardwareId("QuadratureDecoder");
        if (next_hwid == null)  return null;
        return FindQuadratureDecoderInContext(yctx, next_hwid);
    }

#pragma warning restore 1998
    //--- (end of YQuadratureDecoder implementation)
}
}

