# -*- coding: utf-8 -*-
#*********************************************************************
#*
#* $Id: yocto_multicellweighscale.py 29478 2017-12-21 08:10:05Z seb $
#*
#* Implements yFindMultiCellWeighScale(), the high-level API for MultiCellWeighScale functions
#*
#* - - - - - - - - - License information: - - - - - - - - -
#*
#*  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
#*
#*  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
#*  non-exclusive license to use, modify, copy and integrate this
#*  file into your software for the sole purpose of interfacing
#*  with Yoctopuce products.
#*
#*  You may reproduce and distribute copies of this file in
#*  source or object form, as long as the sole purpose of this
#*  code is to interface with Yoctopuce products. You must retain
#*  this notice in the distributed source file.
#*
#*  You should refer to Yoctopuce General Terms and Conditions
#*  for additional information regarding your rights and
#*  obligations.
#*
#*  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
#*  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
#*  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
#*  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
#*  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
#*  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
#*  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
#*  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
#*  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
#*  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
#*  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
#*  WARRANTY, OR OTHERWISE.
#*
#*********************************************************************/


__docformat__ = 'restructuredtext en'
from yocto_api import *


#--- (YMultiCellWeighScale class start)
#noinspection PyProtectedMember
class YMultiCellWeighScale(YSensor):
    """
    The YMultiCellWeighScale class provides a weight measurement from a set of ratiometric load cells
    sensor. It can be used to control the bridge excitation parameters, in order to avoid
    measure shifts caused by temperature variation in the electronics, and can also
    automatically apply an additional correction factor based on temperature to
    compensate for offsets in the load cells themselves.

    """
#--- (end of YMultiCellWeighScale class start)
    #--- (YMultiCellWeighScale return codes)
    #--- (end of YMultiCellWeighScale return codes)
    #--- (YMultiCellWeighScale dlldef)
    #--- (end of YMultiCellWeighScale dlldef)
    #--- (YMultiCellWeighScale definitions)
    CELLCOUNT_INVALID = YAPI.INVALID_UINT
    COMPTEMPADAPTRATIO_INVALID = YAPI.INVALID_DOUBLE
    COMPTEMPAVG_INVALID = YAPI.INVALID_DOUBLE
    COMPTEMPCHG_INVALID = YAPI.INVALID_DOUBLE
    COMPENSATION_INVALID = YAPI.INVALID_DOUBLE
    ZEROTRACKING_INVALID = YAPI.INVALID_DOUBLE
    COMMAND_INVALID = YAPI.INVALID_STRING
    EXCITATION_OFF = 0
    EXCITATION_DC = 1
    EXCITATION_AC = 2
    EXCITATION_INVALID = -1
    #--- (end of YMultiCellWeighScale definitions)

    def __init__(self, func):
        super(YMultiCellWeighScale, self).__init__(func)
        self._className = 'MultiCellWeighScale'
        #--- (YMultiCellWeighScale attributes)
        self._callback = None
        self._cellCount = YMultiCellWeighScale.CELLCOUNT_INVALID
        self._excitation = YMultiCellWeighScale.EXCITATION_INVALID
        self._compTempAdaptRatio = YMultiCellWeighScale.COMPTEMPADAPTRATIO_INVALID
        self._compTempAvg = YMultiCellWeighScale.COMPTEMPAVG_INVALID
        self._compTempChg = YMultiCellWeighScale.COMPTEMPCHG_INVALID
        self._compensation = YMultiCellWeighScale.COMPENSATION_INVALID
        self._zeroTracking = YMultiCellWeighScale.ZEROTRACKING_INVALID
        self._command = YMultiCellWeighScale.COMMAND_INVALID
        #--- (end of YMultiCellWeighScale attributes)

    #--- (YMultiCellWeighScale implementation)
    def _parseAttr(self, json_val):
        if json_val.has("cellCount"):
            self._cellCount = json_val.getInt("cellCount")
        if json_val.has("excitation"):
            self._excitation = json_val.getInt("excitation")
        if json_val.has("compTempAdaptRatio"):
            self._compTempAdaptRatio = round(json_val.getDouble("compTempAdaptRatio") * 1000.0 / 65536.0) / 1000.0
        if json_val.has("compTempAvg"):
            self._compTempAvg = round(json_val.getDouble("compTempAvg") * 1000.0 / 65536.0) / 1000.0
        if json_val.has("compTempChg"):
            self._compTempChg = round(json_val.getDouble("compTempChg") * 1000.0 / 65536.0) / 1000.0
        if json_val.has("compensation"):
            self._compensation = round(json_val.getDouble("compensation") * 1000.0 / 65536.0) / 1000.0
        if json_val.has("zeroTracking"):
            self._zeroTracking = round(json_val.getDouble("zeroTracking") * 1000.0 / 65536.0) / 1000.0
        if json_val.has("command"):
            self._command = json_val.getString("command")
        super(YMultiCellWeighScale, self)._parseAttr(json_val)

    def get_cellCount(self):
        """
        Returns the number of load cells in use.

        @return an integer corresponding to the number of load cells in use

        On failure, throws an exception or returns YMultiCellWeighScale.CELLCOUNT_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.CELLCOUNT_INVALID
        res = self._cellCount
        return res

    def set_cellCount(self, newval):
        """
        Changes the number of load cells in use.

        @param newval : an integer corresponding to the number of load cells in use

        @return YAPI.SUCCESS if the call succeeds.

        On failure, throws an exception or returns a negative error code.
        """
        rest_val = str(newval)
        return self._setAttr("cellCount", rest_val)

    def get_excitation(self):
        """
        Returns the current load cell bridge excitation method.

        @return a value among YMultiCellWeighScale.EXCITATION_OFF, YMultiCellWeighScale.EXCITATION_DC and
        YMultiCellWeighScale.EXCITATION_AC corresponding to the current load cell bridge excitation method

        On failure, throws an exception or returns YMultiCellWeighScale.EXCITATION_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.EXCITATION_INVALID
        res = self._excitation
        return res

    def set_excitation(self, newval):
        """
        Changes the current load cell bridge excitation method.

        @param newval : a value among YMultiCellWeighScale.EXCITATION_OFF,
        YMultiCellWeighScale.EXCITATION_DC and YMultiCellWeighScale.EXCITATION_AC corresponding to the
        current load cell bridge excitation method

        @return YAPI.SUCCESS if the call succeeds.

        On failure, throws an exception or returns a negative error code.
        """
        rest_val = str(newval)
        return self._setAttr("excitation", rest_val)

    def set_compTempAdaptRatio(self, newval):
        """
        Changes the averaged temperature update rate, in percents.
        The averaged temperature is updated every 10 seconds, by applying this adaptation rate
        to the difference between the measures ambiant temperature and the current compensation
        temperature. The standard rate is 0.04 percents, and the maximal rate is 65 percents.

        @param newval : a floating point number corresponding to the averaged temperature update rate, in percents

        @return YAPI.SUCCESS if the call succeeds.

        On failure, throws an exception or returns a negative error code.
        """
        rest_val = str(int(round(newval * 65536.0, 1)))
        return self._setAttr("compTempAdaptRatio", rest_val)

    def get_compTempAdaptRatio(self):
        """
        Returns the averaged temperature update rate, in percents.
        The averaged temperature is updated every 10 seconds, by applying this adaptation rate
        to the difference between the measures ambiant temperature and the current compensation
        temperature. The standard rate is 0.04 percents, and the maximal rate is 65 percents.

        @return a floating point number corresponding to the averaged temperature update rate, in percents

        On failure, throws an exception or returns YMultiCellWeighScale.COMPTEMPADAPTRATIO_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.COMPTEMPADAPTRATIO_INVALID
        res = self._compTempAdaptRatio
        return res

    def get_compTempAvg(self):
        """
        Returns the current averaged temperature, used for thermal compensation.

        @return a floating point number corresponding to the current averaged temperature, used for thermal compensation

        On failure, throws an exception or returns YMultiCellWeighScale.COMPTEMPAVG_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.COMPTEMPAVG_INVALID
        res = self._compTempAvg
        return res

    def get_compTempChg(self):
        """
        Returns the current temperature variation, used for thermal compensation.

        @return a floating point number corresponding to the current temperature variation, used for
        thermal compensation

        On failure, throws an exception or returns YMultiCellWeighScale.COMPTEMPCHG_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.COMPTEMPCHG_INVALID
        res = self._compTempChg
        return res

    def get_compensation(self):
        """
        Returns the current current thermal compensation value.

        @return a floating point number corresponding to the current current thermal compensation value

        On failure, throws an exception or returns YMultiCellWeighScale.COMPENSATION_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.COMPENSATION_INVALID
        res = self._compensation
        return res

    def set_zeroTracking(self, newval):
        """
        Changes the compensation temperature update rate, in percents.

        @param newval : a floating point number corresponding to the compensation temperature update rate, in percents

        @return YAPI.SUCCESS if the call succeeds.

        On failure, throws an exception or returns a negative error code.
        """
        rest_val = str(int(round(newval * 65536.0, 1)))
        return self._setAttr("zeroTracking", rest_val)

    def get_zeroTracking(self):
        """
        Returns the zero tracking threshold value. When this threshold is larger than
        zero, any measure under the threshold will automatically be ignored and the
        zero compensation will be updated.

        @return a floating point number corresponding to the zero tracking threshold value

        On failure, throws an exception or returns YMultiCellWeighScale.ZEROTRACKING_INVALID.
        """
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.ZEROTRACKING_INVALID
        res = self._zeroTracking
        return res

    def get_command(self):
        # res
        if self._cacheExpiration <= YAPI.GetTickCount():
            if self.load(YAPI.DefaultCacheValidity) != YAPI.SUCCESS:
                return YMultiCellWeighScale.COMMAND_INVALID
        res = self._command
        return res

    def set_command(self, newval):
        rest_val = newval
        return self._setAttr("command", rest_val)

    @staticmethod
    def FindMultiCellWeighScale(func):
        """
        Retrieves a multi-cell weighing scale sensor for a given identifier.
        The identifier can be specified using several formats:
        <ul>
        <li>FunctionLogicalName</li>
        <li>ModuleSerialNumber.FunctionIdentifier</li>
        <li>ModuleSerialNumber.FunctionLogicalName</li>
        <li>ModuleLogicalName.FunctionIdentifier</li>
        <li>ModuleLogicalName.FunctionLogicalName</li>
        </ul>

        This function does not require that the multi-cell weighing scale sensor is online at the time
        it is invoked. The returned object is nevertheless valid.
        Use the method YMultiCellWeighScale.isOnline() to test if the multi-cell weighing scale sensor is
        indeed online at a given time. In case of ambiguity when looking for
        a multi-cell weighing scale sensor by logical name, no error is notified: the first instance
        found is returned. The search is performed first by hardware name,
        then by logical name.

        If a call to this object's is_online() method returns FALSE although
        you are certain that the matching device is plugged, make sure that you did
        call registerHub() at application initialization time.

        @param func : a string that uniquely characterizes the multi-cell weighing scale sensor

        @return a YMultiCellWeighScale object allowing you to drive the multi-cell weighing scale sensor.
        """
        # obj
        obj = YFunction._FindFromCache("MultiCellWeighScale", func)
        if obj is None:
            obj = YMultiCellWeighScale(func)
            YFunction._AddToCache("MultiCellWeighScale", func, obj)
        return obj

    def tare(self):
        """
        Adapts the load cells signal bias (stored in the corresponding genericSensor)
        so that the current signal corresponds to a zero weight.

        @return YAPI.SUCCESS if the call succeeds.

        On failure, throws an exception or returns a negative error code.
        """
        return self.set_command("T")

    def setupSpan(self, currWeight, maxWeight):
        """
        Configures the load cells span parameters (stored in the corresponding genericSensors)
        so that the current signal corresponds to the specified reference weight.

        @param currWeight : reference weight presently on the load cell.
        @param maxWeight : maximum weight to be expectect on the load cell.

        @return YAPI.SUCCESS if the call succeeds.

        On failure, throws an exception or returns a negative error code.
        """
        return self.set_command("S" + str(int(round(1000*currWeight))) + ":" + str(int(round(1000*maxWeight))))

    def nextMultiCellWeighScale(self):
        """
        Continues the enumeration of multi-cell weighing scale sensors started using yFirstMultiCellWeighScale().

        @return a pointer to a YMultiCellWeighScale object, corresponding to
                a multi-cell weighing scale sensor currently online, or a None pointer
                if there are no more multi-cell weighing scale sensors to enumerate.
        """
        hwidRef = YRefParam()
        if YAPI.YISERR(self._nextFunction(hwidRef)):
            return None
        if hwidRef.value == "":
            return None
        return YMultiCellWeighScale.FindMultiCellWeighScale(hwidRef.value)

#--- (end of YMultiCellWeighScale implementation)

#--- (YMultiCellWeighScale functions)

    @staticmethod
    def FirstMultiCellWeighScale():
        """
        Starts the enumeration of multi-cell weighing scale sensors currently accessible.
        Use the method YMultiCellWeighScale.nextMultiCellWeighScale() to iterate on
        next multi-cell weighing scale sensors.

        @return a pointer to a YMultiCellWeighScale object, corresponding to
                the first multi-cell weighing scale sensor currently online, or a None pointer
                if there are none.
        """
        devRef = YRefParam()
        neededsizeRef = YRefParam()
        serialRef = YRefParam()
        funcIdRef = YRefParam()
        funcNameRef = YRefParam()
        funcValRef = YRefParam()
        errmsgRef = YRefParam()
        size = YAPI.C_INTSIZE
        #noinspection PyTypeChecker,PyCallingNonCallable
        p = (ctypes.c_int * 1)()
        err = YAPI.apiGetFunctionsByClass("MultiCellWeighScale", 0, p, size, neededsizeRef, errmsgRef)

        if YAPI.YISERR(err) or not neededsizeRef.value:
            return None

        if YAPI.YISERR(
                YAPI.yapiGetFunctionInfo(p[0], devRef, serialRef, funcIdRef, funcNameRef, funcValRef, errmsgRef)):
            return None

        return YMultiCellWeighScale.FindMultiCellWeighScale(serialRef.value + "." + funcIdRef.value)

#--- (end of YMultiCellWeighScale functions)
