/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements YVirtualSensorProxy, the Proxy API for VirtualSensor
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Timers;
using System.Globalization;
using System.Text.RegularExpressions;
using YoctoLib;

namespace YoctoProxyAPI
{
    //--- (YVirtualSensor class start)
    static public partial class YoctoProxyManager
    {
        public static YVirtualSensorProxy FindVirtualSensor(string name)
        {
            // cases to handle:
            // name =""  no matching unknwn
            // name =""  unknown exists
            // name != "" no  matching unknown
            // name !="" unknown exists
            YVirtualSensor func = null;
            YVirtualSensorProxy res = (YVirtualSensorProxy)YFunctionProxy.FindSimilarUnknownFunction("YVirtualSensorProxy");

            if (name == "") {
                if (res != null) return res;
                res = (YVirtualSensorProxy)YFunctionProxy.FindSimilarKnownFunction("YVirtualSensorProxy");
                if (res != null) return res;
                func = YVirtualSensor.FirstVirtualSensor();
                if (func != null) {
                    name = func.get_hardwareId();
                    if (func.get_userData() != null) {
                        return (YVirtualSensorProxy)func.get_userData();
                    }
                }
            } else {
                func = YVirtualSensor.FindVirtualSensor(name);
                if (func.get_userData() != null) {
                    return (YVirtualSensorProxy)func.get_userData();
                }
            }
            if (res == null) {
                res = new YVirtualSensorProxy(func, name);
            }
            if (func != null) {
                res.linkToHardware(name);
                if(func.isOnline()) res.arrival();
            }
            return res;
        }
    }

/**
 * <summary>
 *   The <c>YVirtualSensor</c> class allows you to use Yoctopuce virtual sensors.
 * <para>
 *   These sensors make it possible to show external data collected by the user
 *   as a Yoctopuce Sensor. This class inherits from <c>YSensor</c> class the core
 *   functions to read measurements, to register callback functions, and to access
 *   the autonomous datalogger. It adds the ability to change the sensor value as
 *   needed, or to mark current value as invalid.
 * </para>
 * <para>
 * </para>
 * </summary>
 */
    public class YVirtualSensorProxy : YSensorProxy
    {
        /**
         * <summary>
         *   Retrieves a virtual sensor for a given identifier.
         * <para>
         *   The identifier can be specified using several formats:
         * </para>
         * <para>
         * </para>
         * <para>
         *   - FunctionLogicalName
         * </para>
         * <para>
         *   - ModuleSerialNumber.FunctionIdentifier
         * </para>
         * <para>
         *   - ModuleSerialNumber.FunctionLogicalName
         * </para>
         * <para>
         *   - ModuleLogicalName.FunctionIdentifier
         * </para>
         * <para>
         *   - ModuleLogicalName.FunctionLogicalName
         * </para>
         * <para>
         * </para>
         * <para>
         *   This function does not require that the virtual sensor is online at the time
         *   it is invoked. The returned object is nevertheless valid.
         *   Use the method <c>YVirtualSensor.isOnline()</c> to test if the virtual sensor is
         *   indeed online at a given time. In case of ambiguity when looking for
         *   a virtual sensor by logical name, no error is notified: the first instance
         *   found is returned. The search is performed first by hardware name,
         *   then by logical name.
         * </para>
         * <para>
         *   If a call to this object's is_online() method returns FALSE although
         *   you are certain that the matching device is plugged, make sure that you did
         *   call registerHub() at application initialization time.
         * </para>
         * <para>
         * </para>
         * </summary>
         * <param name="func">
         *   a string that uniquely characterizes the virtual sensor, for instance
         *   <c>MyDevice.virtualSensor1</c>.
         * </param>
         * <returns>
         *   a <c>YVirtualSensor</c> object allowing you to drive the virtual sensor.
         * </returns>
         */
        public static YVirtualSensorProxy FindVirtualSensor(string func)
        {
            return YoctoProxyManager.FindVirtualSensor(func);
        }
        //--- (end of YVirtualSensor class start)
        //--- (YVirtualSensor definitions)
        public const double _InvalidValue_INVALID = Double.NaN;

        // reference to real YoctoAPI object
        protected new YVirtualSensor _func;
        // property cache
        protected double _invalidValue = _InvalidValue_INVALID;
        protected double _currentRawValue = _CurrentRawValue_INVALID;
        //--- (end of YVirtualSensor definitions)

        //--- (YVirtualSensor implementation)
        internal YVirtualSensorProxy(YVirtualSensor hwd, string instantiationName) : base(hwd, instantiationName)
        {
            InternalStuff.log("VirtualSensor " + instantiationName + " instantiation");
            base_init(hwd, instantiationName);
        }

        // perform the initial setup that may be done without a YoctoAPI object (hwd can be null)
        internal override void base_init(YFunction hwd, string instantiationName)
        {
            _func = (YVirtualSensor) hwd;
           	base.base_init(hwd, instantiationName);
        }

        // link the instance to a real YoctoAPI object
        internal override void linkToHardware(string hwdName)
        {
            YVirtualSensor hwd = YVirtualSensor.FindVirtualSensor(hwdName);
            // first redo base_init to update all _func pointers
            base_init(hwd, hwdName);
            // then setup Yocto-API pointers and callbacks
            init(hwd);
        }

        // perform the 2nd stage setup that requires YoctoAPI object
        protected void init(YVirtualSensor hwd)
        {
            if (hwd == null) return;
            base.init(hwd);
            InternalStuff.log("registering VirtualSensor callback");
            _func.registerValueCallback(valueChangeCallback);
        }

        /**
         * <summary>
         *   Enumerates all functions of type VirtualSensor available on the devices
         *   currently reachable by the library, and returns their unique hardware ID.
         * <para>
         *   Each of these IDs can be provided as argument to the method
         *   <c>YVirtualSensor.FindVirtualSensor</c> to obtain an object that can control the
         *   corresponding device.
         * </para>
         * </summary>
         * <returns>
         *   an array of strings, each string containing the unique hardwareId
         *   of a device function currently connected.
         * </returns>
         */
        public static new string[] GetSimilarFunctions()
        {
            List<string> res = new List<string>();
            YVirtualSensor it = YVirtualSensor.FirstVirtualSensor();
            while (it != null)
            {
                res.Add(it.get_hardwareId());
                it = it.nextVirtualSensor();
            }
            return res.ToArray();
        }

        protected override void functionArrival()
        {
            base.functionArrival();
            _currentRawValue = _func.get_currentRawValue();
        }

        protected override void moduleConfigHasChanged()
       	{
            base.moduleConfigHasChanged();
            _invalidValue = _func.get_invalidValue();
        }

        /**
         * <summary>
         *   Changes the measuring unit for the measured value.
         * <para>
         *   Remember to call the <c>saveToFlash()</c> method of the module if the
         *   modification must be kept.
         * </para>
         * <para>
         * </para>
         * </summary>
         * <param name="newval">
         *   a string corresponding to the measuring unit for the measured value
         * </param>
         * <para>
         * </para>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public int set_unit(string newval)
        {
            if (_func == null) {
                throw new YoctoApiProxyException("No VirtualSensor connected");
            }
            if (newval == _Unit_INVALID) {
                return YAPI.SUCCESS;
            }
            return _func.set_unit(newval);
        }

        // property with cached value for instant access (derived from advertised value)
        /// <value>current value of the sensor (raw value, before calibration)</value>
        public double CurrentRawValue
        {
            get
            {
                if (_func == null) {
                    return _CurrentRawValue_INVALID;
                }
                if (_online) {
                    return _currentRawValue;
                }
                return _CurrentRawValue_INVALID;
            }
            set
            {
                setprop_currentRawValue(value);
            }
        }

        // private helper for magic property
        private void setprop_currentRawValue(double newval)
        {
            if (_func == null) {
                return;
            }
            if (!(_online)) {
                return;
            }
            if (newval == _CurrentRawValue_INVALID) {
                return;
            }
            if (newval == _currentRawValue) {
                return;
            }
            _func.set_currentRawValue(newval);
            _currentRawValue = newval;
        }

        protected override void valueChangeCallback(YFunction source, string value)
        {
            base.valueChangeCallback(source, value);
            _currentRawValue = _func.get_currentRawValue();
        }

        /**
         * <summary>
         *   Changes the current value of the sensor (raw value, before calibration).
         * <para>
         * </para>
         * <para>
         * </para>
         * </summary>
         * <param name="newval">
         *   a floating point number corresponding to the current value of the sensor (raw value, before calibration)
         * </param>
         * <para>
         * </para>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public int set_currentRawValue(double newval)
        {
            if (_func == null) {
                throw new YoctoApiProxyException("No VirtualSensor connected");
            }
            if (newval == _CurrentRawValue_INVALID) {
                return YAPI.SUCCESS;
            }
            return _func.set_currentRawValue(newval);
        }

        /**
         * <summary>
         *   Changes the invalid value of the sensor, returned if the sensor is read when in invalid state
         *   (for instance before having been set).
         * <para>
         *   Remember to call the <c>saveToFlash()</c>
         *   method of the module if the modification must be kept.
         * </para>
         * <para>
         * </para>
         * </summary>
         * <param name="newval">
         *   a floating point number corresponding to the invalid value of the sensor, returned if the sensor is
         *   read when in invalid state
         *   (for instance before having been set)
         * </param>
         * <para>
         * </para>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public int set_invalidValue(double newval)
        {
            if (_func == null) {
                throw new YoctoApiProxyException("No VirtualSensor connected");
            }
            if (newval == _InvalidValue_INVALID) {
                return YAPI.SUCCESS;
            }
            return _func.set_invalidValue(newval);
        }

        /**
         * <summary>
         *   Returns the invalid value of the sensor, returned if the sensor is read when in invalid state
         *   (for instance before having been set).
         * <para>
         * </para>
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   a floating point number corresponding to the invalid value of the sensor, returned if the sensor is
         *   read when in invalid state
         *   (for instance before having been set)
         * </returns>
         * <para>
         *   On failure, throws an exception or returns <c>YVirtualSensor.INVALIDVALUE_INVALID</c>.
         * </para>
         */
        public double get_invalidValue()
        {
            double res;
            if (_func == null) {
                throw new YoctoApiProxyException("No VirtualSensor connected");
            }
            res = _func.get_invalidValue();
            if (res == YAPI.INVALID_DOUBLE) {
                res = Double.NaN;
            }
            return res;
        }

        // property with cached value for instant access (configuration)
        /// <value>Invalid value of the sensor, returned if the sensor is read when in invalid state</value>
        public double InvalidValue
        {
            get
            {
                if (_func == null) {
                    return _InvalidValue_INVALID;
                }
                if (_online) {
                    return _invalidValue;
                }
                return _InvalidValue_INVALID;
            }
            set
            {
                setprop_invalidValue(value);
            }
        }

        // private helper for magic property
        private void setprop_invalidValue(double newval)
        {
            if (_func == null) {
                return;
            }
            if (!(_online)) {
                return;
            }
            if (newval == _InvalidValue_INVALID) {
                return;
            }
            if (newval == _invalidValue) {
                return;
            }
            _func.set_invalidValue(newval);
            _invalidValue = newval;
        }

        /**
         * <summary>
         *   Changes the current sensor state to invalid (as if no value would have been ever set).
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int set_sensorAsInvalid()
        {
            if (_func == null) {
                throw new YoctoApiProxyException("No VirtualSensor connected");
            }
            return _func.set_sensorAsInvalid();
        }
    }
    //--- (end of YVirtualSensor implementation)
}

