/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements YSoundLevelProxy, the Proxy API for SoundLevel
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Timers;
using System.Globalization;
using System.Text.RegularExpressions;
using YoctoLib;

namespace YoctoProxyAPI
{
    //--- (YSoundLevel class start)
    static public partial class YoctoProxyManager
    {
        public static YSoundLevelProxy FindSoundLevel(string name)
        {
            // cases to handle:
            // name =""  no matching unknwn
            // name =""  unknown exists
            // name != "" no  matching unknown
            // name !="" unknown exists
            YSoundLevel func = null;
            YSoundLevelProxy res = (YSoundLevelProxy)YFunctionProxy.FindSimilarUnknownFunction("YSoundLevelProxy");

            if (name == "") {
                if (res != null) return res;
                res = (YSoundLevelProxy)YFunctionProxy.FindSimilarKnownFunction("YSoundLevelProxy");
                if (res != null) return res;
                func = YSoundLevel.FirstSoundLevel();
                if (func != null) {
                    name = func.get_hardwareId();
                    if (func.get_userData() != null) {
                        return (YSoundLevelProxy)func.get_userData();
                    }
                }
            } else {
                func = YSoundLevel.FindSoundLevel(name);
                if (func.get_userData() != null) {
                    return (YSoundLevelProxy)func.get_userData();
                }
            }
            if (res == null) {
                res = new YSoundLevelProxy(func, name);
            }
            if (func != null) {
                res.linkToHardware(name);
                if(func.isOnline()) res.arrival();
            }
            return res;
        }
    }

/**
 * <summary>
 *   The <c>YSoundLevel</c> class allows you to read and configure Yoctopuce sound pressure level meters.
 * <para>
 *   It inherits from <c>YSensor</c> class the core functions to read measurements,
 *   to register callback functions, and to access the autonomous datalogger.
 * </para>
 * <para>
 * </para>
 * </summary>
 */
    public class YSoundLevelProxy : YSensorProxy
    {
        /**
         * <summary>
         *   Retrieves a sound pressure level meter for a given identifier.
         * <para>
         *   The identifier can be specified using several formats:
         * </para>
         * <para>
         * </para>
         * <para>
         *   - FunctionLogicalName
         * </para>
         * <para>
         *   - ModuleSerialNumber.FunctionIdentifier
         * </para>
         * <para>
         *   - ModuleSerialNumber.FunctionLogicalName
         * </para>
         * <para>
         *   - ModuleLogicalName.FunctionIdentifier
         * </para>
         * <para>
         *   - ModuleLogicalName.FunctionLogicalName
         * </para>
         * <para>
         * </para>
         * <para>
         *   This function does not require that the sound pressure level meter is online at the time
         *   it is invoked. The returned object is nevertheless valid.
         *   Use the method <c>YSoundLevel.isOnline()</c> to test if the sound pressure level meter is
         *   indeed online at a given time. In case of ambiguity when looking for
         *   a sound pressure level meter by logical name, no error is notified: the first instance
         *   found is returned. The search is performed first by hardware name,
         *   then by logical name.
         * </para>
         * <para>
         *   If a call to this object's is_online() method returns FALSE although
         *   you are certain that the matching device is plugged, make sure that you did
         *   call registerHub() at application initialization time.
         * </para>
         * <para>
         * </para>
         * </summary>
         * <param name="func">
         *   a string that uniquely characterizes the sound pressure level meter, for instance
         *   <c>MyDevice.soundLevel1</c>.
         * </param>
         * <returns>
         *   a <c>YSoundLevel</c> object allowing you to drive the sound pressure level meter.
         * </returns>
         */
        public static YSoundLevelProxy FindSoundLevel(string func)
        {
            return YoctoProxyManager.FindSoundLevel(func);
        }
        //--- (end of YSoundLevel class start)
        //--- (YSoundLevel definitions)
        public const string _Label_INVALID = YAPI.INVALID_STRING;
        public const int _IntegrationTime_INVALID = -1;

        // reference to real YoctoAPI object
        protected new YSoundLevel _func;
        // property cache
        protected string _label = _Label_INVALID;
        //--- (end of YSoundLevel definitions)

        //--- (YSoundLevel implementation)
        internal YSoundLevelProxy(YSoundLevel hwd, string instantiationName) : base(hwd, instantiationName)
        {
            InternalStuff.log("SoundLevel " + instantiationName + " instantiation");
            base_init(hwd, instantiationName);
        }

        // perform the initial setup that may be done without a YoctoAPI object (hwd can be null)
        internal override void base_init(YFunction hwd, string instantiationName)
        {
            _func = (YSoundLevel) hwd;
           	base.base_init(hwd, instantiationName);
        }

        // link the instance to a real YoctoAPI object
        internal override void linkToHardware(string hwdName)
        {
            YSoundLevel hwd = YSoundLevel.FindSoundLevel(hwdName);
            // first redo base_init to update all _func pointers
            base_init(hwd, hwdName);
            // then setup Yocto-API pointers and callbacks
            init(hwd);
        }

        // perform the 2nd stage setup that requires YoctoAPI object
        protected void init(YSoundLevel hwd)
        {
            if (hwd == null) return;
            base.init(hwd);
            InternalStuff.log("registering SoundLevel callback");
            _func.registerValueCallback(valueChangeCallback);
        }

        /**
         * <summary>
         *   Enumerates all functions of type SoundLevel available on the devices
         *   currently reachable by the library, and returns their unique hardware ID.
         * <para>
         *   Each of these IDs can be provided as argument to the method
         *   <c>YSoundLevel.FindSoundLevel</c> to obtain an object that can control the
         *   corresponding device.
         * </para>
         * </summary>
         * <returns>
         *   an array of strings, each string containing the unique hardwareId
         *   of a device function currently connected.
         * </returns>
         */
        public static new string[] GetSimilarFunctions()
        {
            List<string> res = new List<string>();
            YSoundLevel it = YSoundLevel.FirstSoundLevel();
            while (it != null)
            {
                res.Add(it.get_hardwareId());
                it = it.nextSoundLevel();
            }
            return res.ToArray();
        }

        protected override void functionArrival()
        {
            base.functionArrival();
            _label = _func.get_label();
        }

        protected override void moduleConfigHasChanged()
       	{
            base.moduleConfigHasChanged();
        }

        /**
         * <summary>
         *   Changes the measuring unit for the sound pressure level (dBA, dBC or dBZ).
         * <para>
         *   That unit will directly determine frequency weighting to be used to compute
         *   the measured value. Remember to call the <c>saveToFlash()</c> method of the
         *   module if the modification must be kept.
         * </para>
         * <para>
         * </para>
         * </summary>
         * <param name="newval">
         *   a string corresponding to the measuring unit for the sound pressure level (dBA, dBC or dBZ)
         * </param>
         * <para>
         * </para>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public int set_unit(string newval)
        {
            if (_func == null) {
                throw new YoctoApiProxyException("No SoundLevel connected");
            }
            if (newval == _Unit_INVALID) {
                return YAPI.SUCCESS;
            }
            return _func.set_unit(newval);
        }

        /**
         * <summary>
         *   Returns the label for the sound pressure level measurement, as per
         *   IEC standard 61672-1:2013.
         * <para>
         * </para>
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   a string corresponding to the label for the sound pressure level measurement, as per
         *   IEC standard 61672-1:2013
         * </returns>
         * <para>
         *   On failure, throws an exception or returns <c>YSoundLevel.LABEL_INVALID</c>.
         * </para>
         */
        public string get_label()
        {
            if (_func == null) {
                throw new YoctoApiProxyException("No SoundLevel connected");
            }
            return _func.get_label();
        }

        // property with cached value for instant access (constant value)
        /// <value>Label for the sound pressure level measurement, as per</value>
        public string Label
        {
            get
            {
                if (_func == null) {
                    return _Label_INVALID;
                }
                if (_online) {
                    return _label;
                }
                return _Label_INVALID;
            }
        }

        /**
         * <summary>
         *   Returns the integration time in milliseconds for measuring the sound pressure level.
         * <para>
         * </para>
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   an integer corresponding to the integration time in milliseconds for measuring the sound pressure level
         * </returns>
         * <para>
         *   On failure, throws an exception or returns <c>YSoundLevel.INTEGRATIONTIME_INVALID</c>.
         * </para>
         */
        public int get_integrationTime()
        {
            int res;
            if (_func == null) {
                throw new YoctoApiProxyException("No SoundLevel connected");
            }
            res = _func.get_integrationTime();
            if (res == YAPI.INVALID_INT) {
                res = _IntegrationTime_INVALID;
            }
            return res;
        }
    }
    //--- (end of YSoundLevel implementation)
}

