/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements YDisplayLayerProxy, the Proxy API for DisplayLayer
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Timers;
using System.Globalization;
using System.Text.RegularExpressions;
using YoctoLib;

namespace YoctoProxyAPI
{
    //--- (YDisplayLayer class start)
    public class YDisplayLayerProxy
    {
        private YDisplayLayer _objptr;
        internal YDisplayLayerProxy(YDisplayLayer objptr)
        {
             _objptr = objptr;
        }
        //--- (end of YDisplayLayer class start)
        //--- (YDisplayLayer definitions)
    public const int NO_INK = -1;
    public const int BG_INK = -2;
    public const int FG_INK = -3;
        public const int _ALIGN_INVALID = 0;
        public const int _ALIGN_TOP_LEFT = 1;
        public const int _ALIGN_CENTER_LEFT = 2;
        public const int _ALIGN_BASELINE_LEFT = 3;
        public const int _ALIGN_BOTTOM_LEFT = 4;
        public const int _ALIGN_TOP_CENTER = 5;
        public const int _ALIGN_CENTER = 6;
        public const int _ALIGN_BASELINE_CENTER = 7;
        public const int _ALIGN_BOTTOM_CENTER = 8;
        public const int _ALIGN_TOP_DECIMAL = 9;
        public const int _ALIGN_CENTER_DECIMAL = 10;
        public const int _ALIGN_BASELINE_DECIMAL = 11;
        public const int _ALIGN_BOTTOM_DECIMAL = 12;
        public const int _ALIGN_TOP_RIGHT = 13;
        public const int _ALIGN_CENTER_RIGHT = 14;
        public const int _ALIGN_BASELINE_RIGHT = 15;
        public const int _ALIGN_BOTTOM_RIGHT = 16;

        private int _ALIGN2Int(YDisplayLayer.ALIGN realenum)
        {
            switch (realenum) {
                default:
                case YDisplayLayer.ALIGN.TOP_LEFT:
                    return _ALIGN_TOP_LEFT;
                case YDisplayLayer.ALIGN.CENTER_LEFT:
                    return _ALIGN_CENTER_LEFT;
                case YDisplayLayer.ALIGN.BASELINE_LEFT:
                    return _ALIGN_BASELINE_LEFT;
                case YDisplayLayer.ALIGN.BOTTOM_LEFT:
                    return _ALIGN_BOTTOM_LEFT;
                case YDisplayLayer.ALIGN.TOP_CENTER:
                    return _ALIGN_TOP_CENTER;
                case YDisplayLayer.ALIGN.CENTER:
                    return _ALIGN_CENTER;
                case YDisplayLayer.ALIGN.BASELINE_CENTER:
                    return _ALIGN_BASELINE_CENTER;
                case YDisplayLayer.ALIGN.BOTTOM_CENTER:
                    return _ALIGN_BOTTOM_CENTER;
                case YDisplayLayer.ALIGN.TOP_DECIMAL:
                    return _ALIGN_TOP_DECIMAL;
                case YDisplayLayer.ALIGN.CENTER_DECIMAL:
                    return _ALIGN_CENTER_DECIMAL;
                case YDisplayLayer.ALIGN.BASELINE_DECIMAL:
                    return _ALIGN_BASELINE_DECIMAL;
                case YDisplayLayer.ALIGN.BOTTOM_DECIMAL:
                    return _ALIGN_BOTTOM_DECIMAL;
                case YDisplayLayer.ALIGN.TOP_RIGHT:
                    return _ALIGN_TOP_RIGHT;
                case YDisplayLayer.ALIGN.CENTER_RIGHT:
                    return _ALIGN_CENTER_RIGHT;
                case YDisplayLayer.ALIGN.BASELINE_RIGHT:
                    return _ALIGN_BASELINE_RIGHT;
                case YDisplayLayer.ALIGN.BOTTOM_RIGHT:
                    return _ALIGN_BOTTOM_RIGHT;
            }
        }

        private YDisplayLayer.ALIGN _Int2ALIGN(int value)
        {
            switch (value) {
                default:
                case _ALIGN_TOP_LEFT:
                    return YDisplayLayer.ALIGN.TOP_LEFT;
                case _ALIGN_CENTER_LEFT:
                    return YDisplayLayer.ALIGN.CENTER_LEFT;
                case _ALIGN_BASELINE_LEFT:
                    return YDisplayLayer.ALIGN.BASELINE_LEFT;
                case _ALIGN_BOTTOM_LEFT:
                    return YDisplayLayer.ALIGN.BOTTOM_LEFT;
                case _ALIGN_TOP_CENTER:
                    return YDisplayLayer.ALIGN.TOP_CENTER;
                case _ALIGN_CENTER:
                    return YDisplayLayer.ALIGN.CENTER;
                case _ALIGN_BASELINE_CENTER:
                    return YDisplayLayer.ALIGN.BASELINE_CENTER;
                case _ALIGN_BOTTOM_CENTER:
                    return YDisplayLayer.ALIGN.BOTTOM_CENTER;
                case _ALIGN_TOP_DECIMAL:
                    return YDisplayLayer.ALIGN.TOP_DECIMAL;
                case _ALIGN_CENTER_DECIMAL:
                    return YDisplayLayer.ALIGN.CENTER_DECIMAL;
                case _ALIGN_BASELINE_DECIMAL:
                    return YDisplayLayer.ALIGN.BASELINE_DECIMAL;
                case _ALIGN_BOTTOM_DECIMAL:
                    return YDisplayLayer.ALIGN.BOTTOM_DECIMAL;
                case _ALIGN_TOP_RIGHT:
                    return YDisplayLayer.ALIGN.TOP_RIGHT;
                case _ALIGN_CENTER_RIGHT:
                    return YDisplayLayer.ALIGN.CENTER_RIGHT;
                case _ALIGN_BASELINE_RIGHT:
                    return YDisplayLayer.ALIGN.BASELINE_RIGHT;
                case _ALIGN_BOTTOM_RIGHT:
                    return YDisplayLayer.ALIGN.BOTTOM_RIGHT;
            }
        }
        //--- (end of YDisplayLayer definitions)
        //--- (YDisplayLayer implementation)

        /**
         * <summary>
         *   Reverts the layer to its initial state (fully transparent, default settings).
         * <para>
         *   Reinitializes the drawing pointer to the upper left position,
         *   and selects the most visible pen color. If you only want to erase the layer
         *   content, use the method <c>clear()</c> instead.
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int reset()
        {
            return _objptr.reset();
        }

        /**
         * <summary>
         *   Erases the whole content of the layer (makes it fully transparent).
         * <para>
         *   This method does not change any other attribute of the layer.
         *   To reinitialize the layer attributes to defaults settings, use the method
         *   <c>reset()</c> instead.
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int clear()
        {
            return _objptr.clear();
        }

        /**
         * <summary>
         *   Selects the color to be used for all subsequent drawing functions,
         *   for filling as well as for line and text drawing.
         * <para>
         *   To select a different fill and outline color, use
         *   <c>selectFillColor</c> and <c>selectLineColor</c>.
         *   The pen color is provided as an RGB value.
         *   For grayscale or monochrome displays, the value is
         *   automatically converted to the proper range.
         * </para>
         * </summary>
         * <param name="color">
         *   the desired pen color, as a 24-bit RGB value
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectColorPen(int color)
        {
            return _objptr.selectColorPen(color);
        }

        /**
         * <summary>
         *   Selects the pen gray level for all subsequent drawing functions,
         *   for filling as well as for line and text drawing.
         * <para>
         *   To select a different fill and outline color, use
         *   <c>selectFillColor</c> and <c>selectLineColor</c>.
         *   The gray level is provided as a number between
         *   0 (black) and 255 (white, or whichever the lightest color is).
         *   For monochrome displays (without gray levels), any value
         *   lower than 128 is rendered as black, and any value equal
         *   or above to 128 is non-black.
         * </para>
         * </summary>
         * <param name="graylevel">
         *   the desired gray level, from 0 to 255
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectGrayPen(int graylevel)
        {
            return _objptr.selectGrayPen(graylevel);
        }

        /**
         * <summary>
         *   Selects an eraser instead of a pen for all subsequent drawing functions,
         *   except for bitmap copy functions.
         * <para>
         *   Any point drawn using the eraser
         *   becomes transparent (as when the layer is empty), showing the other
         *   layers beneath it.
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectEraser()
        {
            return _objptr.selectEraser();
        }

        /**
         * <summary>
         *   Selects the color to be used for filling rectangular bars,
         *   discs and polygons.
         * <para>
         *   The color is provided as an RGB value.
         *   For grayscale or monochrome displays, the value is
         *   automatically converted to the proper range.
         *   You can also use the constants <c>FG_INK</c> to use the
         *   default drawing colour, <c>BG_INK</c> to use the default
         *   background colour, and <c>NO_INK</c> to disable filling.
         * </para>
         * </summary>
         * <param name="color">
         *   the desired drawing color, as a 24-bit RGB value,
         *   or one of the constants <c>NO_INK</c>, <c>FG_INK</c>
         *   or <c>BG_INK</c>
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectFillColor(int color)
        {
            return _objptr.selectFillColor(color);
        }

        /**
         * <summary>
         *   Selects the color to be used for drawing the outline of rectangular
         *   bars, discs and polygons, as well as for drawing lines and text.
         * <para>
         *   The color is provided as an RGB value.
         *   For grayscale or monochrome displays, the value is
         *   automatically converted to the proper range.
         *   You can also use the constants <c>FG_INK</c> to use the
         *   default drawing colour, <c>BG_INK</c> to use the default
         *   background colour, and <c>NO_INK</c> to disable outline drawing.
         * </para>
         * </summary>
         * <param name="color">
         *   the desired drawing color, as a 24-bit RGB value,
         *   or one of the constants <c>NO_INK</c>, <c>FG_INK</c>
         *   or <c>BG_INK</c>
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectLineColor(int color)
        {
            return _objptr.selectLineColor(color);
        }

        /**
         * <summary>
         *   Selects the line width for drawing the outline of rectangular
         *   bars, discs and polygons, as well as for drawing lines.
         * <para>
         * </para>
         * </summary>
         * <param name="width">
         *   the desired line width, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectLineWidth(int width)
        {
            return _objptr.selectLineWidth(width);
        }

        /**
         * <summary>
         *   Draws a single pixel at the specified position.
         * <para>
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawPixel(int x, int y)
        {
            return _objptr.drawPixel(x, y);
        }

        /**
         * <summary>
         *   Draws an empty rectangle at a specified position.
         * <para>
         * </para>
         * </summary>
         * <param name="x1">
         *   the distance from left of layer to the left border of the rectangle, in pixels
         * </param>
         * <param name="y1">
         *   the distance from top of layer to the top border of the rectangle, in pixels
         * </param>
         * <param name="x2">
         *   the distance from left of layer to the right border of the rectangle, in pixels
         * </param>
         * <param name="y2">
         *   the distance from top of layer to the bottom border of the rectangle, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawRect(int x1, int y1, int x2, int y2)
        {
            return _objptr.drawRect(x1, y1, x2, y2);
        }

        /**
         * <summary>
         *   Draws a filled rectangular bar at a specified position.
         * <para>
         * </para>
         * </summary>
         * <param name="x1">
         *   the distance from left of layer to the left border of the rectangle, in pixels
         * </param>
         * <param name="y1">
         *   the distance from top of layer to the top border of the rectangle, in pixels
         * </param>
         * <param name="x2">
         *   the distance from left of layer to the right border of the rectangle, in pixels
         * </param>
         * <param name="y2">
         *   the distance from top of layer to the bottom border of the rectangle, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawBar(int x1, int y1, int x2, int y2)
        {
            return _objptr.drawBar(x1, y1, x2, y2);
        }

        /**
         * <summary>
         *   Draws an empty circle at a specified position.
         * <para>
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the center of the circle, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the center of the circle, in pixels
         * </param>
         * <param name="r">
         *   the radius of the circle, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawCircle(int x, int y, int r)
        {
            return _objptr.drawCircle(x, y, r);
        }

        /**
         * <summary>
         *   Draws a filled disc at a given position.
         * <para>
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the center of the disc, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the center of the disc, in pixels
         * </param>
         * <param name="r">
         *   the radius of the disc, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawDisc(int x, int y, int r)
        {
            return _objptr.drawDisc(x, y, r);
        }

        /**
         * <summary>
         *   Selects a font to use for the next text drawing functions, by providing the name of the
         *   font file.
         * <para>
         *   You can use a built-in font as well as a font file that you have previously
         *   uploaded to the device built-in memory. If you experience problems selecting a font
         *   file, check the device logs for any error message such as missing font file or bad font
         *   file format.
         * </para>
         * </summary>
         * <param name="fontname">
         *   the font file name, embedded fonts are 8x8.yfm, Small.yfm, Medium.yfm, Large.yfm (not available on
         *   Yocto-MiniDisplay).
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int selectFont(string fontname)
        {
            return _objptr.selectFont(fontname);
        }

        /**
         * <summary>
         *   Draws a text string at the specified position.
         * <para>
         *   The point of the text that is aligned
         *   to the specified pixel position is called the anchor point, and can be chosen among
         *   several options. Text is rendered from left to right, without implicit wrapping.
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the text anchor point, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the text anchor point, in pixels
         * </param>
         * <param name="anchor">
         *   the text anchor point, chosen among the <c>YDisplayLayer.ALIGN</c> enumeration:
         *   <c>YDisplayLayer.ALIGN_TOP_LEFT</c>,         <c>YDisplayLayer.ALIGN_CENTER_LEFT</c>,
         *   <c>YDisplayLayer.ALIGN_BASELINE_LEFT</c>,    <c>YDisplayLayer.ALIGN_BOTTOM_LEFT</c>,
         *   <c>YDisplayLayer.ALIGN_TOP_CENTER</c>,       <c>YDisplayLayer.ALIGN_CENTER</c>,
         *   <c>YDisplayLayer.ALIGN_BASELINE_CENTER</c>,  <c>YDisplayLayer.ALIGN_BOTTOM_CENTER</c>,
         *   <c>YDisplayLayer.ALIGN_TOP_DECIMAL</c>,      <c>YDisplayLayer.ALIGN_CENTER_DECIMAL</c>,
         *   <c>YDisplayLayer.ALIGN_BASELINE_DECIMAL</c>, <c>YDisplayLayer.ALIGN_BOTTOM_DECIMAL</c>,
         *   <c>YDisplayLayer.ALIGN_TOP_RIGHT</c>,        <c>YDisplayLayer.ALIGN_CENTER_RIGHT</c>,
         *   <c>YDisplayLayer.ALIGN_BASELINE_RIGHT</c>,   <c>YDisplayLayer.ALIGN_BOTTOM_RIGHT</c>.
         * </param>
         * <param name="text">
         *   the text string to draw
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawText(int x, int y, int anchor, string text)
        {
            return _objptr.drawText(x, y, _Int2ALIGN(anchor), text);
        }

        /**
         * <summary>
         *   Draws an image previously uploaded to the device filesystem, at the specified position.
         * <para>
         *   At present time, GIF images are the only supported image format. If you experience
         *   problems using an image file, check the device logs for any error message such as
         *   missing image file or bad image file format.
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the left of the image, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the top of the image, in pixels
         * </param>
         * <param name="imagename">
         *   the GIF file name
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawImage(int x, int y, string imagename)
        {
            return _objptr.drawImage(x, y, imagename);
        }

        /**
         * <summary>
         *   Draws a bitmap at the specified position.
         * <para>
         *   The bitmap is provided as a binary object,
         *   where each pixel maps to a bit, from left to right and from top to bottom.
         *   The most significant bit of each byte maps to the leftmost pixel, and the least
         *   significant bit maps to the rightmost pixel. Bits set to 1 are drawn using the
         *   layer selected pen color. Bits set to 0 are drawn using the specified background
         *   gray level, unless -1 is specified, in which case they are not drawn at all
         *   (as if transparent).
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the left of the bitmap, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the top of the bitmap, in pixels
         * </param>
         * <param name="w">
         *   the width of the bitmap, in pixels
         * </param>
         * <param name="bitmap">
         *   a binary object
         * </param>
         * <param name="bgcol">
         *   the background gray level to use for zero bits (0 = black,
         *   255 = white), or -1 to leave the pixels unchanged
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawBitmap(int x, int y, int w, byte[] bitmap, int bgcol)
        {
            return _objptr.drawBitmap(x, y, w, bitmap, bgcol);
        }

        /**
         * <summary>
         *   Draws a GIF image provided as a binary buffer at the specified position.
         * <para>
         *   If the image drawing must be included in an animation sequence, save it
         *   in the device filesystem first and use <c>drawImage</c> instead.
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the left of the image, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the top of the image, in pixels
         * </param>
         * <param name="gifimage">
         *   a binary object with the content of a GIF file
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int drawGIF(int x, int y, byte[] gifimage)
        {
            return _objptr.drawGIF(x, y, gifimage);
        }

        /**
         * <summary>
         *   Moves the drawing pointer of this layer to the specified position.
         * <para>
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int moveTo(int x, int y)
        {
            return _objptr.moveTo(x, y);
        }

        /**
         * <summary>
         *   Draws a line from current drawing pointer position to the specified position.
         * <para>
         *   The specified destination pixel is included in the line. The pointer position
         *   is then moved to the end point of the line.
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the end point of the line, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the end point of the line, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int lineTo(int x, int y)
        {
            return _objptr.lineTo(x, y);
        }

        /**
         * <summary>
         *   Starts drawing a polygon with the first corner at the specified position.
         * <para>
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int polygonStart(int x, int y)
        {
            return _objptr.polygonStart(x, y);
        }

        /**
         * <summary>
         *   Adds a point to the currently open polygon, previously opened using
         *   <c>polygonStart</c>.
         * <para>
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of layer to the new point, in pixels
         * </param>
         * <param name="y">
         *   the distance from top of layer to the new point, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int polygonAdd(int x, int y)
        {
            return _objptr.polygonAdd(x, y);
        }

        /**
         * <summary>
         *   Close the currently open polygon, fill its content the fill color currently
         *   selected for the layer, and draw its outline using the selected line color.
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int polygonEnd()
        {
            return _objptr.polygonEnd();
        }

        /**
         * <summary>
         *   Outputs a message in the console area, and advances the console pointer accordingly.
         * <para>
         *   The console pointer position is automatically moved to the beginning
         *   of the next line when a newline character is met, or when the right margin
         *   is hit. When the new text to display extends below the lower margin, the
         *   console area is automatically scrolled up.
         * </para>
         * </summary>
         * <param name="text">
         *   the message to display
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int consoleOut(string text)
        {
            return _objptr.consoleOut(text);
        }

        /**
         * <summary>
         *   Sets up display margins for the <c>consoleOut</c> function.
         * <para>
         * </para>
         * </summary>
         * <param name="x1">
         *   the distance from left of layer to the left margin, in pixels
         * </param>
         * <param name="y1">
         *   the distance from top of layer to the top margin, in pixels
         * </param>
         * <param name="x2">
         *   the distance from left of layer to the right margin, in pixels
         * </param>
         * <param name="y2">
         *   the distance from top of layer to the bottom margin, in pixels
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int setConsoleMargins(int x1, int y1, int x2, int y2)
        {
            return _objptr.setConsoleMargins(x1, y1, x2, y2);
        }

        /**
         * <summary>
         *   Sets up the background color used by the <c>clearConsole</c> function and by
         *   the console scrolling feature.
         * <para>
         * </para>
         * </summary>
         * <param name="bgcol">
         *   the background gray level to use when scrolling (0 = black,
         *   255 = white), or -1 for transparent
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int setConsoleBackground(int bgcol)
        {
            return _objptr.setConsoleBackground(bgcol);
        }

        /**
         * <summary>
         *   Sets up the wrapping behavior used by the <c>consoleOut</c> function.
         * <para>
         * </para>
         * </summary>
         * <param name="wordwrap">
         *   <c>true</c> to wrap only between words,
         *   <c>false</c> to wrap on the last column anyway.
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int setConsoleWordWrap(bool wordwrap)
        {
            return _objptr.setConsoleWordWrap(wordwrap);
        }

        /**
         * <summary>
         *   Blanks the console area within console margins, and resets the console pointer
         *   to the upper left corner of the console.
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int clearConsole()
        {
            return _objptr.clearConsole();
        }

        /**
         * <summary>
         *   Sets the position of the layer relative to the display upper left corner.
         * <para>
         *   When smooth scrolling is used, the display offset of the layer is
         *   automatically updated during the next milliseconds to animate the move of the layer.
         * </para>
         * </summary>
         * <param name="x">
         *   the distance from left of display to the upper left corner of the layer
         * </param>
         * <param name="y">
         *   the distance from top of display to the upper left corner of the layer
         * </param>
         * <param name="scrollTime">
         *   number of milliseconds to use for smooth scrolling, or
         *   0 if the scrolling should be immediate.
         * </param>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int setLayerPosition(int x, int y, int scrollTime)
        {
            return _objptr.setLayerPosition(x, y, scrollTime);
        }

        /**
         * <summary>
         *   Hides the layer.
         * <para>
         *   The state of the layer is preserved but the layer is not displayed
         *   on the screen until the next call to <c>unhide()</c>. Hiding the layer can positively
         *   affect the drawing speed, since it postpones the rendering until all operations are
         *   completed (double-buffering).
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int hide()
        {
            return _objptr.hide();
        }

        /**
         * <summary>
         *   Shows the layer.
         * <para>
         *   Shows the layer again after a hide command.
         * </para>
         * </summary>
         * <returns>
         *   <c>0</c> if the call succeeds.
         * </returns>
         * <para>
         *   On failure, throws an exception or returns a negative error code.
         * </para>
         */
        public virtual int unhide()
        {
            return _objptr.unhide();
        }

        /**
         * <summary>
         *   Gets parent YDisplay.
         * <para>
         *   Returns the parent YDisplay object of the current YDisplayLayer.
         * </para>
         * </summary>
         * <returns>
         *   an <c>YDisplay</c> object
         * </returns>
         */
        public virtual YDisplayProxy get_display()
        {
            return YDisplayProxy.FindDisplay(_objptr.get_display().get_serialNumber());
        }

        /**
         * <summary>
         *   Returns the display width, in pixels.
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   an integer corresponding to the display width, in pixels
         * </returns>
         * <para>
         *   On failure, throws an exception or returns displaylayer._Displaywidth_INVALID.
         * </para>
         */
        public virtual int get_displayWidth()
        {
            return _objptr.get_displayWidth();
        }

        /**
         * <summary>
         *   Returns the display height, in pixels.
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   an integer corresponding to the display height, in pixels
         * </returns>
         * <para>
         *   On failure, throws an exception or returns displaylayer._Displayheight_INVALID.
         * </para>
         */
        public virtual int get_displayHeight()
        {
            return _objptr.get_displayHeight();
        }

        /**
         * <summary>
         *   Returns the width of the layers to draw on, in pixels.
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   an integer corresponding to the width of the layers to draw on, in pixels
         * </returns>
         * <para>
         *   On failure, throws an exception or returns displaylayer._Layerwidth_INVALID.
         * </para>
         */
        public virtual int get_layerWidth()
        {
            return _objptr.get_layerWidth();
        }

        /**
         * <summary>
         *   Returns the height of the layers to draw on, in pixels.
         * <para>
         * </para>
         * </summary>
         * <returns>
         *   an integer corresponding to the height of the layers to draw on, in pixels
         * </returns>
         * <para>
         *   On failure, throws an exception or returns displaylayer._Layerheight_INVALID.
         * </para>
         */
        public virtual int get_layerHeight()
        {
            return _objptr.get_layerHeight();
        }
    }
    //--- (end of YDisplayLayer implementation)
}

