/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle ColorSensor functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YColorSensor.h"
#include "yocto_api.h"
#include "yocto_colorsensor.h"

using namespace std;

//--- (YColorSensor definitions)

static const char *enumWorkingMode[] = {
    "AUTO",
    "EXPERT",
    NULL
};

static const char *enumEstimationModel[] = {
    "REFLECTION",
    "EMISSION",
    NULL
};

static const char *enumSimpleColor[] = {
    "BROWN",
    "RED",
    "ORANGE",
    "YELLOW",
    "WHITE",
    "GRAY",
    "BLACK",
    "GREEN",
    "BLUE",
    "PURPLE",
    "PINK",
    NULL
};

//--- (end of YColorSensor definitions)


//--- (YColorSensor implementation)
bool   YColorSensorCmdLine::isModule(void)      { return false; }
string YColorSensorCmdLine::YFunctionName(void) { return "YColorSensor"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of estimationModel is int (EstimationModel)
// type of workingMode is int (WorkingMode)
// type of ledCurrent is int (UInt31)
// type of ledCalibration is int (UInt31)
// type of integrationTime is int (UInt31)
// type of gain is int (UInt31)
// type of saturation is int (SaturationBits)
// type of estimatedRGB is int (U24Color)
// type of estimatedHSL is int (U24Color)
// type of estimatedXYZ is string (ColorCoord)
// type of estimatedOkLab is string (ColorCoord)
// type of nearRAL1 is string (Text)
// type of nearRAL2 is string (Text)
// type of nearRAL3 is string (Text)
// type of nearHTMLColor is string (Text)
// type of nearSimpleColorIndex is int (SimpleColor)
// type of nearSimpleColor is string (Text)
/**
 * Returns the logical name of the color sensor.
 *
 * @return a string corresponding to the logical name of the color sensor
 *
 * On failure, throws an exception or returns YColorSensor.LOGICALNAME_INVALID.
 */
class ColorSensor_get_logicalName : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the color sensor.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the color sensor. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the color sensor
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the color sensor.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the color sensor", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the color sensor.
 *
 * @return a string corresponding to a short string representing the current state of the color sensor
 *
 * On failure, throws an exception or returns YColorSensor.ADVERTISEDVALUE_INVALID.
 */
class ColorSensor_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the color sensor.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the predictive model used for color estimation (reflective or emissive).
 *
 * @return either YColorSensor.ESTIMATIONMODEL_REFLECTION or YColorSensor.ESTIMATIONMODEL_EMISSION,
 * according to the predictive model used for color estimation (reflective or emissive)
 *
 * On failure, throws an exception or returns YColorSensor.ESTIMATIONMODEL_INVALID.
 */
class ColorSensor_get_estimationModel : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_estimationModel(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimationModel";
  }

  string getDescription()
  {
    return "Returns the predictive model used for color estimation (reflective or emissive).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_estimationModel();
        string strval =  enumString(attrval,enumEstimationModel);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the predictive model to be used for color estimation (reflective or emissive).
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : either YColorSensor.ESTIMATIONMODEL_REFLECTION or
 * YColorSensor.ESTIMATIONMODEL_EMISSION, according to the predictive model to be used for color
 * estimation (reflective or emissive)
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_estimationModel : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_estimationModel(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_estimationModel";
  }

  string getDescription()
  {
    return "Changes the predictive model to be used for color estimation (reflective or emissive).";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either REFLECTION or EMISSION, according to the predictive model to be used for color estimation (reflective or emissive)", "REFLECTION=0,EMISSION=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_estimationModel(Y_ESTIMATIONMODEL_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the sensor working mode.
 * In Auto mode, sensor parameters are automatically set based on the selected estimation model.
 * In Expert mode, sensor parameters such as gain and integration time are configured manually.
 *
 * @return either YColorSensor.WORKINGMODE_AUTO or YColorSensor.WORKINGMODE_EXPERT, according to the
 * sensor working mode
 *
 * On failure, throws an exception or returns YColorSensor.WORKINGMODE_INVALID.
 */
class ColorSensor_get_workingMode : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_workingMode(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_workingMode";
  }

  string getDescription()
  {
    return "Returns the sensor working mode.";
  }

  string getMoreInfo()
  {
    return "In Auto mode, sensor parameters are automatically set based on the selected estimation model. In Expert mode, sensor parameters such as gain and integration time are configured manually.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_workingMode();
        string strval =  enumString(attrval,enumWorkingMode);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the sensor working mode.
 * In Auto mode, sensor parameters are automatically set based on the selected estimation model.
 * In Expert mode, sensor parameters such as gain and integration time are configured manually.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : either YColorSensor.WORKINGMODE_AUTO or YColorSensor.WORKINGMODE_EXPERT, according
 * to the sensor working mode
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_workingMode : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_workingMode(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_workingMode";
  }

  string getDescription()
  {
    return "Changes the sensor working mode.";
  }

  string getMoreInfo()
  {
    return "In Auto mode, sensor parameters are automatically set based on the selected estimation model. In Expert mode, sensor parameters such as gain and integration time are configured manually. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either AUTO or EXPERT, according to the sensor working mode", "AUTO=0,EXPERT=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_workingMode(Y_WORKINGMODE_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the amount of current sent to the illumination LEDs, for reflection measures.
 * The value is an integer ranging from 0 (LEDs off) to 254 (LEDs at maximum intensity).
 *
 * @return an integer corresponding to the amount of current sent to the illumination LEDs, for reflection measures
 *
 * On failure, throws an exception or returns YColorSensor.LEDCURRENT_INVALID.
 */
class ColorSensor_get_ledCurrent : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_ledCurrent(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_ledCurrent";
  }

  string getDescription()
  {
    return "Returns the amount of current sent to the illumination LEDs, for reflection measures.";
  }

  string getMoreInfo()
  {
    return "The value is an integer ranging from 0 (LEDs off) to 254 (LEDs at maximum intensity).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_ledCurrent();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the amount of current sent to the illumination LEDs, for reflection measures.
 * The value is an integer ranging from 0 (LEDs off) to 254 (LEDs at maximum intensity).
 *
 * @param newval : an integer corresponding to the amount of current sent to the illumination LEDs,
 * for reflection measures
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_ledCurrent : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_ledCurrent(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_ledCurrent";
  }

  string getDescription()
  {
    return "Changes the amount of current sent to the illumination LEDs, for reflection measures.";
  }

  string getMoreInfo()
  {
    return "The value is an integer ranging from 0 (LEDs off) to 254 (LEDs at maximum intensity).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the amount of current sent to the illumination LEDs, for reflection measures", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_ledCurrent(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the current sent to the illumination LEDs during the latest calibration.
 *
 * @return an integer corresponding to the current sent to the illumination LEDs during the latest calibration
 *
 * On failure, throws an exception or returns YColorSensor.LEDCALIBRATION_INVALID.
 */
class ColorSensor_get_ledCalibration : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_ledCalibration(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_ledCalibration";
  }

  string getDescription()
  {
    return "Returns the current sent to the illumination LEDs during the latest calibration.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_ledCalibration();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Remember the LED current sent to the illumination LEDs during a calibration.
 * Thanks to this, the device is able to use the same current when taking measures.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : an integer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_ledCalibration : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_ledCalibration(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_ledCalibration";
  }

  string getDescription()
  {
    return "Remember the LED current sent to the illumination LEDs during a calibration.";
  }

  string getMoreInfo()
  {
    return "Thanks to this, the device is able to use the same current when taking measures. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_ledCalibration(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the current integration time for spectral measure, in milliseconds.
 * A longer integration time increase the sensitivity for low light conditions,
 * but reduces the measure taking rate and may lead to saturation for lighter colors.
 *
 * @return an integer corresponding to the current integration time for spectral measure, in milliseconds
 *
 * On failure, throws an exception or returns YColorSensor.INTEGRATIONTIME_INVALID.
 */
class ColorSensor_get_integrationTime : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_integrationTime(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_integrationTime";
  }

  string getDescription()
  {
    return "Returns the current integration time for spectral measure, in milliseconds.";
  }

  string getMoreInfo()
  {
    return "A longer integration time increase the sensitivity for low light conditions, but reduces the measure taking rate and may lead to saturation for lighter colors.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_integrationTime();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the integration time for spectral measure, in milliseconds.
 * A longer integration time increase the sensitivity for low light conditions,
 * but reduces the measure taking rate and may lead to saturation for lighter colors.
 * This method can only be used when the sensor is configured in expert mode;
 * when running in auto mode, the change is ignored.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the integration time for spectral measure, in milliseconds
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_integrationTime : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_integrationTime(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_integrationTime";
  }

  string getDescription()
  {
    return "Changes the integration time for spectral measure, in milliseconds.";
  }

  string getMoreInfo()
  {
    return "A longer integration time increase the sensitivity for low light conditions, but reduces the measure taking rate and may lead to saturation for lighter colors. This method can only be used when the sensor is configured in expert mode; when running in auto mode, the change is ignored. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the integration time for spectral measure, in milliseconds", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_integrationTime(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the current spectral channel detector gain exponent.
 * For a value n ranging from 0 to 12, the applied gain is 2^(n-1).
 * 0 corresponds to a gain of 0.5, and 12 corresponds to a gain of 2048.
 *
 * @return an integer corresponding to the current spectral channel detector gain exponent
 *
 * On failure, throws an exception or returns YColorSensor.GAIN_INVALID.
 */
class ColorSensor_get_gain : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_gain(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_gain";
  }

  string getDescription()
  {
    return "Returns the current spectral channel detector gain exponent.";
  }

  string getMoreInfo()
  {
    return "For a value n ranging from 0 to 12, the applied gain is 2^(n-1). 0 corresponds to a gain of 0.5, and 12 corresponds to a gain of 2048.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_gain();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the spectral channel detector gain exponent.
 * For a value n ranging from 0 to 12, the applied gain is 2^(n-1).
 * 0 corresponds to a gain of 0.5, and 12 corresponds to a gain of 2048.
 * This method can only be used when the sensor is configured in expert mode;
 * when running in auto mode, the change is ignored.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the spectral channel detector gain exponent
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class ColorSensor_set_gain : public YapiCommand /* arguments: newval */
{
public:
  ColorSensor_set_gain(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_gain";
  }

  string getDescription()
  {
    return "Changes the spectral channel detector gain exponent.";
  }

  string getMoreInfo()
  {
    return "For a value n ranging from 0 to 12, the applied gain is 2^(n-1). 0 corresponds to a gain of 0.5, and 12 corresponds to a gain of 2048. This method can only be used when the sensor is configured in expert mode; when running in auto mode, the change is ignored. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the spectral channel detector gain exponent", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_gain(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the current saturation state of the sensor, as an integer.
 * Bit 0 indicates saturation of the analog sensor, which can only
 * be corrected by reducing the gain parameters or the luminosity.
 * Bit 1 indicates saturation of the digital interface, which can
 * be corrected by reducing the integration time or the gain.
 *
 * @return an integer corresponding to the current saturation state of the sensor, as an integer
 *
 * On failure, throws an exception or returns YColorSensor.SATURATION_INVALID.
 */
class ColorSensor_get_saturation : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_saturation(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_saturation";
  }

  string getDescription()
  {
    return "Returns the current saturation state of the sensor, as an integer.";
  }

  string getMoreInfo()
  {
    return "Bit 0 indicates saturation of the analog sensor, which can only be corrected by reducing the gain parameters or the luminosity. Bit 1 indicates saturation of the digital interface, which can be corrected by reducing the integration time or the gain.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_saturation();
        string strval =  bitString("AD", attrval);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color in RGB color model (0xRRGGBB).
 * The RGB color model describes each color using a combination of 3 components:
 * - Red (R): the intensity of red, in the 0...255 range
 * - Green (G): the intensity of green, in the 0...255 range
 * - Blue (B): the intensity of blue, in the 0...255 range
 *
 * @return an integer corresponding to the estimated color in RGB color model (0xRRGGBB)
 *
 * On failure, throws an exception or returns YColorSensor.ESTIMATEDRGB_INVALID.
 */
class ColorSensor_get_estimatedRGB : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_estimatedRGB(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedRGB";
  }

  string getDescription()
  {
    return "Returns the estimated color in RGB color model (0xRRGGBB).";
  }

  string getMoreInfo()
  {
    return "The RGB color model describes each color using a combination of 3 components:\n - Red (R): the intensity of red, in the 0...255 range\n - Green (G): the intensity of green, in the 0...255 range\n - Blue (B): the intensity of blue, in the 0...255 range";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_estimatedRGB();
        char buf[32]; SAFE_SPRINTF(buf,32,"#%06x",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color in HSL color model (0xHHSSLL).
 * The HSL color model describes each color using a combination of 3 components:
 * - Hue (H): the angle on the color wheel (0-360 degrees), mapped to 0...255
 * - Saturation (S): the intensity of the color (0-100%), mapped to 0...255
 * - Lightness (L): the brightness of the color (0-100%), mapped to 0...255
 *
 * @return an integer corresponding to the estimated color in HSL color model (0xHHSSLL)
 *
 * On failure, throws an exception or returns YColorSensor.ESTIMATEDHSL_INVALID.
 */
class ColorSensor_get_estimatedHSL : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_estimatedHSL(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedHSL";
  }

  string getDescription()
  {
    return "Returns the estimated color in HSL color model (0xHHSSLL).";
  }

  string getMoreInfo()
  {
    return "The HSL color model describes each color using a combination of 3 components:\n - Hue (H): the angle on the color wheel (0-360 degrees), mapped to 0...255\n - Saturation (S): the intensity of the color (0-100%), mapped to 0...255\n - Lightness (L): the brightness of the color (0-100%), mapped to 0...255";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_estimatedHSL();
        char buf[32]; SAFE_SPRINTF(buf,32,"#%06x",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color according to the CIE XYZ color model.
 * This color model is based on human vision and light perception, with three components
 * represented by real numbers between 0 and 1:
 * - X: corresponds to a component mixing sensitivity to red and green
 * - Y: represents luminance (perceived brightness)
 * - Z: corresponds to sensitivity to blue
 *
 * @return a string corresponding to the estimated color according to the CIE XYZ color model
 *
 * On failure, throws an exception or returns YColorSensor.ESTIMATEDXYZ_INVALID.
 */
class ColorSensor_get_estimatedXYZ : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_estimatedXYZ(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedXYZ";
  }

  string getDescription()
  {
    return "Returns the estimated color according to the CIE XYZ color model.";
  }

  string getMoreInfo()
  {
    return "This color model is based on human vision and light perception, with three components represented by real numbers between 0 and 1:\n - X: corresponds to a component mixing sensitivity to red and green\n - Y: represents luminance (perceived brightness)\n - Z: corresponds to sensitivity to blue";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_estimatedXYZ();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color according to the OkLab color model.
 * OkLab is a perceptual color model that aims to align human color perception with numerical
 * values, so that colors that are visually near are also numerically near. Colors are represented
 * using three components:
 * - L: lightness, a real number between 0 and 1
 * - a: color variations between green and red, between -0.5 and 0.5
 * - b: color variations between blue and yellow, between -0.5 and 0.5.
 *
 * @return a string corresponding to the estimated color according to the OkLab color model
 *
 * On failure, throws an exception or returns YColorSensor.ESTIMATEDOKLAB_INVALID.
 */
class ColorSensor_get_estimatedOkLab : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_estimatedOkLab(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedOkLab";
  }

  string getDescription()
  {
    return "Returns the estimated color according to the OkLab color model.";
  }

  string getMoreInfo()
  {
    return "OkLab is a perceptual color model that aims to align human color perception with numerical values, so that colors that are visually near are also numerically near. Colors are represented using three components:\n - L: lightness, a real number between 0 and 1\n - a: color variations between green and red, between -0.5 and 0.5\n - b: color variations between blue and yellow, between -0.5 and 0.5.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_estimatedOkLab();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the RAL Classic color closest to the estimated color, with a similarity ratio.
 *
 * @return a string corresponding to the RAL Classic color closest to the estimated color, with a similarity ratio
 *
 * On failure, throws an exception or returns YColorSensor.NEARRAL1_INVALID.
 */
class ColorSensor_get_nearRAL1 : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_nearRAL1(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearRAL1";
  }

  string getDescription()
  {
    return "Returns the RAL Classic color closest to the estimated color, with a similarity ratio.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_nearRAL1();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the second closest RAL Classic color to the estimated color, with a similarity ratio.
 *
 * @return a string corresponding to the second closest RAL Classic color to the estimated color, with
 * a similarity ratio
 *
 * On failure, throws an exception or returns YColorSensor.NEARRAL2_INVALID.
 */
class ColorSensor_get_nearRAL2 : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_nearRAL2(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearRAL2";
  }

  string getDescription()
  {
    return "Returns the second closest RAL Classic color to the estimated color, with a similarity ratio.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_nearRAL2();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the third closest RAL Classic color to the estimated color, with a similarity ratio.
 *
 * @return a string corresponding to the third closest RAL Classic color to the estimated color, with
 * a similarity ratio
 *
 * On failure, throws an exception or returns YColorSensor.NEARRAL3_INVALID.
 */
class ColorSensor_get_nearRAL3 : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_nearRAL3(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearRAL3";
  }

  string getDescription()
  {
    return "Returns the third closest RAL Classic color to the estimated color, with a similarity ratio.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_nearRAL3();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of the HTML color closest to the estimated color.
 *
 * @return a string corresponding to the name of the HTML color closest to the estimated color
 *
 * On failure, throws an exception or returns YColorSensor.NEARHTMLCOLOR_INVALID.
 */
class ColorSensor_get_nearHTMLColor : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_nearHTMLColor(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearHTMLColor";
  }

  string getDescription()
  {
    return "Returns the name of the HTML color closest to the estimated color.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_nearHTMLColor();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the index of the basic color typically used to refer to the estimated color (enumerated value).
 * The list of basic colors recognized is:
 * - 0 - Brown
 * - 1 - Red
 * - 2 - Orange
 * - 3 - Yellow
 * - 4 - White
 * - 5 - Gray
 * - 6 - Black
 * - 7 - Green
 * - 8 - Blue
 * - 9 - Purple
 * - 10 - Pink
 *
 * @return a value among YColorSensor.NEARSIMPLECOLORINDEX_BROWN,
 * YColorSensor.NEARSIMPLECOLORINDEX_RED, YColorSensor.NEARSIMPLECOLORINDEX_ORANGE,
 * YColorSensor.NEARSIMPLECOLORINDEX_YELLOW, YColorSensor.NEARSIMPLECOLORINDEX_WHITE,
 * YColorSensor.NEARSIMPLECOLORINDEX_GRAY, YColorSensor.NEARSIMPLECOLORINDEX_BLACK,
 * YColorSensor.NEARSIMPLECOLORINDEX_GREEN, YColorSensor.NEARSIMPLECOLORINDEX_BLUE,
 * YColorSensor.NEARSIMPLECOLORINDEX_PURPLE and YColorSensor.NEARSIMPLECOLORINDEX_PINK corresponding
 * to the index of the basic color typically used to refer to the estimated color (enumerated value)
 *
 * On failure, throws an exception or returns YColorSensor.NEARSIMPLECOLORINDEX_INVALID.
 */
class ColorSensor_get_nearSimpleColorIndex : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_nearSimpleColorIndex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearSimpleColorIndex";
  }

  string getDescription()
  {
    return "Returns the index of the basic color typically used to refer to the estimated color (enumerated value).";
  }

  string getMoreInfo()
  {
    return "The list of basic colors recognized is:\n - 0 - Brown\n - 1 - Red\n - 2 - Orange\n - 3 - Yellow\n - 4 - White\n - 5 - Gray\n - 6 - Black\n - 7 - Green\n - 8 - Blue\n - 9 - Purple\n - 10 - Pink";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_nearSimpleColorIndex();
        string strval =  enumString(attrval,enumSimpleColor);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of the basic color typically used to refer to the estimated color.
 *
 * @return a string corresponding to the name of the basic color typically used to refer to the estimated color
 *
 * On failure, throws an exception or returns YColorSensor.NEARSIMPLECOLOR_INVALID.
 */
class ColorSensor_get_nearSimpleColor : public YapiCommand /* arguments: */
{
public:
  ColorSensor_get_nearSimpleColor(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearSimpleColor";
  }

  string getDescription()
  {
    return "Returns the name of the basic color typically used to refer to the estimated color.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_nearSimpleColor();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

// type of valueCallbackColorSensor is YColorSensorValueCallback* (YColorSensorValueCallback)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_ColorSensor_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_ColorSensor_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_ColorSensor_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_ColorSensor_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_ColorSensor_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_ColorSensor_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_ColorSensor_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_ColorSensor_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Turns on the built-in illumination LEDs using the same current as used during the latest calibration.
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_ColorSensor_turnLedOn : public YapiCommand /* arguments: */
{
public:
  apifun_ColorSensor_turnLedOn(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "turnLedOn";
  }

  string getDescription()
  {
    return "Turns on the built-in illumination LEDs using the same current as used during the latest calibration.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->turnLedOn();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Turns off the built-in illumination LEDs.
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_ColorSensor_turnLedOff : public YapiCommand /* arguments: */
{
public:
  apifun_ColorSensor_turnLedOff(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "turnLedOff";
  }

  string getDescription()
  {
    return "Turns off the built-in illumination LEDs.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YColorSensor*>* list = enumerateTargets<YColorSensor>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->turnLedOff();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YColorSensor implementation)

//--- (YColorSensor functions)
void YColorSensorCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YColorSensor>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_estimationModel(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_estimationModel(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_workingMode(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_workingMode(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_ledCurrent(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_ledCurrent(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_ledCalibration(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_ledCalibration(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_integrationTime(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_integrationTime(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_gain(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_set_gain(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_saturation(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_estimatedRGB(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_estimatedHSL(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_estimatedXYZ(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_estimatedOkLab(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_nearRAL1(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_nearRAL2(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_nearRAL3(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_nearHTMLColor(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_nearSimpleColorIndex(this)));
    cmdList->push_back((YapiCommand*) (new ColorSensor_get_nearSimpleColor(this)));
    cmdList->push_back((YapiCommand*) (new apifun_ColorSensor_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_ColorSensor_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_ColorSensor_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_ColorSensor_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_ColorSensor_turnLedOn(this)));
    cmdList->push_back((YapiCommand*) (new apifun_ColorSensor_turnLedOff(this)));
  }

//--- (end of YColorSensor functions)
