/*********************************************************************
 *
 *  $Id: YBluetoothLink.cpp 52599 2023-01-05 08:33:38Z seb $
 *
 *  Implements commands to handle BluetoothLink functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YBluetoothLink.h"
#include "yocto_api.h"
#include "yocto_bluetoothlink.h"

using namespace std;

//--- (YBluetoothLink definitions)

static const char *enumBool[] = {
    "FALSE",
    "TRUE",
    NULL
};

static const char *enumBtState[] = {
    "DOWN",
    "FREE",
    "SEARCH",
    "EXISTS",
    "LINKED",
    "PLAY",
    NULL
};

//--- (end of YBluetoothLink definitions)


//--- (YBluetoothLink implementation)
bool   YBluetoothLinkCmdLine::isModule(void)      { return false; }
string YBluetoothLinkCmdLine::YFunctionName(void) { return "YBluetoothLink"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of ownAddress is string (MACAddress)
// type of pairingPin is string (Text)
// type of remoteAddress is string (MACAddress)
// type of remoteName is string (Text)
// type of mute is int (Bool)
// type of preAmplifier is int (Percent)
// type of volume is int (Percent)
// type of linkState is int (BtState)
// type of linkQuality is int (Percent)
// type of command is string (Text)
/**
 * Returns the logical name of the Bluetooth sound controller.
 *
 * @return a string corresponding to the logical name of the Bluetooth sound controller
 *
 * On failure, throws an exception or returns YBluetoothLink.LOGICALNAME_INVALID.
 */
class BluetoothLink_get_logicalName : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the Bluetooth sound controller.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the Bluetooth sound controller. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the Bluetooth sound controller
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class BluetoothLink_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  BluetoothLink_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the Bluetooth sound controller.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the Bluetooth sound controller", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the Bluetooth sound controller.
 *
 * @return a string corresponding to a short string representing the current state of the Bluetooth
 * sound controller
 *
 * On failure, throws an exception or returns YBluetoothLink.ADVERTISEDVALUE_INVALID.
 */
class BluetoothLink_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the Bluetooth sound controller.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the MAC-48 address of the bluetooth interface, which is unique on the bluetooth network.
 *
 * @return a string corresponding to the MAC-48 address of the bluetooth interface, which is unique on
 * the bluetooth network
 *
 * On failure, throws an exception or returns YBluetoothLink.OWNADDRESS_INVALID.
 */
class BluetoothLink_get_ownAddress : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_ownAddress(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_ownAddress";
  }

  string getDescription()
  {
    return "Returns the MAC-48 address of the bluetooth interface, which is unique on the bluetooth network.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_ownAddress();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns an opaque string if a PIN code has been configured in the device to access
 * the SIM card, or an empty string if none has been configured or if the code provided
 * was rejected by the SIM card.
 *
 * @return a string corresponding to an opaque string if a PIN code has been configured in the device to access
 *         the SIM card, or an empty string if none has been configured or if the code provided
 *         was rejected by the SIM card
 *
 * On failure, throws an exception or returns YBluetoothLink.PAIRINGPIN_INVALID.
 */
class BluetoothLink_get_pairingPin : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_pairingPin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_pairingPin";
  }

  string getDescription()
  {
    return "Returns an opaque string if a PIN code has been configured in the device to access the SIM card, or an empty string if none has been configured or if the code providedwas rejected by the SIM card.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_pairingPin();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the PIN code used by the module for bluetooth pairing.
 * Remember to call the saveToFlash() method of the module to save the
 * new value in the device flash.
 *
 * @param newval : a string corresponding to the PIN code used by the module for bluetooth pairing
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class BluetoothLink_set_pairingPin : public YapiCommand /* arguments: newval */
{
public:
  BluetoothLink_set_pairingPin(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_pairingPin";
  }

  string getDescription()
  {
    return "Changes the PIN code used by the module for bluetooth pairing.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" to save the new value in the device flash.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the PIN code used by the module for bluetooth pairing", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_pairingPin(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the MAC-48 address of the remote device to connect to.
 *
 * @return a string corresponding to the MAC-48 address of the remote device to connect to
 *
 * On failure, throws an exception or returns YBluetoothLink.REMOTEADDRESS_INVALID.
 */
class BluetoothLink_get_remoteAddress : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_remoteAddress(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_remoteAddress";
  }

  string getDescription()
  {
    return "Returns the MAC-48 address of the remote device to connect to.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_remoteAddress();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the MAC-48 address defining which remote device to connect to.
 * Remember to call the saveToFlash()
 * method of the module if the modification must be kept.
 *
 * @param newval : a string corresponding to the MAC-48 address defining which remote device to connect to
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class BluetoothLink_set_remoteAddress : public YapiCommand /* arguments: newval */
{
public:
  BluetoothLink_set_remoteAddress(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_remoteAddress";
  }

  string getDescription()
  {
    return "Changes the MAC-48 address defining which remote device to connect to.";
  }

  string getMoreInfo()
  {
    return "Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the MAC-48 address defining which remote device to connect to", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_remoteAddress(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the bluetooth name the remote device, if found on the bluetooth network.
 *
 * @return a string corresponding to the bluetooth name the remote device, if found on the bluetooth network
 *
 * On failure, throws an exception or returns YBluetoothLink.REMOTENAME_INVALID.
 */
class BluetoothLink_get_remoteName : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_remoteName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_remoteName";
  }

  string getDescription()
  {
    return "Returns the bluetooth name the remote device, if found on the bluetooth network.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_remoteName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of the mute function.
 *
 * @return either YBluetoothLink.MUTE_FALSE or YBluetoothLink.MUTE_TRUE, according to the state of the
 * mute function
 *
 * On failure, throws an exception or returns YBluetoothLink.MUTE_INVALID.
 */
class BluetoothLink_get_mute : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_mute(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_mute";
  }

  string getDescription()
  {
    return "Returns the state of the mute function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_mute();
        string strval =  enumString(attrval,enumBool);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the state of the mute function. Remember to call the matching module
 * saveToFlash() method to save the setting permanently.
 *
 * @param newval : either YBluetoothLink.MUTE_FALSE or YBluetoothLink.MUTE_TRUE, according to the
 * state of the mute function
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class BluetoothLink_set_mute : public YapiCommand /* arguments: newval */
{
public:
  BluetoothLink_set_mute(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_mute";
  }

  string getDescription()
  {
    return "Changes the state of the mute function.";
  }

  string getMoreInfo()
  {
    return "Remember to call the matching module saveToFlash() method to save the setting permanently.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either FALSE or TRUE, according to the state of the mute function", "FALSE=0,TRUE=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_mute(Y_MUTE_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the audio pre-amplifier volume, in per cents.
 *
 * @return an integer corresponding to the audio pre-amplifier volume, in per cents
 *
 * On failure, throws an exception or returns YBluetoothLink.PREAMPLIFIER_INVALID.
 */
class BluetoothLink_get_preAmplifier : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_preAmplifier(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_preAmplifier";
  }

  string getDescription()
  {
    return "Returns the audio pre-amplifier volume, in per cents.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_preAmplifier();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the audio pre-amplifier volume, in per cents.
 * Remember to call the saveToFlash()
 * method of the module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the audio pre-amplifier volume, in per cents
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class BluetoothLink_set_preAmplifier : public YapiCommand /* arguments: newval */
{
public:
  BluetoothLink_set_preAmplifier(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_preAmplifier";
  }

  string getDescription()
  {
    return "Changes the audio pre-amplifier volume, in per cents.";
  }

  string getMoreInfo()
  {
    return "Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the audio pre-amplifier volume, in per cents", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_preAmplifier(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the connected headset volume, in per cents.
 *
 * @return an integer corresponding to the connected headset volume, in per cents
 *
 * On failure, throws an exception or returns YBluetoothLink.VOLUME_INVALID.
 */
class BluetoothLink_get_volume : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_volume(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_volume";
  }

  string getDescription()
  {
    return "Returns the connected headset volume, in per cents.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_volume();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the connected headset volume, in per cents.
 *
 * @param newval : an integer corresponding to the connected headset volume, in per cents
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class BluetoothLink_set_volume : public YapiCommand /* arguments: newval */
{
public:
  BluetoothLink_set_volume(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_volume";
  }

  string getDescription()
  {
    return "Changes the connected headset volume, in per cents.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the connected headset volume, in per cents", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_volume(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the bluetooth link state.
 *
 * @return a value among YBluetoothLink.LINKSTATE_DOWN, YBluetoothLink.LINKSTATE_FREE,
 * YBluetoothLink.LINKSTATE_SEARCH, YBluetoothLink.LINKSTATE_EXISTS, YBluetoothLink.LINKSTATE_LINKED
 * and YBluetoothLink.LINKSTATE_PLAY corresponding to the bluetooth link state
 *
 * On failure, throws an exception or returns YBluetoothLink.LINKSTATE_INVALID.
 */
class BluetoothLink_get_linkState : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_linkState(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_linkState";
  }

  string getDescription()
  {
    return "Returns the bluetooth link state.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_linkState();
        string strval =  enumString(attrval,enumBtState);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the bluetooth receiver signal strength, in pourcents, or 0 if no connection is established.
 *
 * @return an integer corresponding to the bluetooth receiver signal strength, in pourcents, or 0 if
 * no connection is established
 *
 * On failure, throws an exception or returns YBluetoothLink.LINKQUALITY_INVALID.
 */
class BluetoothLink_get_linkQuality : public YapiCommand /* arguments: */
{
public:
  BluetoothLink_get_linkQuality(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_linkQuality";
  }

  string getDescription()
  {
    return "Returns the bluetooth receiver signal strength, in pourcents, or 0 if no connection is established.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_linkQuality();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

// type of valueCallbackBluetoothLink is YBluetoothLinkValueCallback* (YBluetoothLinkValueCallback)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_BluetoothLink_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_BluetoothLink_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_BluetoothLink_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_BluetoothLink_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Test if the function is readOnly. Return true if the function is write protected
 * or that the function is not available.
 *
 * @return true if the function is readOnly or not online.
 */
class apifun_BluetoothLink_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_BluetoothLink_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Test if the function is readOnly.";
  }

  string getMoreInfo()
  {
    return "Return true if the function is write protected or that the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_BluetoothLink_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_BluetoothLink_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Attempt to connect to the previously selected remote device.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_BluetoothLink_connect : public YapiCommand /* arguments: */
{
public:
  apifun_BluetoothLink_connect(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "connect";
  }

  string getDescription()
  {
    return "Attempt to connect to the previously selected remote device.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->connect();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Disconnect from the previously selected remote device.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_BluetoothLink_disconnect : public YapiCommand /* arguments: */
{
public:
  apifun_BluetoothLink_disconnect(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "disconnect";
  }

  string getDescription()
  {
    return "Disconnect from the previously selected remote device.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YBluetoothLink*>* list = enumerateTargets<YBluetoothLink>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->disconnect();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YBluetoothLink implementation)

//--- (YBluetoothLink functions)
void YBluetoothLinkCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YBluetoothLink>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_ownAddress(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_pairingPin(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_set_pairingPin(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_remoteAddress(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_set_remoteAddress(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_remoteName(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_mute(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_set_mute(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_preAmplifier(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_set_preAmplifier(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_volume(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_set_volume(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_linkState(this)));
    cmdList->push_back((YapiCommand*) (new BluetoothLink_get_linkQuality(this)));
    cmdList->push_back((YapiCommand*) (new apifun_BluetoothLink_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_BluetoothLink_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_BluetoothLink_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_BluetoothLink_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_BluetoothLink_connect(this)));
    cmdList->push_back((YapiCommand*) (new apifun_BluetoothLink_disconnect(this)));
  }

//--- (end of YBluetoothLink functions)
