/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle SoundSpectrum functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YSoundSpectrum.h"
#include "yocto_api.h"
#include "yocto_soundspectrum.h"

using namespace std;

//--- (YSoundSpectrum definitions)

//--- (end of YSoundSpectrum definitions)


//--- (YSoundSpectrum implementation)
bool   YSoundSpectrumCmdLine::isModule(void)      { return false; }
string YSoundSpectrumCmdLine::YFunctionName(void) { return "YSoundSpectrum"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of integrationTime is int (UInt31)
// type of spectrumData is string (BinaryBuffer)
/**
 * Returns the logical name of the sound spectrum analyzer.
 *
 * @return a string corresponding to the logical name of the sound spectrum analyzer
 *
 * On failure, throws an exception or returns YSoundSpectrum.LOGICALNAME_INVALID.
 */
class SoundSpectrum_get_logicalName : public YapiCommand /* arguments: */
{
public:
  SoundSpectrum_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the sound spectrum analyzer.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the sound spectrum analyzer. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the sound spectrum analyzer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SoundSpectrum_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  SoundSpectrum_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the sound spectrum analyzer.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the sound spectrum analyzer", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the sound spectrum analyzer.
 *
 * @return a string corresponding to a short string representing the current state of the sound spectrum analyzer
 *
 * On failure, throws an exception or returns YSoundSpectrum.ADVERTISEDVALUE_INVALID.
 */
class SoundSpectrum_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  SoundSpectrum_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the sound spectrum analyzer.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the integration time in milliseconds for calculating time
 * weighted spectrum data.
 *
 * @return an integer corresponding to the integration time in milliseconds for calculating time
 *         weighted spectrum data
 *
 * On failure, throws an exception or returns YSoundSpectrum.INTEGRATIONTIME_INVALID.
 */
class SoundSpectrum_get_integrationTime : public YapiCommand /* arguments: */
{
public:
  SoundSpectrum_get_integrationTime(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_integrationTime";
  }

  string getDescription()
  {
    return "Returns the integration time in milliseconds for calculating timeweighted spectrum data.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_integrationTime();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the integration time in milliseconds for computing time weighted
 * spectrum data. Be aware that on some devices, changing the integration
 * time for time-weighted spectrum data may also affect the integration
 * period for one or more sound pressure level measurements.
 * Remember to call the saveToFlash() method of the
 * module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the integration time in milliseconds for computing time weighted
 *         spectrum data
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SoundSpectrum_set_integrationTime : public YapiCommand /* arguments: newval */
{
public:
  SoundSpectrum_set_integrationTime(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_integrationTime";
  }

  string getDescription()
  {
    return "Changes the integration time in milliseconds for computing time weightedspectrum data.";
  }

  string getMoreInfo()
  {
    return "Be aware that on some devices, changing the integration time for time-weighted spectrum data may also affect the integration period for one or more sound pressure level measurements. Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the integration time in milliseconds for computing time weighted spectrum data", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_integrationTime(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackSoundSpectrum is YSoundSpectrumValueCallback* (YSoundSpectrumValueCallback)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SoundSpectrum_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_SoundSpectrum_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SoundSpectrum_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_SoundSpectrum_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_SoundSpectrum_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_SoundSpectrum_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_SoundSpectrum_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_SoundSpectrum_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSoundSpectrum*>* list = enumerateTargets<YSoundSpectrum>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YSoundSpectrum implementation)

//--- (YSoundSpectrum functions)
void YSoundSpectrumCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YSoundSpectrum>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new SoundSpectrum_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new SoundSpectrum_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new SoundSpectrum_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new SoundSpectrum_get_integrationTime(this)));
    cmdList->push_back((YapiCommand*) (new SoundSpectrum_set_integrationTime(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SoundSpectrum_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SoundSpectrum_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SoundSpectrum_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SoundSpectrum_get_serialNumber(this)));
  }

//--- (end of YSoundSpectrum functions)
