/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle Relay functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YRelay.h"
#include "yocto_api.h"
#include "yocto_relay.h"

using namespace std;

//--- (YRelay definitions)

static const char *enumOnOff[] = {
    "OFF",
    "ON",
    NULL
};

static const char *enumToggle[] = {
    "A",
    "B",
    NULL
};

static const char *enumToggleAtPowerOn[] = {
    "UNCHANGED",
    "A",
    "B",
    NULL
};

//--- (end of YRelay definitions)


//--- (YRelay implementation)
bool   YRelayCmdLine::isModule(void)      { return false; }
string YRelayCmdLine::YFunctionName(void) { return "YRelay"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of state is int (Toggle)
// type of stateAtPowerOn is int (ToggleAtPowerOn)
// type of maxTimeOnStateA is s64 (Time)
// type of maxTimeOnStateB is s64 (Time)
// type of output is int (OnOff)
// type of pulseTimer is s64 (Time)
// type of countdown is s64 (Time)
/**
 * Returns the logical name of the relay.
 *
 * @return a string corresponding to the logical name of the relay
 *
 * On failure, throws an exception or returns YRelay.LOGICALNAME_INVALID.
 */
class Relay_get_logicalName : public YapiCommand /* arguments: */
{
public:
  Relay_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the relay.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the relay. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the relay
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Relay_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  Relay_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the relay.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the relay", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the relay.
 *
 * @return a string corresponding to a short string representing the current state of the relay
 *
 * On failure, throws an exception or returns YRelay.ADVERTISEDVALUE_INVALID.
 */
class Relay_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  Relay_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the relay.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of the relays (A for the idle position, B for the active position).
 *
 * @return either YRelay.STATE_A or YRelay.STATE_B, according to the state of the relays (A for the
 * idle position, B for the active position)
 *
 * On failure, throws an exception or returns YRelay.STATE_INVALID.
 */
class Relay_get_state : public YapiCommand /* arguments: */
{
public:
  Relay_get_state(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_state";
  }

  string getDescription()
  {
    return "Returns the state of the relays (A for the idle position, B for the active position).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_state();
        string strval =  enumString(attrval,enumToggle);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the state of the relays (A for the idle position, B for the active position).
 *
 * @param newval : either YRelay.STATE_A or YRelay.STATE_B, according to the state of the relays (A
 * for the idle position, B for the active position)
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Relay_set_state : public YapiCommand /* arguments: newval */
{
public:
  Relay_set_state(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_state";
  }

  string getDescription()
  {
    return "Changes the state of the relays (A for the idle position, B for the active position).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either A or B, according to the state of the relays (A for the idle position, B for the active position)", "A=0,B=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_state(Y_STATE_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the state of the relays at device startup (A for the idle position,
 * B for the active position, UNCHANGED to leave the relay state as is).
 *
 * @return a value among YRelay.STATEATPOWERON_UNCHANGED, YRelay.STATEATPOWERON_A and
 * YRelay.STATEATPOWERON_B corresponding to the state of the relays at device startup (A for the idle position,
 *         B for the active position, UNCHANGED to leave the relay state as is)
 *
 * On failure, throws an exception or returns YRelay.STATEATPOWERON_INVALID.
 */
class Relay_get_stateAtPowerOn : public YapiCommand /* arguments: */
{
public:
  Relay_get_stateAtPowerOn(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_stateAtPowerOn";
  }

  string getDescription()
  {
    return "Returns the state of the relays at device startup (A for the idle position,B for the active position, UNCHANGED to leave the relay state as is).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_stateAtPowerOn();
        string strval =  enumString(attrval,enumToggleAtPowerOn);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the state of the relays at device startup (A for the idle position,
 * B for the active position, UNCHANGED to leave the relay state as is).
 * Remember to call the matching module saveToFlash()
 * method, otherwise this call will have no effect.
 *
 * @param newval : a value among YRelay.STATEATPOWERON_UNCHANGED, YRelay.STATEATPOWERON_A and
 * YRelay.STATEATPOWERON_B corresponding to the state of the relays at device startup (A for the idle position,
 *         B for the active position, UNCHANGED to leave the relay state as is)
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Relay_set_stateAtPowerOn : public YapiCommand /* arguments: newval */
{
public:
  Relay_set_stateAtPowerOn(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_stateAtPowerOn";
  }

  string getDescription()
  {
    return "Changes the state of the relays at device startup (A for the idle position,B for the active position, UNCHANGED to leave the relay state as is).";
  }

  string getMoreInfo()
  {
    return "Remember to call the matching module saveToFlash() method, otherwise this call will have no effect.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "a value among UNCHANGED, A and B corresponding to the state of the relays at device startup (A for the idle position, B for the active position, UNCHANGED to leave the relay state as is)", "UNCHANGED=0,A=1,B=2", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_stateAtPowerOn(Y_STATEATPOWERON_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the maximum time (ms) allowed for the relay to stay in state
 * A before automatically switching back in to B state. Zero means no time limit.
 *
 * @return an integer corresponding to the maximum time (ms) allowed for the relay to stay in state
 *         A before automatically switching back in to B state
 *
 * On failure, throws an exception or returns YRelay.MAXTIMEONSTATEA_INVALID.
 */
class Relay_get_maxTimeOnStateA : public YapiCommand /* arguments: */
{
public:
  Relay_get_maxTimeOnStateA(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_maxTimeOnStateA";
  }

  string getDescription()
  {
    return "Returns the maximum time (ms) allowed for the relay to stay in stateA before automatically switching back in to B state.";
  }

  string getMoreInfo()
  {
    return "Zero means no time limit.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        s64 attrval = (*list)[i]->get_maxTimeOnStateA();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%u [ms]", (u32)attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the maximum time (ms) allowed for the relay to stay in state A
 * before automatically switching back in to B state. Use zero for no time limit.
 * Remember to call the saveToFlash()
 * method of the module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the maximum time (ms) allowed for the relay to stay in state A
 *         before automatically switching back in to B state
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Relay_set_maxTimeOnStateA : public YapiCommand /* arguments: newval */
{
public:
  Relay_set_maxTimeOnStateA(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_maxTimeOnStateA";
  }

  string getDescription()
  {
    return "Changes the maximum time (ms) allowed for the relay to stay in state Abefore automatically switching back in to B state.";
  }

  string getMoreInfo()
  {
    return "Use zero for no time limit. Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(S64_ARG, "newval", "an integer corresponding to the maximum time (ms) allowed for the relay to stay in state A before automatically switching back in to B state", "_S64", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_maxTimeOnStateA(s64((*args)[0]->get_s64Value()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Retourne the maximum time (ms) allowed for the relay to stay in state B
 * before automatically switching back in to A state. Zero means no time limit.
 *
 * @return an integer
 *
 * On failure, throws an exception or returns YRelay.MAXTIMEONSTATEB_INVALID.
 */
class Relay_get_maxTimeOnStateB : public YapiCommand /* arguments: */
{
public:
  Relay_get_maxTimeOnStateB(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_maxTimeOnStateB";
  }

  string getDescription()
  {
    return "Retourne the maximum time (ms) allowed for the relay to stay in state Bbefore automatically switching back in to A state.";
  }

  string getMoreInfo()
  {
    return "Zero means no time limit.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        s64 attrval = (*list)[i]->get_maxTimeOnStateB();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%u [ms]", (u32)attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the maximum time (ms) allowed for the relay to stay in state B before
 * automatically switching back in to A state. Use zero for no time limit.
 * Remember to call the saveToFlash()
 * method of the module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the maximum time (ms) allowed for the relay to stay in
 * state B before
 *         automatically switching back in to A state
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Relay_set_maxTimeOnStateB : public YapiCommand /* arguments: newval */
{
public:
  Relay_set_maxTimeOnStateB(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_maxTimeOnStateB";
  }

  string getDescription()
  {
    return "Changes the maximum time (ms) allowed for the relay to stay in state B beforeautomatically switching back in to A state.";
  }

  string getMoreInfo()
  {
    return "Use zero for no time limit. Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(S64_ARG, "newval", "an integer corresponding to the maximum time (ms) allowed for the relay to stay in state B before automatically switching back in to A state", "_S64", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_maxTimeOnStateB(s64((*args)[0]->get_s64Value()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the output state of the relays, when used as a simple switch (single throw).
 *
 * @return either YRelay.OUTPUT_OFF or YRelay.OUTPUT_ON, according to the output state of the relays,
 * when used as a simple switch (single throw)
 *
 * On failure, throws an exception or returns YRelay.OUTPUT_INVALID.
 */
class Relay_get_output : public YapiCommand /* arguments: */
{
public:
  Relay_get_output(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_output";
  }

  string getDescription()
  {
    return "Returns the output state of the relays, when used as a simple switch (single throw).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_output();
        string strval =  enumString(attrval,enumOnOff);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the output state of the relays, when used as a simple switch (single throw).
 *
 * @param newval : either YRelay.OUTPUT_OFF or YRelay.OUTPUT_ON, according to the output state of the
 * relays, when used as a simple switch (single throw)
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Relay_set_output : public YapiCommand /* arguments: newval */
{
public:
  Relay_set_output(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_output";
  }

  string getDescription()
  {
    return "Changes the output state of the relays, when used as a simple switch (single throw).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either OFF or ON, according to the output state of the relays, when used as a simple switch (single throw)", "OFF=0,ON=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_output(Y_OUTPUT_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the number of milliseconds remaining before the relays is returned to idle position
 * (state A), during a measured pulse generation. When there is no ongoing pulse, returns zero.
 *
 * @return an integer corresponding to the number of milliseconds remaining before the relays is
 * returned to idle position
 *         (state A), during a measured pulse generation
 *
 * On failure, throws an exception or returns YRelay.PULSETIMER_INVALID.
 */
class Relay_get_pulseTimer : public YapiCommand /* arguments: */
{
public:
  Relay_get_pulseTimer(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_pulseTimer";
  }

  string getDescription()
  {
    return "Returns the number of milliseconds remaining before the relays is returned to idle position(state A), during a measured pulse generation.";
  }

  string getMoreInfo()
  {
    return "When there is no ongoing pulse, returns zero.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        s64 attrval = (*list)[i]->get_pulseTimer();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%u [ms]", (u32)attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Sets the relay to output B (active) for a specified duration, then brings it
 * automatically back to output A (idle state).
 *
 * @param ms_duration : pulse duration, in milliseconds
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class alias_Relay_pulse : public YapiCommand /* arguments: ms_duration */
{
public:
  alias_Relay_pulse(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "pulse";
  }

  string getDescription()
  {
    return "Sets the relay to output B (active) for a specified duration, then brings itautomatically back to output A (idle state).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "ms_duration", "pulse duration, in milliseconds", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->pulse((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Schedules a pulse.
 *
 * @param ms_delay : waiting time before the pulse, in milliseconds
 * @param ms_duration : pulse duration, in milliseconds
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class alias_Relay_delayedPulse : public YapiCommand /* arguments: ms_delay ms_duration */
{
public:
  alias_Relay_delayedPulse(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "delayedPulse";
  }

  string getDescription()
  {
    return "Schedules a pulse.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "ms_delay", "waiting time before the pulse, in milliseconds", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "ms_duration", "pulse duration, in milliseconds", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->delayedPulse((int)(*args)[0]->get_intValue(),
                                 (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the number of milliseconds remaining before a pulse (delayedPulse() call)
 * When there is no scheduled pulse, returns zero.
 *
 * @return an integer corresponding to the number of milliseconds remaining before a pulse (delayedPulse() call)
 *         When there is no scheduled pulse, returns zero
 *
 * On failure, throws an exception or returns YRelay.COUNTDOWN_INVALID.
 */
class Relay_get_countdown : public YapiCommand /* arguments: */
{
public:
  Relay_get_countdown(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_countdown";
  }

  string getDescription()
  {
    return "Returns the number of milliseconds remaining before a pulse (delayedPulse() call)When there is no scheduled pulse, returns zero.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        s64 attrval = (*list)[i]->get_countdown();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%u [ms]", (u32)attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

// type of valueCallbackRelay is YRelayValueCallback* (YRelayValueCallback)
// type of firm is int (int)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_Relay_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_Relay_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_Relay_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_Relay_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_Relay_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_Relay_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_Relay_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_Relay_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Switch the relay to the opposite state.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_Relay_toggle : public YapiCommand /* arguments: */
{
public:
  apifun_Relay_toggle(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "toggle";
  }

  string getDescription()
  {
    return "Switch the relay to the opposite state.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRelay*>* list = enumerateTargets<YRelay>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->toggle();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YRelay implementation)

//--- (YRelay functions)
void YRelayCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YRelay>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new Relay_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new Relay_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_state(this)));
    cmdList->push_back((YapiCommand*) (new Relay_set_state(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_stateAtPowerOn(this)));
    cmdList->push_back((YapiCommand*) (new Relay_set_stateAtPowerOn(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_maxTimeOnStateA(this)));
    cmdList->push_back((YapiCommand*) (new Relay_set_maxTimeOnStateA(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_maxTimeOnStateB(this)));
    cmdList->push_back((YapiCommand*) (new Relay_set_maxTimeOnStateB(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_output(this)));
    cmdList->push_back((YapiCommand*) (new Relay_set_output(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_pulseTimer(this)));
    cmdList->push_back((YapiCommand*) (new alias_Relay_pulse(this)));
    cmdList->push_back((YapiCommand*) (new alias_Relay_delayedPulse(this)));
    cmdList->push_back((YapiCommand*) (new Relay_get_countdown(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Relay_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Relay_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Relay_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Relay_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Relay_toggle(this)));
  }

//--- (end of YRelay functions)
