/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle MicroPython functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YMicroPython.h"
#include "yocto_api.h"
#include "yocto_micropython.h"

using namespace std;

//--- (YMicroPython definitions)

static const char *enumOnOff[] = {
    "OFF",
    "ON",
    NULL
};

//--- (end of YMicroPython definitions)


//--- (YMicroPython implementation)
bool   YMicroPythonCmdLine::isModule(void)      { return false; }
string YMicroPythonCmdLine::YFunctionName(void) { return "YMicroPython"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of lastMsg is string (Text)
// type of heapUsage is int (Percent)
// type of xheapUsage is int (Percent)
// type of currentScript is string (Text)
// type of startupScript is string (Text)
// type of debugMode is int (OnOff)
// type of command is string (Text)
/**
 * Returns the logical name of the MicroPython interpreter.
 *
 * @return a string corresponding to the logical name of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.LOGICALNAME_INVALID.
 */
class MicroPython_get_logicalName : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the MicroPython interpreter.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the MicroPython interpreter. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the MicroPython interpreter
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the MicroPython interpreter.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the MicroPython interpreter", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the MicroPython interpreter.
 *
 * @return a string corresponding to a short string representing the current state of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.ADVERTISEDVALUE_INVALID.
 */
class MicroPython_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the MicroPython interpreter.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the last message produced by a python script.
 *
 * @return a string corresponding to the last message produced by a python script
 *
 * On failure, throws an exception or returns YMicroPython.LASTMSG_INVALID.
 */
class MicroPython_get_lastMsg : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_lastMsg(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastMsg";
  }

  string getDescription()
  {
    return "Returns the last message produced by a python script.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_lastMsg();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the percentage of micropython main memory in use,
 * as observed at the end of the last garbage collection.
 *
 * @return an integer corresponding to the percentage of micropython main memory in use,
 *         as observed at the end of the last garbage collection
 *
 * On failure, throws an exception or returns YMicroPython.HEAPUSAGE_INVALID.
 */
class MicroPython_get_heapUsage : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_heapUsage(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_heapUsage";
  }

  string getDescription()
  {
    return "Returns the percentage of micropython main memory in use,as observed at the end of the last garbage collection.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_heapUsage();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the percentage of micropython external memory in use,
 * as observed at the end of the last garbage collection.
 *
 * @return an integer corresponding to the percentage of micropython external memory in use,
 *         as observed at the end of the last garbage collection
 *
 * On failure, throws an exception or returns YMicroPython.XHEAPUSAGE_INVALID.
 */
class MicroPython_get_xheapUsage : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_xheapUsage(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_xheapUsage";
  }

  string getDescription()
  {
    return "Returns the percentage of micropython external memory in use,as observed at the end of the last garbage collection.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_xheapUsage();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of currently active script, if any.
 *
 * @return a string corresponding to the name of currently active script, if any
 *
 * On failure, throws an exception or returns YMicroPython.CURRENTSCRIPT_INVALID.
 */
class MicroPython_get_currentScript : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_currentScript(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_currentScript";
  }

  string getDescription()
  {
    return "Returns the name of currently active script, if any.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_currentScript();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Stops current running script, and/or selects a script to run immediately in a
 * fresh new environment. If the MicroPython interpreter is busy running a script,
 * this function will abort it immediately and reset the execution environment.
 * If a non-empty string is given as argument, the new script will be started.
 *
 * @param newval : a string
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_currentScript : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_currentScript(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_currentScript";
  }

  string getDescription()
  {
    return "Stops current running script, and/or selects a script to run immediately in afresh new environment.";
  }

  string getMoreInfo()
  {
    return "If the MicroPython interpreter is busy running a script, this function will abort it immediately and reset the execution environment. If a non-empty string is given as argument, the new script will be started.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_currentScript(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the name of the script to run when the device is powered on.
 *
 * @return a string corresponding to the name of the script to run when the device is powered on
 *
 * On failure, throws an exception or returns YMicroPython.STARTUPSCRIPT_INVALID.
 */
class MicroPython_get_startupScript : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_startupScript(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_startupScript";
  }

  string getDescription()
  {
    return "Returns the name of the script to run when the device is powered on.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_startupScript();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the script to run when the device is powered on.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the script to run when the device is powered on
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_startupScript : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_startupScript(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_startupScript";
  }

  string getDescription()
  {
    return "Changes the script to run when the device is powered on.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the script to run when the device is powered on", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_startupScript(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the activation state of micropython debugging interface.
 *
 * @return either YMicroPython.DEBUGMODE_OFF or YMicroPython.DEBUGMODE_ON, according to the activation
 * state of micropython debugging interface
 *
 * On failure, throws an exception or returns YMicroPython.DEBUGMODE_INVALID.
 */
class MicroPython_get_debugMode : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_debugMode(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_debugMode";
  }

  string getDescription()
  {
    return "Returns the activation state of micropython debugging interface.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_debugMode();
        string strval =  enumString(attrval,enumOnOff);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the activation state of micropython debugging interface.
 *
 * @param newval : either YMicroPython.DEBUGMODE_OFF or YMicroPython.DEBUGMODE_ON, according to the
 * activation state of micropython debugging interface
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_debugMode : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_debugMode(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_debugMode";
  }

  string getDescription()
  {
    return "Changes the activation state of micropython debugging interface.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either OFF or ON, according to the activation state of micropython debugging interface", "OFF=0,ON=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_debugMode(Y_DEBUGMODE_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackMicroPython is YMicroPythonValueCallback* (YMicroPythonValueCallback)
// type of logCallback is YMicroPythonLogCallback* (YMicroPythonLogCallback)
// type of isFirstCb is bool (bool)
// type of prevCbPos is int (int)
// type of logPos is int (int)
// type of prevPartialLog is string (str)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_MicroPython_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_MicroPython_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Submit MicroPython code for execution in the interpreter.
 * If the MicroPython interpreter is busy, this function will
 * block until it becomes available. The code is then uploaded,
 * compiled and executed on the fly, without beeing stored on the device filesystem.
 *
 * There is no implicit reset of the MicroPython interpreter with
 * this function. Use method reset() if you need to start
 * from a fresh environment to run your code.
 *
 * Note that although MicroPython is mostly compatible with recent Python 3.x
 * interpreters, the limited ressources on the device impose some restrictions,
 * in particular regarding the libraries that can be used. Please refer to
 * the documentation for more details.
 *
 * @param codeName : name of the code file (used for error reporting only)
 * @param mpyCode : MicroPython code to compile and execute
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_eval : public YapiCommand /* arguments: codeName mpyCode */
{
public:
  apifun_MicroPython_eval(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "eval";
  }

  string getDescription()
  {
    return "Submit MicroPython code for execution in the interpreter.";
  }

  string getMoreInfo()
  {
    return "If the MicroPython interpreter is busy, this function will block until it becomes available. The code is then uploaded, compiled and executed on the fly, without beeing stored on the device filesystem.\nThere is no implicit reset of the MicroPython interpreter with this function. Use method reset() if you need to start from a fresh environment to run your code.\nNote that although MicroPython is mostly compatible with recent Python 3.x interpreters, the limited ressources on the device impose some restrictions, in particular regarding the libraries that can be used. Please refer to the documentation for more details.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "codeName", "name of the code file (used for error reporting only)", "_STRING", false));
    res->push_back(new ArgumentDesc(STRING_ARG, "mpyCode", "MicroPython code to compile and execute", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->eval((*args)[0]->get_stringValue(),
                         (*args)[1]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Stops current execution, and reset the MicroPython interpreter to initial state.
 * All global variables are cleared, and all imports are forgotten.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_reset : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_reset(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "reset";
  }

  string getDescription()
  {
    return "Stops current execution, and reset the MicroPython interpreter to initial state.";
  }

  string getMoreInfo()
  {
    return "All global variables are cleared, and all imports are forgotten.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->reset();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns a string with last logs of the MicroPython interpreter.
 * This method return only logs that are still in the module.
 *
 * @return a string with last MicroPython logs.
 *         On failure, throws an exception or returns  YAPI_INVALID_STRING.
 */
class apifun_MicroPython_get_lastLogs : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_get_lastLogs(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastLogs";
  }

  string getDescription()
  {
    return "Returns a string with last logs of the MicroPython interpreter.";
  }

  string getMoreInfo()
  {
    return "This method return only logs that are still in the module.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_lastLogs();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YMicroPython implementation)

//--- (YMicroPython functions)
void YMicroPythonCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YMicroPython>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new MicroPython_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_lastMsg(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_heapUsage(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_xheapUsage(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_currentScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_currentScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_startupScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_startupScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_debugMode(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_debugMode(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_eval(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_reset(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_get_lastLogs(this)));
  }

//--- (end of YMicroPython functions)
