/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle RfidReader functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YRfidReader.h"
#include "yocto_api.h"
#include "yocto_rfidreader.h"

using namespace std;

//--- (generated code: YRfidReader definitions)

//--- (end of generated code: YRfidReader definitions)


//--- (generated code: YRfidReader implementation)
bool   YRfidReaderCmdLine::isModule(void)      { return false; }
string YRfidReaderCmdLine::YFunctionName(void) { return "YRfidReader"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of nTags is int (UInt31)
// type of refreshRate is int (UInt31)
/**
 * Returns the logical name of the RFID reader.
 *
 * @return a string corresponding to the logical name of the RFID reader
 *
 * On failure, throws an exception or returns YRfidReader.LOGICALNAME_INVALID.
 */
class RfidReader_get_logicalName : public YapiCommand /* arguments: */
{
public:
  RfidReader_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the RFID reader.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the RFID reader. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the RFID reader
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class RfidReader_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  RfidReader_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the RFID reader.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the RFID reader", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the RFID reader.
 *
 * @return a string corresponding to a short string representing the current state of the RFID reader
 *
 * On failure, throws an exception or returns YRfidReader.ADVERTISEDVALUE_INVALID.
 */
class RfidReader_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  RfidReader_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the RFID reader.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the number of RFID tags currently detected.
 *
 * @return an integer corresponding to the number of RFID tags currently detected
 *
 * On failure, throws an exception or returns YRfidReader.NTAGS_INVALID.
 */
class RfidReader_get_nTags : public YapiCommand /* arguments: */
{
public:
  RfidReader_get_nTags(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nTags";
  }

  string getDescription()
  {
    return "Returns the number of RFID tags currently detected.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_nTags();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the tag list refresh rate, measured in Hz.
 *
 * @return an integer corresponding to the tag list refresh rate, measured in Hz
 *
 * On failure, throws an exception or returns YRfidReader.REFRESHRATE_INVALID.
 */
class RfidReader_get_refreshRate : public YapiCommand /* arguments: */
{
public:
  RfidReader_get_refreshRate(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_refreshRate";
  }

  string getDescription()
  {
    return "Returns the tag list refresh rate, measured in Hz.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_refreshRate();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the present tag list refresh rate, measured in Hz. The reader will do
 * its best to respect it. Note that the reader cannot detect tag arrival or removal
 * while it is  communicating with a tag.  Maximum frequency is limited to 100Hz,
 * but in real life it will be difficult to do better than 50Hz.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : an integer corresponding to the present tag list refresh rate, measured in Hz
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class RfidReader_set_refreshRate : public YapiCommand /* arguments: newval */
{
public:
  RfidReader_set_refreshRate(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_refreshRate";
  }

  string getDescription()
  {
    return "Changes the present tag list refresh rate, measured in Hz.";
  }

  string getMoreInfo()
  {
    return "The reader will do its best to respect it. Note that the reader cannot detect tag arrival or removal while it is communicating with a tag. Maximum frequency is limited to 100Hz, but in real life it will be difficult to do better than 50Hz. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the present tag list refresh rate, measured in Hz", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_refreshRate(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackRfidReader is YRfidReaderValueCallback* (YRfidReaderValueCallback)
// type of eventCallback is YEventCallback* (YEventCallback)
// type of isFirstCb is bool (bool)
// type of prevCbPos is int (int)
// type of eventPos is int (int)
// type of eventStamp is int (int)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_RfidReader_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_RfidReader_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_RfidReader_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_RfidReader_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_RfidReader_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_RfidReader_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_RfidReader_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_RfidReader_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the list of RFID tags currently detected by the reader.
 *
 * @return a list of strings, corresponding to each tag identifier.
 *
 * On failure, throws an exception or returns an empty list.
 */
class apifun_RfidReader_get_tagIdList : public YapiCommand /* arguments: */
{
public:
  apifun_RfidReader_get_tagIdList(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_tagIdList";
  }

  string getDescription()
  {
    return "Returns the list of RFID tags currently detected by the reader.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<string> value = (*list)[i]->get_tagIdList();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Retourne la description des propriétés d'un tag RFID présent.
 * Cette fonction peut causer des communications avec le tag.
 *
 * @param tagId : identifier of the tag to check
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns an empty YRfidTagInfo objact.
 * When it happens, you can get more information from the status object.
 */
class apifun_RfidReader_get_tagInfo : public YapiCommand /* arguments: tagId status */
{
public:
  apifun_RfidReader_get_tagInfo(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_tagInfo";
  }

  string getDescription()
  {
    return "Retourne la description des propri\303\251t\303\251s d'un tag RFID pr\303\251sent.";
  }

  string getMoreInfo()
  {
    return "Cette fonction peut causer des communications avec le tag.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to check", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        YRfidTagInfo value = (*list)[i]->get_tagInfo((*args)[0]->get_stringValue(),
                                                     status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Change an RFID tag configuration to prevents any further write to
 * the selected blocks. This operation is definitive and irreversible.
 * Depending on the tag type and block index, adjascent blocks may become
 * read-only as well, based on the locking granularity.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : first block to lock
 * @param nBlocks : number of blocks to lock
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagLockBlocks : public YapiCommand /* arguments: tagId firstBlock nBlocks options status */
{
public:
  apifun_RfidReader_tagLockBlocks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagLockBlocks";
  }

  string getDescription()
  {
    return "Change an RFID tag configuration to prevents any further write tothe selected blocks.";
  }

  string getMoreInfo()
  {
    return "This operation is definitive and irreversible. Depending on the tag type and block index, adjascent blocks may become read-only as well, based on the locking granularity.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "first block to lock", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBlocks", "number of blocks to lock", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    value = value + " "+ (*args)[3]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        (*list)[i]->tagLockBlocks((*args)[0]->get_stringValue(),
                                  (int)(*args)[1]->get_intValue(),
                                  (int)(*args)[2]->get_intValue(),
                                  (*args)[3]->get_YRfidOptionsValue(),
                                  status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Reads the locked state for RFID tag memory data blocks.
 * FirstBlock cannot be a special block, and any special
 * block encountered in the middle of the read operation will be
 * skipped automatically.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : number of the first block to check
 * @param nBlocks : number of blocks to check
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return a list of booleans with the lock state of selected blocks
 *
 * On failure, throws an exception or returns an empty list. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_get_tagLockState : public YapiCommand /* arguments: tagId firstBlock nBlocks options status */
{
public:
  apifun_RfidReader_get_tagLockState(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_tagLockState";
  }

  string getDescription()
  {
    return "Reads the locked state for RFID tag memory data blocks.";
  }

  string getMoreInfo()
  {
    return "FirstBlock cannot be a special block, and any special block encountered in the middle of the read operation will be skipped automatically.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "number of the first block to check", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBlocks", "number of blocks to check", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        vector<bool> value = (*list)[i]->get_tagLockState((*args)[0]->get_stringValue(),
                                                          (int)(*args)[1]->get_intValue(),
                                                          (int)(*args)[2]->get_intValue(),
                                                          (*args)[3]->get_YRfidOptionsValue(),
                                                          status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Tells which block of a RFID tag memory are special and cannot be used
 * to store user data. Mistakely writing a special block can lead to
 * an irreversible alteration of the tag.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : number of the first block to check
 * @param nBlocks : number of blocks to check
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return a list of booleans with the lock state of selected blocks
 *
 * On failure, throws an exception or returns an empty list. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_get_tagSpecialBlocks : public YapiCommand /* arguments: tagId firstBlock nBlocks options status */
{
public:
  apifun_RfidReader_get_tagSpecialBlocks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_tagSpecialBlocks";
  }

  string getDescription()
  {
    return "Tells which block of a RFID tag memory are special and cannot be usedto store user data.";
  }

  string getMoreInfo()
  {
    return "Mistakely writing a special block can lead to an irreversible alteration of the tag.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "number of the first block to check", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBlocks", "number of blocks to check", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        vector<bool> value = (*list)[i]->get_tagSpecialBlocks((*args)[0]->get_stringValue(),
                                                              (int)(*args)[1]->get_intValue(),
                                                              (int)(*args)[2]->get_intValue(),
                                                              (*args)[3]->get_YRfidOptionsValue(),
                                                              status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Reads data from an RFID tag memory, as an hexadecimal string.
 * The read operation may span accross multiple blocks if the requested
 * number of bytes is larger than the RFID tag block size. By default
 * firstBlock cannot be a special block, and any special block encountered
 * in the middle of the read operation will be skipped automatically.
 * If you rather want to read special blocks, use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where read should start
 * @param nBytes : total number of bytes to read
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return an hexadecimal string if the call succeeds.
 *
 * On failure, throws an exception or returns an empty binary buffer. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagReadHex : public YapiCommand /* arguments: tagId firstBlock nBytes options status */
{
public:
  apifun_RfidReader_tagReadHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagReadHex";
  }

  string getDescription()
  {
    return "Reads data from an RFID tag memory, as an hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The read operation may span accross multiple blocks if the requested number of bytes is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the read operation will be skipped automatically. If you rather want to read special blocks, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where read should start", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBytes", "total number of bytes to read", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        string value = (*list)[i]->tagReadHex((*args)[0]->get_stringValue(),
                                              (int)(*args)[1]->get_intValue(),
                                              (int)(*args)[2]->get_intValue(),
                                              (*args)[3]->get_YRfidOptionsValue(),
                                              status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Reads data from an RFID tag memory, as a binary buffer. The read operation
 * may span accross multiple blocks if the requested number of bytes
 * is larger than the RFID tag block size.  By default
 * firstBlock cannot be a special block, and any special block encountered
 * in the middle of the read operation will be skipped automatically.
 * If you rather want to read special blocks, use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where read should start
 * @param nBytes : total number of bytes to read
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return a binary object with the data read if the call succeeds.
 *
 * On failure, throws an exception or returns an empty binary buffer. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagReadBin : public YapiCommand /* arguments: tagId firstBlock nBytes options status */
{
public:
  apifun_RfidReader_tagReadBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagReadBin";
  }

  string getDescription()
  {
    return "Reads data from an RFID tag memory, as a binary buffer.";
  }

  string getMoreInfo()
  {
    return "The read operation may span accross multiple blocks if the requested number of bytes is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the read operation will be skipped automatically. If you rather want to read special blocks, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where read should start", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBytes", "total number of bytes to read", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        string value = (*list)[i]->tagReadBin((*args)[0]->get_stringValue(),
                                              (int)(*args)[1]->get_intValue(),
                                              (int)(*args)[2]->get_intValue(),
                                              (*args)[3]->get_YRfidOptionsValue(),
                                              status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Reads data from an RFID tag memory, as a byte list. The read operation
 * may span accross multiple blocks if the requested number of bytes
 * is larger than the RFID tag block size.  By default
 * firstBlock cannot be a special block, and any special block encountered
 * in the middle of the read operation will be skipped automatically.
 * If you rather want to read special blocks, use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where read should start
 * @param nBytes : total number of bytes to read
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return a byte list with the data read if the call succeeds.
 *
 * On failure, throws an exception or returns an empty list. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagReadArray : public YapiCommand /* arguments: tagId firstBlock nBytes options status */
{
public:
  apifun_RfidReader_tagReadArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagReadArray";
  }

  string getDescription()
  {
    return "Reads data from an RFID tag memory, as a byte list.";
  }

  string getMoreInfo()
  {
    return "The read operation may span accross multiple blocks if the requested number of bytes is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the read operation will be skipped automatically. If you rather want to read special blocks, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where read should start", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBytes", "total number of bytes to read", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        vector<int> value = (*list)[i]->tagReadArray((*args)[0]->get_stringValue(),
                                                     (int)(*args)[1]->get_intValue(),
                                                     (int)(*args)[2]->get_intValue(),
                                                     (*args)[3]->get_YRfidOptionsValue(),
                                                     status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Reads data from an RFID tag memory, as a text string. The read operation
 * may span accross multiple blocks if the requested number of bytes
 * is larger than the RFID tag block size.  By default
 * firstBlock cannot be a special block, and any special block encountered
 * in the middle of the read operation will be skipped automatically.
 * If you rather want to read special blocks, use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where read should start
 * @param nChars : total number of characters to read
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return a text string with the data read if the call succeeds.
 *
 * On failure, throws an exception or returns an empty string. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagReadStr : public YapiCommand /* arguments: tagId firstBlock nChars options status */
{
public:
  apifun_RfidReader_tagReadStr(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagReadStr";
  }

  string getDescription()
  {
    return "Reads data from an RFID tag memory, as a text string.";
  }

  string getMoreInfo()
  {
    return "The read operation may span accross multiple blocks if the requested number of bytes is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the read operation will be skipped automatically. If you rather want to read special blocks, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where read should start", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nChars", "total number of characters to read", "_INT", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        string value = (*list)[i]->tagReadStr((*args)[0]->get_stringValue(),
                                              (int)(*args)[1]->get_intValue(),
                                              (int)(*args)[2]->get_intValue(),
                                              (*args)[3]->get_YRfidOptionsValue(),
                                              status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Writes data provided as a binary buffer to an RFID tag memory.
 * The write operation may span accross multiple blocks if the
 * number of bytes to write is larger than the RFID tag block size.
 * By default firstBlock cannot be a special block, and any special block
 * encountered in the middle of the write operation will be skipped
 * automatically. If you rather want to rewrite special blocks as well,
 * use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where write should start
 * @param buff : the binary buffer to write
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagWriteBin : public YapiCommand /* arguments: tagId firstBlock buff options status */
{
public:
  apifun_RfidReader_tagWriteBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagWriteBin";
  }

  string getDescription()
  {
    return "Writes data provided as a binary buffer to an RFID tag memory.";
  }

  string getMoreInfo()
  {
    return "The write operation may span accross multiple blocks if the number of bytes to write is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the write operation will be skipped automatically. If you rather want to rewrite special blocks as well, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where write should start", "_INT", false));
    res->push_back(new ArgumentDesc(BIN_ARG, "buff", "the binary buffer to write", "_BIN", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    value = value + " "+ (*args)[3]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        (*list)[i]->tagWriteBin((*args)[0]->get_stringValue(),
                                (int)(*args)[1]->get_intValue(),
                                (*args)[2]->get_binValue(),
                                (*args)[3]->get_YRfidOptionsValue(),
                                status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Writes data provided as a list of bytes to an RFID tag memory.
 * The write operation may span accross multiple blocks if the
 * number of bytes to write is larger than the RFID tag block size.
 * By default firstBlock cannot be a special block, and any special block
 * encountered in the middle of the write operation will be skipped
 * automatically. If you rather want to rewrite special blocks as well,
 * use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where write should start
 * @param byteList : a list of byte to write
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagWriteArray : public YapiCommand /* arguments: tagId firstBlock byteList options status */
{
public:
  apifun_RfidReader_tagWriteArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagWriteArray";
  }

  string getDescription()
  {
    return "Writes data provided as a list of bytes to an RFID tag memory.";
  }

  string getMoreInfo()
  {
    return "The write operation may span accross multiple blocks if the number of bytes to write is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the write operation will be skipped automatically. If you rather want to rewrite special blocks as well, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where write should start", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "byteList", "a list of byte to write", "_INTARR", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    value = value + " "+ (*args)[3]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        (*list)[i]->tagWriteArray((*args)[0]->get_stringValue(),
                                  (int)(*args)[1]->get_intValue(),
                                  (*args)[2]->get_intArrValue(),
                                  (*args)[3]->get_YRfidOptionsValue(),
                                  status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Writes data provided as an hexadecimal string to an RFID tag memory.
 * The write operation may span accross multiple blocks if the
 * number of bytes to write is larger than the RFID tag block size.
 * By default firstBlock cannot be a special block, and any special block
 * encountered in the middle of the write operation will be skipped
 * automatically. If you rather want to rewrite special blocks as well,
 * use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where write should start
 * @param hexString : a string of hexadecimal byte codes to write
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagWriteHex : public YapiCommand /* arguments: tagId firstBlock hexString options status */
{
public:
  apifun_RfidReader_tagWriteHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagWriteHex";
  }

  string getDescription()
  {
    return "Writes data provided as an hexadecimal string to an RFID tag memory.";
  }

  string getMoreInfo()
  {
    return "The write operation may span accross multiple blocks if the number of bytes to write is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the write operation will be skipped automatically. If you rather want to rewrite special blocks as well, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where write should start", "_INT", false));
    res->push_back(new ArgumentDesc(STRING_ARG, "hexString", "a string of hexadecimal byte codes to write", "_STRING", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    value = value + " "+ (*args)[3]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        (*list)[i]->tagWriteHex((*args)[0]->get_stringValue(),
                                (int)(*args)[1]->get_intValue(),
                                (*args)[2]->get_stringValue(),
                                (*args)[3]->get_YRfidOptionsValue(),
                                status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Writes data provided as an ASCII string to an RFID tag memory.
 * The write operation may span accross multiple blocks if the
 * number of bytes to write is larger than the RFID tag block size.
 * By default firstBlock cannot be a special block, and any special block
 * encountered in the middle of the write operation will be skipped
 * automatically. If you rather want to rewrite special blocks as well,
 * use EnableRawAccess option.
 *
 * @param tagId : identifier of the tag to use
 * @param firstBlock : block number where write should start
 * @param text : the text string to write
 * @param options : an YRfidOptions object with the optional
 *         command execution parameters, such as security key
 *         if required
 * @param status : an RfidStatus object that will contain
 *         the detailled status of the operation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code. When it
 * happens, you can get more information from the status object.
 */
class apifun_RfidReader_tagWriteStr : public YapiCommand /* arguments: tagId firstBlock text options status */
{
public:
  apifun_RfidReader_tagWriteStr(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "tagWriteStr";
  }

  string getDescription()
  {
    return "Writes data provided as an ASCII string to an RFID tag memory.";
  }

  string getMoreInfo()
  {
    return "The write operation may span accross multiple blocks if the number of bytes to write is larger than the RFID tag block size. By default firstBlock cannot be a special block, and any special block encountered in the middle of the write operation will be skipped automatically. If you rather want to rewrite special blocks as well, use EnableRawAccess option.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "tagId", "identifier of the tag to use", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "firstBlock", "block number where write should start", "_INT", false));
    res->push_back(new ArgumentDesc(STRING_ARG, "text", "the text string to write", "_STRING", false));
    res->push_back(new ArgumentDesc(YRFIDOPTIONS_ARG, "options", "an YRfidOptions object with the optional command execution parameters, such as security key if required", "_YRFIDOPTIONS", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    value = value + " "+ (*args)[3]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        YRfidStatus status;
        (*list)[i]->tagWriteStr((*args)[0]->get_stringValue(),
                                (int)(*args)[1]->get_intValue(),
                                (*args)[2]->get_stringValue(),
                                (*args)[3]->get_YRfidOptionsValue(),
                                status);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
        PrintResult(resultformat, this->getName()+".status",YFunctionInfoCache((*list)[i]), status, true);
      }
  }
};

/**
 * Returns a string with last tag arrival/removal events observed.
 * This method return only events that are still buffered in the device memory.
 *
 * @return a string with last events observed (one per line).
 *
 * On failure, throws an exception or returns  YAPI_INVALID_STRING.
 */
class apifun_RfidReader_get_lastEvents : public YapiCommand /* arguments: */
{
public:
  apifun_RfidReader_get_lastEvents(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastEvents";
  }

  string getDescription()
  {
    return "Returns a string with last tag arrival/removal events observed.";
  }

  string getMoreInfo()
  {
    return "This method return only events that are still buffered in the device memory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YRfidReader*>* list = enumerateTargets<YRfidReader>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_lastEvents();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of generated code: YRfidReader implementation)

//--- (generated code: YRfidReader functions)
void YRfidReaderCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YRfidReader>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new RfidReader_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new RfidReader_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new RfidReader_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new RfidReader_get_nTags(this)));
    cmdList->push_back((YapiCommand*) (new RfidReader_get_refreshRate(this)));
    cmdList->push_back((YapiCommand*) (new RfidReader_set_refreshRate(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_get_tagIdList(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_get_tagInfo(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagLockBlocks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_get_tagLockState(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_get_tagSpecialBlocks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagReadHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagReadBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagReadArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagReadStr(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagWriteBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagWriteArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagWriteHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_tagWriteStr(this)));
    cmdList->push_back((YapiCommand*) (new apifun_RfidReader_get_lastEvents(this)));
  }

//--- (end of generated code: YRfidReader functions)
