/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle MicroPython functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YMicroPython.h"

#include <complex>
#include <fstream>
#include <iostream>


#include "yocto_api.h"
#include "yocto_micropython.h"
#include "yapi/yproto.h"

using namespace std;

//--- (generated code: YMicroPython definitions)

static const char *enumOnOff[] = {
    "OFF",
    "ON",
    NULL
};

//--- (end of generated code: YMicroPython definitions)

#ifdef WITH_MPY_CROSS
#include "ympy_cross.h"

#ifndef WINDOWS_API
#include <dirent.h>
#include <sys/stat.h>
#endif

/**
 * Returns the logical name of the MicroPython interpreter.
 *
 * @return a string corresponding to the logical name of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.LOGICALNAME_INVALID.
 */
class MicroPython_compileFile : public YapiCommand /* arguments: */
{
public:
    MicroPython_compileFile(YFunctionCmdLine *function): YapiCommand(function)
    {}

    string getName()
    {
        return "compileFile";
    }

    bool requireTarget()
    {
        return false;
    }

    bool requireRegisterHub()
    {
        return false;
    }

    string getDescription()
    {
        return "Compile python source file into mpy file.";
    }


    vector<ArgumentDesc*>* getArgumentDesc()
    {
        vector<ArgumentDesc*> *res = new vector<ArgumentDesc*>();
        res->push_back(new ArgumentDesc(STRING_ARG, "inputFile", "The path of the source file", "_STRING", false));
        res->push_back(new ArgumentDesc(STRING_ARG, "outFile", "The path of the compiled file", "_STRING", false));
        return res;
    }


    void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*> *args, vector<SwitchDesc*> *switches)
    {
        string src_file = (*args)[0]->get_stringValue();
        string dst_file = (*args)[1]->get_stringValue();
        int res = ycompile(src_file.c_str(), dst_file.c_str());
        if (res < 0) {
            YapiCommand::fatalError("Compilation failed (file=" + src_file + ")");
        }
    }
};


static void load_file(const string &path, string &content)
{
    content = "";
    std::ifstream is(path.c_str(), std::ifstream::binary);
    if (!is) {
        YapiCommand::fatalError("cannot open file " + path);
    } else {
        is.seekg(0, is.end);
        int length = (int)is.tellg();
        is.seekg(0, is.beg);
        char *buffer = new char [length];
        is.read(buffer, length);
        is.close();
        content = string(buffer, length);
        delete[] buffer;
    }
}

/**
 * Returns the logical name of the MicroPython interpreter.
 *
 * @return a string corresponding to the logical name of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.LOGICALNAME_INVALID.
 */
class MicroPython_uploadFile : public YapiCommand /* arguments: */
{
public:
    MicroPython_uploadFile(YFunctionCmdLine *function): YapiCommand(function)
    {}

    string getName()
    {
        return "uploadFile";
    }

    string getDescription()
    {
        return "Uploads a file to the filesystem, to the specified full path name.";
    }


    vector<ArgumentDesc*>* getArgumentDesc()
    {
        vector<ArgumentDesc*> *res = new vector<ArgumentDesc*>();
        res->push_back(new ArgumentDesc(STRING_ARG, "sourceFile", "The path of the file to upload", "_STRING", false));
        res->push_back(new ArgumentDesc(STRING_ARG, "destination", "The destination where to upload the file", "_STRING", false));
        return res;
    }


    void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*> *args, vector<SwitchDesc*> *switches)
    {
        vector<YMicroPython*> *list = enumerateTargets<YMicroPython>(_function, target, modulelist);
        unsigned int i;
        string value = "";
        value = value + (*args)[0]->to_str();
        value = value + " -> " + (*args)[1]->to_str();
        string path = (*args)[0]->get_stringValue();
        string content;
        load_file(path, content);
        for (i = 0; i < list->size(); i++) {
            (*list)[i]->_upload((*args)[1]->get_stringValue(),
                                content);
            PrintResult(resultformat, this->getName(), YFunctionInfoCache((*list)[i]), value, true);
        }
    }
};

/**
 * Returns the logical name of the MicroPython interpreter.
 *
 * @return a string corresponding to the logical name of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.LOGICALNAME_INVALID.
 */
class MicroPython_compileAndUpload : public YapiCommand /* arguments: */
{
public:
    MicroPython_compileAndUpload(YFunctionCmdLine *function): YapiCommand(function)
    {}

    string getName()
    {
        return "compileAndUpload";
    }

    string getDescription()
    {
        return "compile and uploads a file or directory to the filesystem.";
    }

    vector<ArgumentDesc*>* getArgumentDesc()
    {
        vector<ArgumentDesc*> *res = new vector<ArgumentDesc*>();
        res->push_back(new ArgumentDesc(STRING_ARG, "sourceFile", "The path of the file to upload", "_STRING", false));
        res->push_back(new ArgumentDesc(STRING_ARG, "destination", "The destination where to upload the file", "_STRING", true));
        res->push_back(new ArgumentDesc(STRING_ARG, "patern", "Optional pattern to apply on filename (*.py)", "_STRING", true));
        return res;
    }


    int compile_and_upload_file(const string &resultformat, vector<YMicroPython*> *list, string src_file, string base_path, const string &patern)
    {
        unsigned i;
        string filename;

        size_t lastSlash = src_file.find_last_of("/\\");
        if (lastSlash != string::npos) {
            filename = src_file.substr(lastSlash + 1, string::npos);
        } else {
            filename = src_file;
        }
        if (!StrMatch(filename.c_str(), patern.c_str())) {
            return 0;
        }
        string to_upload = src_file;
        size_t lastDot = src_file.find_last_of(".");
        if (lastDot != std::string::npos && (lastDot > lastSlash || lastSlash == string::npos)) {
            string ext = src_file.substr(lastDot);
            if (ext == ".py") {
                string mpy_file = src_file.substr(0, lastDot) + ".mpy";
                if (ycompile(src_file.c_str(), mpy_file.c_str()) < 0) {
                    YapiCommand::fatalError("Compilation of " + src_file + " failed");
                }
                src_file = mpy_file;
                if (lastSlash != string::npos) {
                    filename = mpy_file.substr(lastSlash + 1, string::npos);
                } else {
                    filename = mpy_file;
                }
            }
        }
        string dst_path = base_path + filename;
        string content;
        load_file(src_file, content);
        for (i = 0; i < list->size(); i++) {
            (*list)[i]->_upload(dst_path,
                                content);
            PrintResult(resultformat, this->getName(), YFunctionInfoCache((*list)[i]), to_upload + " -> " + dst_path, true);
        }
        return 1;
    }

    int StrMatch(const char *str, const char *expr)
    {
        const char *next = NULL;
        char c1, c2;

        while (*expr) {
            switch (*expr) {
            case '?':
                if (*str) return 0;
                str++;
                expr++;
                break;
            case '*':
                while (expr[1] == '*') expr++;
                if (*++expr == 0) return 1;
                c2 = *expr;
            tryagain:
                while (*str) {
                    c1 = *str;
                    next = str + 1;
                    if (c1 == c2) break;
                    str = next;
                }
                if (*str && StrMatch(str, expr))
                    return 1;
                if (*str && *next) {
                    str = next;
                    goto tryagain;
                }
                return 0;
            default:
                c1 = *str++;
                c2 = *expr++;
                if (c1 != c2) return 0;
            }
        }
        return !*str;
    }


    void compile_and_upload_dir(const string &resultformat, vector<YMicroPython*> *list, const string &src_dir, const string &base_path, const string &patern)
    {
#ifdef WINDOWS_API
        WIN32_FIND_DATAA ffd;
        HANDLE hFind;
#else
        struct dirent *pDirent;
        DIR *pDir;
#endif

#ifdef WINDOWS_API
#else

        pDir = opendir(src_dir.c_str());
        if (pDir == NULL) {
            compile_and_upload_file(resultformat, list, src_dir, base_path, patern);
            return;
        }
#endif

        string working_dir = src_dir;
        char lastChar = working_dir[working_dir.length() - 1];
        if (lastChar != '/' && lastChar != '\\') {
#ifdef WINDOWS_API
            working_dir += "\\";
#else
            working_dir += "/";
#endif
        }


#ifdef WINDOWS_API
        // Find the first file in the directory.
        string mask = working_dir + "*";
        hFind = FindFirstFileA(mask.c_str(), &ffd);
        if (INVALID_HANDLE_VALUE == hFind) {
            compile_and_upload_file(resultformat, list, src_dir, base_path, patern);
            return;
        }
        do {
            char *name = ffd.cFileName;
#else
    while ((pDirent = readdir(pDir)) != NULL) {
        char *name = pDirent->d_name;
        struct stat buf;
#endif
            int isdir;
            if (*name == '.')
                continue;

            string entry_name(name);
            string sub_entry_path = working_dir + entry_name;
#ifdef WINDOWS_API
            isdir = ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY;
#else
        stat(sub_entry_path.c_str(), &buf);
        isdir = S_ISDIR(buf.st_mode);
#endif
            if (isdir) {
                compile_and_upload_dir(resultformat, list, sub_entry_path, base_path + entry_name + "/", patern);
            } else {
                compile_and_upload_file(resultformat, list, sub_entry_path, base_path, patern);
            }

#ifdef WINDOWS_API
        } while (FindNextFileA(hFind, &ffd) != 0);
#else
    }
    closedir(pDir);
#endif
    }


    bool isDirectory(const std::string &path)
    {
#ifdef WINDOWS_API
        DWORD fileAttributes = GetFileAttributesA(path.c_str());
        if (fileAttributes == INVALID_FILE_ATTRIBUTES) {
            return false;
        }
        return (fileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
#else
        struct stat buf;
        stat(path.c_str(), &buf);
        return  S_ISDIR(buf.st_mode);
#endif
    }

    void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*> *args, vector<SwitchDesc*> *switches)
    {
        vector<YMicroPython*> *list = enumerateTargets<YMicroPython>(_function, target, modulelist);
        string path = (*args)[0]->get_stringValue();
        string patern = (*args)[2]->get_stringValue();
        if (patern == "") {
            patern = "*.py";
        }
        string base_path = (*args)[1]->get_stringValue();
        if (base_path.length() == 0 || base_path[base_path.length() - 1] != '/') {
            base_path += "/";
        }
        compile_and_upload_dir(resultformat, list, path, base_path, patern);
    }
};


#endif

static void RegisterMpyCrossCommands(YMicroPythonCmdLine *function, vector<YapiCommand*> *cmdList)
{
#ifdef WITH_MPY_CROSS
    cmdList->push_back((YapiCommand*)(new MicroPython_compileFile(function)));
    cmdList->push_back((YapiCommand*)(new MicroPython_uploadFile(function)));
    cmdList->push_back((YapiCommand*)(new MicroPython_compileAndUpload(function)));
#endif
}

//--- (generated code: YMicroPython implementation)
bool   YMicroPythonCmdLine::isModule(void)      { return false; }
string YMicroPythonCmdLine::YFunctionName(void) { return "YMicroPython"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of lastMsg is string (Text)
// type of heapUsage is int (Percent)
// type of heapFrag is int (Percent)
// type of xheapUsage is int (Percent)
// type of stackUsage is int (Percent)
// type of currentScript is string (Text)
// type of startupScript is string (Text)
// type of startupDelay is double (MeasureVal)
// type of debugMode is int (OnOff)
// type of command is string (Text)
/**
 * Returns the logical name of the MicroPython interpreter.
 *
 * @return a string corresponding to the logical name of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.LOGICALNAME_INVALID.
 */
class MicroPython_get_logicalName : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the MicroPython interpreter.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the MicroPython interpreter. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the MicroPython interpreter
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the MicroPython interpreter.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the MicroPython interpreter", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the MicroPython interpreter.
 *
 * @return a string corresponding to a short string representing the current state of the MicroPython interpreter
 *
 * On failure, throws an exception or returns YMicroPython.ADVERTISEDVALUE_INVALID.
 */
class MicroPython_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the MicroPython interpreter.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the last message produced by a python script.
 *
 * @return a string corresponding to the last message produced by a python script
 *
 * On failure, throws an exception or returns YMicroPython.LASTMSG_INVALID.
 */
class MicroPython_get_lastMsg : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_lastMsg(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastMsg";
  }

  string getDescription()
  {
    return "Returns the last message produced by a python script.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_lastMsg();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the percentage of MicroPython main memory in use,
 * as observed at the end of the last garbage collection.
 *
 * @return an integer corresponding to the percentage of MicroPython main memory in use,
 *         as observed at the end of the last garbage collection
 *
 * On failure, throws an exception or returns YMicroPython.HEAPUSAGE_INVALID.
 */
class MicroPython_get_heapUsage : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_heapUsage(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_heapUsage";
  }

  string getDescription()
  {
    return "Returns the percentage of MicroPython main memory in use,as observed at the end of the last garbage collection.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_heapUsage();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the fragmentation ratio of MicroPython main memory,
 * as observed at the end of the last garbage collection.
 *
 * @return an integer corresponding to the fragmentation ratio of MicroPython main memory,
 *         as observed at the end of the last garbage collection
 *
 * On failure, throws an exception or returns YMicroPython.HEAPFRAG_INVALID.
 */
class MicroPython_get_heapFrag : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_heapFrag(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_heapFrag";
  }

  string getDescription()
  {
    return "Returns the fragmentation ratio of MicroPython main memory,as observed at the end of the last garbage collection.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_heapFrag();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the percentage of MicroPython external memory in use,
 * as observed at the end of the last garbage collection.
 *
 * @return an integer corresponding to the percentage of MicroPython external memory in use,
 *         as observed at the end of the last garbage collection
 *
 * On failure, throws an exception or returns YMicroPython.XHEAPUSAGE_INVALID.
 */
class MicroPython_get_xheapUsage : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_xheapUsage(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_xheapUsage";
  }

  string getDescription()
  {
    return "Returns the percentage of MicroPython external memory in use,as observed at the end of the last garbage collection.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_xheapUsage();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the maximum percentage of MicroPython call stack in use,
 * as observed at the end of the last garbage collection.
 *
 * @return an integer corresponding to the maximum percentage of MicroPython call stack in use,
 *         as observed at the end of the last garbage collection
 *
 * On failure, throws an exception or returns YMicroPython.STACKUSAGE_INVALID.
 */
class MicroPython_get_stackUsage : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_stackUsage(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_stackUsage";
  }

  string getDescription()
  {
    return "Returns the maximum percentage of MicroPython call stack in use,as observed at the end of the last garbage collection.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_stackUsage();
        char buf[32]; SAFE_SPRINTF(buf, 32,"%u%%", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of currently active script, if any.
 *
 * @return a string corresponding to the name of currently active script, if any
 *
 * On failure, throws an exception or returns YMicroPython.CURRENTSCRIPT_INVALID.
 */
class MicroPython_get_currentScript : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_currentScript(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_currentScript";
  }

  string getDescription()
  {
    return "Returns the name of currently active script, if any.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_currentScript();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Stops current running script, and/or selects a script to run immediately in a
 * fresh new environment. If the MicroPython interpreter is busy running a script,
 * this function will abort it immediately and reset the execution environment.
 * If a non-empty string is given as argument, the new script will be started.
 *
 * @param newval : a string
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_currentScript : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_currentScript(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_currentScript";
  }

  string getDescription()
  {
    return "Stops current running script, and/or selects a script to run immediately in afresh new environment.";
  }

  string getMoreInfo()
  {
    return "If the MicroPython interpreter is busy running a script, this function will abort it immediately and reset the execution environment. If a non-empty string is given as argument, the new script will be started.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_currentScript(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the name of the script to run when the device is powered on.
 *
 * @return a string corresponding to the name of the script to run when the device is powered on
 *
 * On failure, throws an exception or returns YMicroPython.STARTUPSCRIPT_INVALID.
 */
class MicroPython_get_startupScript : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_startupScript(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_startupScript";
  }

  string getDescription()
  {
    return "Returns the name of the script to run when the device is powered on.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_startupScript();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the script to run when the device is powered on.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the script to run when the device is powered on
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_startupScript : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_startupScript(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_startupScript";
  }

  string getDescription()
  {
    return "Changes the script to run when the device is powered on.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the script to run when the device is powered on", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_startupScript(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Changes the wait time before running the startup script on power on, between 0.1
 * second and 25 seconds. Remember to call the saveToFlash() method of the
 * module if the modification must be kept.
 *
 * @param newval : a floating point number corresponding to the wait time before running the startup
 * script on power on, between 0.1
 *         second and 25 seconds
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_startupDelay : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_startupDelay(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_startupDelay";
  }

  string getDescription()
  {
    return "Changes the wait time before running the startup script on power on, between 0.";
  }

  string getMoreInfo()
  {
    return "1 second and 25 seconds. Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(DOUBLE_ARG, "newval", "a floating point number corresponding to the wait time before running the startup script on power on, between 0.1 second and 25 seconds", "_DOUBLE", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_startupDelay(double((*args)[0]->get_doubleValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the wait time before running the startup script on power on,
 * measured in seconds.
 *
 * @return a floating point number corresponding to the wait time before running the startup script on power on,
 *         measured in seconds
 *
 * On failure, throws an exception or returns YMicroPython.STARTUPDELAY_INVALID.
 */
class MicroPython_get_startupDelay : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_startupDelay(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_startupDelay";
  }

  string getDescription()
  {
    return "Returns the wait time before running the startup script on power on,measured in seconds.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        double attrval = (*list)[i]->get_startupDelay();
        char buf[32]; SAFE_SPRINTF(buf,32,"%.3f",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the activation state of MicroPython debugging interface.
 *
 * @return either YMicroPython.DEBUGMODE_OFF or YMicroPython.DEBUGMODE_ON, according to the activation
 * state of MicroPython debugging interface
 *
 * On failure, throws an exception or returns YMicroPython.DEBUGMODE_INVALID.
 */
class MicroPython_get_debugMode : public YapiCommand /* arguments: */
{
public:
  MicroPython_get_debugMode(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_debugMode";
  }

  string getDescription()
  {
    return "Returns the activation state of MicroPython debugging interface.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_debugMode();
        string strval =  enumString(attrval,enumOnOff);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the activation state of MicroPython debugging interface.
 *
 * @param newval : either YMicroPython.DEBUGMODE_OFF or YMicroPython.DEBUGMODE_ON, according to the
 * activation state of MicroPython debugging interface
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class MicroPython_set_debugMode : public YapiCommand /* arguments: newval */
{
public:
  MicroPython_set_debugMode(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_debugMode";
  }

  string getDescription()
  {
    return "Changes the activation state of MicroPython debugging interface.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either OFF or ON, according to the activation state of MicroPython debugging interface", "OFF=0,ON=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_debugMode(Y_DEBUGMODE_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackMicroPython is YMicroPythonValueCallback* (YMicroPythonValueCallback)
// type of logCallback is YMicroPythonLogCallback* (YMicroPythonLogCallback)
// type of isFirstCb is bool (bool)
// type of prevCbPos is int (int)
// type of logPos is int (int)
// type of prevPartialLog is string (str)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_MicroPython_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_MicroPython_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Submit MicroPython code for execution in the interpreter.
 * If the MicroPython interpreter is busy, this function will
 * block until it becomes available. The code is then uploaded,
 * compiled and executed on the fly, without beeing stored on the device filesystem.
 *
 * There is no implicit reset of the MicroPython interpreter with
 * this function. Use method reset() if you need to start
 * from a fresh environment to run your code.
 *
 * Note that although MicroPython is mostly compatible with recent Python 3.x
 * interpreters, the limited ressources on the device impose some restrictions,
 * in particular regarding the libraries that can be used. Please refer to
 * the documentation for more details.
 *
 * @param codeName : name of the code file (used for error reporting only)
 * @param mpyCode : MicroPython code to compile and execute
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_eval : public YapiCommand /* arguments: codeName mpyCode */
{
public:
  apifun_MicroPython_eval(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "eval";
  }

  string getDescription()
  {
    return "Submit MicroPython code for execution in the interpreter.";
  }

  string getMoreInfo()
  {
    return "If the MicroPython interpreter is busy, this function will block until it becomes available. The code is then uploaded, compiled and executed on the fly, without beeing stored on the device filesystem.\nThere is no implicit reset of the MicroPython interpreter with this function. Use method reset() if you need to start from a fresh environment to run your code.\nNote that although MicroPython is mostly compatible with recent Python 3.x interpreters, the limited ressources on the device impose some restrictions, in particular regarding the libraries that can be used. Please refer to the documentation for more details.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "codeName", "name of the code file (used for error reporting only)", "_STRING", false));
    res->push_back(new ArgumentDesc(STRING_ARG, "mpyCode", "MicroPython code to compile and execute", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->eval((*args)[0]->get_stringValue(),
                         (*args)[1]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Stops current execution, and reset the MicroPython interpreter to initial state.
 * All global variables are cleared, and all imports are forgotten.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_reset : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_reset(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "reset";
  }

  string getDescription()
  {
    return "Stops current execution, and reset the MicroPython interpreter to initial state.";
  }

  string getMoreInfo()
  {
    return "All global variables are cleared, and all imports are forgotten.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->reset();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Clears MicroPython interpreter console log buffer.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_MicroPython_clearLogs : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_clearLogs(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "clearLogs";
  }

  string getDescription()
  {
    return "Clears MicroPython interpreter console log buffer.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->clearLogs();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns a string with last logs of the MicroPython interpreter.
 * This method return only logs that are still in the module.
 *
 * @return a string with last MicroPython logs.
 *         On failure, throws an exception or returns  YAPI_INVALID_STRING.
 */
class apifun_MicroPython_get_lastLogs : public YapiCommand /* arguments: */
{
public:
  apifun_MicroPython_get_lastLogs(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastLogs";
  }

  string getDescription()
  {
    return "Returns a string with last logs of the MicroPython interpreter.";
  }

  string getMoreInfo()
  {
    return "This method return only logs that are still in the module.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YMicroPython*>* list = enumerateTargets<YMicroPython>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_lastLogs();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of generated code: YMicroPython implementation)

//--- (generated code: YMicroPython functions)
void YMicroPythonCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YMicroPython>(this, cmdList);
    RegisterMpyCrossCommands(this, cmdList);
    cmdList->push_back((YapiCommand*) (new MicroPython_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_lastMsg(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_heapUsage(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_heapFrag(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_xheapUsage(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_stackUsage(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_currentScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_currentScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_startupScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_startupScript(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_startupDelay(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_startupDelay(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_get_debugMode(this)));
    cmdList->push_back((YapiCommand*) (new MicroPython_set_debugMode(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_eval(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_reset(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_clearLogs(this)));
    cmdList->push_back((YapiCommand*) (new apifun_MicroPython_get_lastLogs(this)));
  }

//--- (end of generated code: YMicroPython functions)
