﻿
//  Windows service used to send mail alerts when temperature raises
//  above a preset limit.
//  more info on
//  http://www.yoctopuce.com/EN/article/temperature-warnings-under-windows


//  Service Code structure inspired from John Storer's example code
//  http://www.codeproject.com/Articles/14353/Creating-a-Basic-Windows-Service-in-C

using System;
using System.Diagnostics;
using System.ServiceProcess;
using System.ServiceModel;
using System.ServiceModel.Description;
using Microsoft.Win32;
using System.Threading;


namespace WindowsService
{
    class WindowsService : ServiceBase
    {
        /// <summary>
        /// Public Constructor for WindowsService.
        /// - Put all of your Initialization code here.
        /// </summary>
        /// 
        static string serviceDisplayName = "Thermal Control";
        static string serviceName = "ThermalControl";

        static ServiceHost selfHost = null;

        Thread PayLoadThread;  // this is the thread containing the "active code"

        public WindowsService()
        {
            this.ServiceName = serviceName;
            this.EventLog.Log = "Application";
            this.EventLog.Source = serviceDisplayName;

            // These Flags set whether or not to handle that specific
            //  type of event. Set to true if you need it, false otherwise.
            this.CanHandlePowerEvent = true;
            this.CanHandleSessionChangeEvent = true;
            this.CanPauseAndContinue = true;
            this.CanShutdown = true;
            this.CanStop = true;
        }


        //This is your service entry point
        static void Main(string[] args)
        {
            string execPath = System.Reflection.Assembly.GetEntryAssembly().Location;
            string execFile = System.AppDomain.CurrentDomain.FriendlyName;

            if (args.Length > 0)
            {
                switch (args[0].ToUpper())
                {
                    case "/RUN":
                        staticRun();
                        break;
                    
                    case "/INSTALL":
                        if (WindowsServiceControl.ServiceIsInstalled(serviceName))
                            Console.WriteLine("Service " + serviceName + " is already installed");
                        else
                        {
                            WindowsServiceControl.InstallAndStart(serviceName, serviceDisplayName, execPath);
                            Console.WriteLine(serviceDisplayName + " installed and started");
                        }
                        break;

                    case "/UNINSTALL":
                        if (WindowsServiceControl.ServiceIsInstalled(serviceName))
                        {
                            WindowsServiceControl.Uninstall(serviceName);
                            Console.WriteLine(serviceDisplayName + " uninstalled");
                        }
                        else Console.WriteLine("Service " + serviceName + " is not installed");
                        break;

                    case "/START":
                        if (!WindowsServiceControl.ServiceIsInstalled(serviceName))
                            Console.WriteLine("Service " + serviceName + " is not installed");
                        else
                        {
                            WindowsServiceControl.StartService(serviceName);
                            Console.WriteLine(serviceDisplayName + " started");
                        }
                        break;

                    case "/STOP":
                        if (!WindowsServiceControl.ServiceIsInstalled(serviceName))
                            Console.WriteLine("Service " + serviceName + " is not installed");
                        else
                        {
                            WindowsServiceControl.StopService(serviceName);
                            Console.WriteLine(serviceDisplayName + " stopped");
                        }
                        break;

                    default:
                        Console.WriteLine(execFile + " is meant to run as a Windows service, you need to install it, then it will");
                        Console.WriteLine("start automatically each time your computer starts.");
                        Console.WriteLine("If you just want to debug it, make it run with the /RUN parameter");
                        
                        Console.WriteLine("Usage:");
                        Console.WriteLine(execFile + " [/INSTALL | /START | /STOP | /UNINSTALL | /RUN");

                        break;
                }

            }
            else
                try
                {
                    ServiceBase.Run(new WindowsService());
                } 
                catch (Exception e)
                    { Console.WriteLine(e.Message);
                    }
        }

        /// <summary>
        /// Dispose of objects that need it here.
        /// </summary>
        /// <param name="disposing">Whether
        ///    or not disposing is going on.</param>
        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
        }

        /// <summary>
        /// OnStart(): Put startup code here
        ///  - Start threads, get inital data, etc.
        /// </summary>
        /// <param name="args"></param>
        protected override void OnStart(string[] args)
        {
            base.OnStart(args);
            PayLoadThread = new Thread(new ThreadStart(Payload));
            PayLoadThread.Start();
        }

        /// <summary>
        /// OnStop(): Put your stop code here
        /// - Stop threads, set final data, etc.
        /// </summary>
        protected override void OnStop()
        {
            base.OnStop();
            if (PayLoadThread.IsAlive) PayLoadThread.Abort();
            this.EventLog.WriteEntry("Thermal Control Thread Stopped");
        }

        /// <summary>
        /// starts the WCF communication channel
        /// </summary>   

        static protected bool StartCommunicationChannel(ref string errmsg, ref bool fatal)
        {
            Uri baseAddress = new Uri(ThermalControlService.controlURI);

            // Step 2 of the hosting procedure: Create ServiceHost
            selfHost = new ServiceHost(typeof(ThermalControlService), baseAddress);

          
           

            try
            {   // Step 3 of the hosting procedure: Add a service endpoint.
                selfHost.AddServiceEndpoint(
                    typeof(IThermalControl),
                    new WSHttpBinding(),
                    "ThermalControlService");


                // Step 4 of the hosting procedure: Enable metadata exchange.
                ServiceMetadataBehavior smb = new ServiceMetadataBehavior();
                smb.HttpGetEnabled = true;
                selfHost.Description.Behaviors.Add(smb);

                // Step 5 of the hosting procedure: Start (and then stop) the service.
                selfHost.Open();

               
            }
            catch (CommunicationException ce)
            {  // oops something went wrong
                selfHost.Abort();
                errmsg = "Unable to create WCF communication channel" + ce.Message;
                fatal = true;
                return false;
            }
            return true;

        }


        /// <summary>
        /// stops the WCF communication channel
        /// </summary>   

        static protected void StopCommunicationChannel()
        {
            selfHost.Close();
        }



        /// <summary>
        /// OnShutdown(): Called when the System is shutting down
        /// - Put code here when you need special handling
        ///   of code that deals with a system shutdown, such
        ///   as saving special data before shutdown.
        /// </summary>
        protected override void OnShutdown()
        {
            if (PayLoadThread.IsAlive) PayLoadThread.Abort();
            base.OnShutdown();
        }


        /// <summary>
        /// static run, for debug purpose
        /// </summary>
        static protected void staticRun()
        {   string errmsg = "";
            bool fatal = false;

            
            Console.WriteLine("Thermal Control running (debug mode)");

            // init thermal controller
            if (!ThermalController.init(true,ref errmsg, ref fatal))
                Console.WriteLine("errmsg");
         
            // start setup-UI communication channel
            if (!fatal)
              if (!StartCommunicationChannel(ref errmsg, ref fatal))
                Console.WriteLine("InitFailed: " + errmsg);

            if (!fatal)
                while (true)
                {
                    try
                    {
                        if (!ThermalController.Check4Temp(ref errmsg))
                            Console.WriteLine("failure: " + errmsg, EventLogEntryType.Error);
                    }
                    catch (Exception e)
                    {
                        Console.WriteLine("Exception: " + e.Message);
                    }
                    Thread.Sleep(5000);  // debug mode checks temp every 5 seconds  
                }
        }

        // this is where the "real" stuff is happening
        protected void Payload()
        {
            string errmsg = "";
            bool fatal = false;



           
            this.EventLog.WriteEntry("Thermal Control Thread Starting");
            ThermalController ctrl = new ThermalController();

            // init thermal controller
            if (!ThermalController.init(false,ref errmsg, ref fatal))
                this.EventLog.WriteEntry("InitFailed: " + errmsg, fatal ? EventLogEntryType.Error : EventLogEntryType.Warning);

            // start setup-UI communication channel
            if (!fatal)
             if (!StartCommunicationChannel(ref errmsg, ref fatal))
                this.EventLog.WriteEntry("InitFailed: " + errmsg, fatal ? EventLogEntryType.Error : EventLogEntryType.Warning);

            //  and make in run from time to time
            if (!fatal)
            {
                while (Thread.CurrentThread.IsAlive)
                {
                    try  // this is a network based feature, a lot of bad things can happen
                    {
                        if (!ThermalController.Check4Temp(ref errmsg))
                            this.EventLog.WriteEntry("failure: " + errmsg, EventLogEntryType.Error);
                    }
                    catch (Exception e)
                    {
                        this.EventLog.WriteEntry("Exception: " + e.Message, EventLogEntryType.Warning);
                    }
                    Thread.Sleep(15000);  // checks temp every 15 sec  

                }

                // start setup-UI communication
                StopCommunicationChannel();
            }

        }
    }
}