#!/usr/bin/python
# -*- coding: utf-8 -*-

# some classes to drive  HT16K33 based LED  displays with a Yocto-I2C

import os, sys
from yocto_api import *
from yocto_i2cport import *

class HT16K33:
  i2cPort = None

  def __init__(self, I2C_port, addr):  # YI2cport function and A2A1A0 (0..7)
    self.addr = (0x70 | ( addr & 3)) << 1
    self.addrStr = format(self.addr, '02x')
    self.i2cPort = I2C_port
    self.i2cPort.set_i2cMode("400kbps")
    self.i2cPort.set_i2cVoltageLevel(YI2cPort.I2CVOLTAGELEVEL_3V3)
    self.i2cPort.reset()
    self.i2cPort.writeLine("{S}" + self.addrStr + "21{P}") # wake up
    self.i2cPort.writeLine("{S}" + self.addrStr + "A0{P}") # INT/ROW output pin is set to ROW driver output
    self.i2cPort.writeLine("{S}" + self.addrStr + "81{P}") # display ON , no blinking
    self.i2cPort.writeLine("{S}" + self.addrStr + "E8{P}") # intensity = 50%

  def setDimming(self, dimming):
    if dimming < 0: return;
    if dimming == 0:
      self.i2cPort.writeLine("{S}" + self.addrStr + "80{P}") # display OFF
    elif dimming < 16:
      self.i2cPort.writeLine("{S}" + self.addrStr + format(0xE0 | dimming, '02x') + "{P}") # dimming value
      self.i2cPort.writeLine("{S}" + self.addrStr + "81[P]") # display ON

  def setBlinking(self, blinking):
    if (blinking < 0 or blinking > 3): return;
    self.i2cPort.writeLine("{S}" + self.addrStr + format(0x81 | ( blinking << 1), '02x') + "{P}") # blinking value

  def sendData(self, rawdata):
    cmd = '{S}' + format(self.addr, '02x') + "00";
    for i in range(0, len(rawdata)):
      cmd = cmd + format(rawdata[i], '02x')
    cmd += "{P}"
    self.i2cPort.writeLine(cmd)


class HT16K33_8x8(HT16K33):
  def showBitmap(self, bitmap):
    data = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
    for i in range(0, 8):
      line = bitmap[i]
      data[i * 2] = ((line & 1) << 7) | ((line & 0xFE) >> 1)
    self.sendData(data)


class HT16K33_7seg(HT16K33):
  digits = [0x3F, 0x06, 0x5B, 0x4F, 0x66, 0x6D, 0x7D, 0x07, 0x7F, 0x6F]

  def showInteger(self, number):
    number = round(number)
    if number < -999:  raise Exception("min value is -999")
    if number > 9999:  raise Exception("max value is 9999")
    data = [0] * 10  #array of 10 zeros
    number = str(number)
    index = 8
    for i in range(0, len(number)):
      if number[-i - 1] == "-":  # negative sign
        data[index] = 64
      else:
        data[index] = self.digits[ord(number[-i - 1]) - 48];
      index = index - 2
      if index == 4: index = 2 # skip middle colon
    self.sendData(data)

  def charToDigit(self, string, n):
    char = string[n]
    if char == " ": return 0;
    if char == "-": return 64;
    if 48 <= ord(char) < 58: return self.digits[ord(char) - 48]
    raise Exception("Character \"" + char + "\" at position #" + str(
      n + 1) + " of string \"" + string + "\" cannot be displayed ");

  def showString(self, number):
    data = [0] * 10 #array of 10 zeros
    col = number.find(":")
    if col > 0:
      data[4] = 2 # center colon
      #first two digit
      n = col - 1
      index = 2
      while index >= 0:
        decimal = 0
        digit = 0
        if n >= 0 and number[n] == ".":
          decimal = 128
          n = n - 1
        if (n >= 0):
          digit = self.charToDigit(number, n)
          n = n - 1
        data[index] = digit | decimal
        index = index - 2
      if (n >= 0):  raise Exception("string " + number + " cannot be displayed ");
      n = col + 1
      index = 6
      while index < 10:
        decimal = 0
        digit = 0
        if n < len(number):
          digit = self.charToDigit(number, n)
          n = n + 1
        if n < len(number) and number[n] == ".":
          decimal = 128
          n = n + 1
        data[index] = digit | decimal
        index = index + 2
      if (n < len(number)):  raise Exception("string " + number + " cannot be displayed ");
    else:
      index = 8
      n = len(number) - 1
      while (n >= 0) and (index >= 0):
        decimal = 0
        digit = 0
        if number[n] == '.':
          decimal = 128
          n = n - 1
        if n >= 0:
          digit = self.charToDigit(number, n)
          n = n - 1
        data[index] = digit | decimal
        index = index - 2
        if index == 4: index = 2;
      if (n > 0):    raise Exception("string " + number + " cannot be displayed ");

    self.sendData(data)


class HT16K33_14seg(HT16K33):
  # representation for ASCII characters 32..127
  font = [0x0000, 0x0006, 0x0220, 0x12CE, 0x12ED, 0x0C24, 0x235D, 0x0400,
          0x2400, 0x0900, 0x3FC0, 0x12C0, 0x0800, 0x00C0, 0x0000, 0x0C00,
          0x0C3F, 0x0006, 0x00DB, 0x008F, 0x00E6, 0x2069, 0x00FD, 0x0007,
          0x00FF, 0x00EF, 0x1200, 0x0A00, 0x2400, 0x00C8, 0x0900, 0x1083,
          0x02BB, 0x00F7, 0x128F, 0x0039, 0x120F, 0x00F9, 0x0071, 0x00BD,
          0x00F6, 0x1200, 0x001E, 0x2470, 0x0038, 0x0536, 0x2136, 0x003F,
          0x00F3, 0x203F, 0x20F3, 0x00ED, 0x1201, 0x003E, 0x0C30, 0x2836,
          0x2D00, 0x1500, 0x0C09, 0x0039, 0x2100, 0x000F, 0x0C03, 0x0008,
          0x0100, 0x1058, 0x2078, 0x00D8, 0x088E, 0x0858, 0x0071, 0x048E,
          0x1070, 0x1000, 0x000E, 0x3600, 0x0030, 0x10D4, 0x1050, 0x00DC,
          0x0170, 0x0486, 0x0050, 0x2088, 0x0078, 0x001C, 0x2004, 0x2814,
          0x28C0, 0x200C, 0x0848, 0x0949, 0x1200, 0x2489, 0x0520, 0x3FFF]

  def showString(self, string):
    data = [0, 0, 0, 0, 0, 0, 0, 0]
    n = 0
    index = 0
    while n < len(string):
      if index >= 8:
        raise Exception("String \"" + string + "\"  cannot be displayed ")
      code = ord(string[n])
      if code < 32 or code > 127:  raise \
        Exception("Character \"" + string[n] + "\" at position #" + str(
            n + 1) + " of string \"" + string + "\" cannot be displayed ");
      digit = 0
      if code == 46: digit = 0x4000  # decimal point
      digit = digit | self.font[code - 32]
      n = n + 1
      if code != 46 and n < len(string) and string[n] == ".":
        digit = digit | 0x4000
        n = n + 1
      data[index] = digit & 0xff
      data[index + 1] = digit >> 8
      index = index + 2
    self.sendData(data)

class HT16K33_8x8(HT16K33):
   def showBitmap(self, bitmap):  ### array of 8 bytes
     data = [0] * 16  #array of 10 zeros
     for i in range(0,8):
       b = bitmap[i] &0xff
       b = (b & 0xF0) >> 4 | (b & 0x0F) << 4
       b = (b & 0xCC) >> 2 | (b & 0x33) << 2
       b = (b & 0xAA) >> 1 | (b & 0x55) << 1
       data[i*2] = (b >>1) | (b & 0x01)<<7
     self.sendData(data)

class HT16K33_16x8(HT16K33):
   def showBitmap(self, bitmap):  ### array of 8 bytes
     data = [0] * 16  #array of 10 zeros
     for x in range (0,16):
       index =  ((x & 7) <<1) | (x & 8) >> 3
       for y in range(0,8):
         bit = (bitmap[7-y] >> (15-x)) &1
         data[index] |= bit<<y
     self.sendData(data)


#setup API
errmsg = YRefParam()
if YAPI.RegisterHub("usb", errmsg) != YAPI.SUCCESS:
  sys.exit("init error" + errmsg.value)

#find I2C port
i2cPort = YI2cPort.FirstI2cPort()
if i2cPort is None:
  sys.exit('No Yocto-I2C found (check cable)')

# uncomment the section matching your display

#### seven segments   (www.adafruit.com/product/877)
##display=HT16K33_7seg(i2cPort,0)
##display.showString("124.4")

#### 14 segments (www.adafruit.com/product/1910)
##display= HT16K33_14seg(i2cPort, 0)
##display.showString("ABC.D")

#### 8x8 matrix (www.adafruit.com/product/1048)
#display= HT16K33_8x8(i2cPort, 0)
#display.showBitmap([int('01111110', 2),
#                    int('10000001', 2),
#                    int('10100101', 2),
#                    int('10000001', 2),
#                    int('10100101', 2),
#                    int('10011001', 2),
#                    int('10000001', 2),
#                    int('01111110', 2)])

#### 16x8 matrix (www.adafruit.com/product/3151)
#display= HT16K33_16x8(i2cPort, 0)
#display.showBitmap([int('0000000011000000', 2),
#                    int('0000000010100000', 2),
#                    int('0011111110010000', 2),
#                    int('0010000000001000', 2),
#                    int('0011111110010000', 2),
#                    int('0000000010100000', 2),
#                    int('0000000011000000', 2),
#                    int('0000000000000000', 2)])
#display.setBlinking(2)


YAPI.FreeAPI()

