import * as PAKO from "./Pakofull.js";
import * as UTILS from "./utils/utilsFull.js";

/* Public constants ==========================================================*/
/* ===========================================================================*/

/* ===========================================================================*/

/**
 * class Deflate
 *
 * Generic JS-style wrapper for zlib calls. If you don't need
 * streaming behaviour - use more simple functions: [[deflate]],
 * [[deflateRaw]] and [[gzip]].
 **/

/* internal
 * Deflate.chunks -> Array
 *
 * Chunks of output data, if [[Deflate#onData]] not overridden.
 **/

/**
 * Deflate.result -> Uint8Array
 *
 * Compressed result, generated by default [[Deflate#onData]]
 * and [[Deflate#onEnd]] handlers. Filled after you push last chunk
 * (call [[Deflate#push]] with `Z_FINISH` / `true` param).
 **/

/**
 * Deflate.err -> Number
 *
 * Error code after deflate finished. 0 (Z_OK) on success.
 * You will not need it in real life, because deflate errors
 * are possible only on wrong options or bad `onData` / `onEnd`
 * custom handlers.
 **/

/**
 * Deflate.msg -> String
 *
 * Error message, if [[Deflate.err]] != 0
 **/

/**
 * new Deflate(options)
 * - options (Object): zlib deflate options.
 *
 * Creates new deflator instance with specified params. Throws exception
 * on bad params. Supported options:
 *
 * - `level`
 * - `windowBits`
 * - `memLevel`
 * - `strategy`
 * - `dictionary`
 *
 * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
 * for more information on these.
 *
 * Additional options, for internal needs:
 *
 * - `chunkSize` - size of generated data chunks (16K by default)
 * - `raw` (Boolean) - do raw deflate
 * - `gzip` (Boolean) - create gzip wrapper
 * - `header` (Object) - custom header for gzip
 *   - `text` (Boolean) - true if compressed data believed to be text
 *   - `time` (Number) - modification time, unix timestamp
 *   - `os` (Number) - operation system code
 *   - `extra` (Array) - array of bytes with extra data (max 65536)
 *   - `name` (String) - file name (binary string)
 *   - `comment` (String) - comment (binary string)
 *   - `hcrc` (Boolean) - true if header crc should be added
 *
 * ##### Example:
 *
 * ```javascript
 * const pako = require('pako')
 *   , chunk1 = new Uint8Array([1,2,3,4,5,6,7,8,9])
 *   , chunk2 = new Uint8Array([10,11,12,13,14,15,16,17,18,19]);
 *
 * const deflate = new pako.Deflate({ level: 3});
 *
 * deflate.push(chunk1, false);
 * deflate.push(chunk2, true);  // true -> last chunk
 *
 * if (deflate.err) { throw new Error(deflate.err); }
 *
 * console.log(deflate.result);
 * ```
 **/

export class Pako_deflate_option
{
    public level: number = PAKO.zlib_constants.Z_DEFAULT_COMPRESSION;
    public method: number = PAKO.zlib_constants.Z_DEFLATED;
    public chunkSize: number = 16384;
    public windowBits: number = 15;
    public memLevel: number = 8;
    public strategy: number = PAKO.zlib_constants.Z_DEFAULT_STRATEGY
    public raw: boolean = false;
    public gzip: boolean = false;
    public header: PAKO.zlib_GZheader | null = null
    public dictionary: Uint8Array | null = null;
    public ignore_os: boolean = false;

    constructor(CustOption?: {})
    {
        Pako_deflate_option.ApplyCustomOptions(this, CustOption)

    }
    public static ApplyCustomOptions(StdOption: Pako_deflate_option, CustOption?: {})
    {
        if (typeof CustOption != "undefined")
        {
            let stdKeys: string[] = Object.keys(StdOption);
            let custKeys: string[] = Object.keys(CustOption);
            for (let i: number = 0; i < custKeys.length; i++)
            {
                if (stdKeys.indexOf(custKeys[i]) < 0)
                {
                    debugger
                    throw "Invalid Pako option name '" + custKeys[i] + "', check Pako_deflate_option class";
                }
                let srcType: string = typeof (Reflect.get(CustOption, custKeys[i]));
                let trgtType: string = typeof (Reflect.get(StdOption, custKeys[i]));
                if (srcType != trgtType) throw  "Invalid Pako type for option '" + custKeys[i] + "' (expected '" + trgtType + "', got '" + srcType + "')";
                Reflect.set(StdOption, custKeys[i], Reflect.get(CustOption, custKeys[i]));
            }
        }
    }
}

export class Pako_Deflate
{

    private options: Pako_deflate_option
    private err: number;      // error code, if happens (0 = Z_OK)
    private msg: string;     // error message
    private ended: boolean;  // used to avoid multiple onEnd() calls
    private chunks: Uint8Array[];
    private strm: PAKO.zlib_ZStream;
    private _dict_set: boolean = false;
    private result: Uint8Array | null = null;

    constructor(options?: {})
    {
        this.options = new Pako_deflate_option(options);

        /*
         this.options = utils.assign({
           level: Z_DEFAULT_COMPRESSION,
           method: Z_DEFLATED,
          chunkSize: 16384,
          windowBits: 15,
          memLevel: 8,
          strategy: Z_DEFAULT_STRATEGY
        }, options || {});
        */

        let opt: Pako_deflate_option = this.options;

        if (opt.raw && (opt.windowBits > 0))
        {
            opt.windowBits = -opt.windowBits;
        }

        else if (opt.gzip && (opt.windowBits > 0) && (opt.windowBits < 16))
        {
            opt.windowBits += 16;
        }

        this.err = 0;      // error code, if happens (0 = Z_OK)
        this.msg = '';     // error message
        this.ended = false;  // used to avoid multiple onEnd() calls
        this.chunks = [];     // chunks of compressed data

        this.strm = new PAKO.zlib_ZStream();
        this.strm.avail_out = 0;

        let status = PAKO.zlib_deflate.deflateInit2(
            this.strm,
            opt.level,
            opt.method,
            opt.windowBits,
            opt.memLevel,
            opt.strategy
        );

        if (status !== PAKO.zlib_constants.Z_OK)
        {
            throw new Error(PAKO.zlib_messages.msg(status));
        }

        if (opt.header)
        {
            PAKO.zlib_deflate.deflateSetHeader(this.strm, opt.header);
        }

        if (opt.dictionary)
        {
            let dict: Uint8Array;
            // Convert data if needed
            if (typeof opt.dictionary === 'string')
            {
                // If we need to compress text, change encoding to utf8.
                dict = PAKO.Pako_strings.string2buf(opt.dictionary);
            }
            else if (toString.call(opt.dictionary) === '[object ArrayBuffer]')
            {
                dict = new Uint8Array(opt.dictionary);
            }
            else
            {
                dict = opt.dictionary;
            }

            status = PAKO.zlib_deflate.deflateSetDictionary(this.strm, dict);

            if (status !== PAKO.zlib_constants.Z_OK)
            {
                throw new Error(PAKO.zlib_messages.msg(status));
            }

            this._dict_set = true;
        }
    }

    /**
     * Deflate#push(data[, flush_mode]) -> Boolean
     * - data (Uint8Array|ArrayBuffer|String): input data. Strings will be
     *   converted to utf8 byte sequence.
     * - flush_mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE modes.
     *   See constants. Skipped or `false` means Z_NO_FLUSH, `true` means Z_FINISH.
     *
     * Sends input data to deflate pipe, generating [[Deflate#onData]] calls with
     * new compressed chunks. Returns `true` on success. The last data block must
     * have `flush_mode` Z_FINISH (or `true`). That will flush internal pending
     * buffers and call [[Deflate#onEnd]].
     *
     * On fail call [[Deflate#onEnd]] with error code and return false.
     *
     * ##### Example
     *
     * ```javascript
     * push(chunk, false); // push one of data chunks
     * ...
     * push(chunk, true);  // push last chunk
     * ```
     **/
    public push(data: Uint8Array | ArrayBuffer | String, flush_mode: Number | Boolean): boolean
    {
        const strm = this.strm;
        const chunkSize = this.options.chunkSize;
        let status: number;
        let _flush_mode: number;

        if (this.ended)
        { return false; }

        if (flush_mode === ~~flush_mode)
        {
            _flush_mode = <number>flush_mode;
        }
        else
        {
            _flush_mode = flush_mode === true ? PAKO.zlib_constants.Z_FINISH : PAKO.zlib_constants.Z_NO_FLUSH;
        }

        // Convert data if needed
        if (typeof data === 'string')
        {
            // If we need to compress text, change encoding to utf8.
            strm.input = PAKO.Pako_strings.string2buf(data);
        }
        else if (toString.call(data) === '[object ArrayBuffer]')
        {
            strm.input = new Uint8Array(<ArrayBuffer>data);
        }
        else
        {
            strm.input = data;
        }

        strm.next_in = 0;
        strm.avail_in = strm.input.length;

        for (; ;)
        {
            if (strm.avail_out === 0)
            {
                strm.output = new Uint8Array(chunkSize);
                strm.next_out = 0;
                strm.avail_out = chunkSize;
            }

            // Make sure avail_out > 6 to avoid repeating markers
            if ((_flush_mode === PAKO.zlib_constants.Z_SYNC_FLUSH || _flush_mode === PAKO.zlib_constants.Z_FULL_FLUSH) && strm.avail_out <= 6)
            {
                this.onData(strm.output.subarray(0, strm.next_out));
                strm.avail_out = 0;
                continue;
            }

            status = PAKO.zlib_deflate.deflate(strm, _flush_mode);

            // Ended => flush and finish
            if (status === PAKO.zlib_constants.Z_STREAM_END)
            {
                if (strm.next_out > 0)
                {
                    this.onData(strm.output.subarray(0, strm.next_out));
                }
                status = PAKO.zlib_deflate.deflateEnd(this.strm);
                this.onEnd(status);
                this.ended = true;
                return status === PAKO.zlib_constants.Z_OK;
            }

            // Flush if out buffer full
            if (strm.avail_out === 0)
            {
                this.onData(strm.output);
                continue;
            }

            // Flush if requested and has data
            if (_flush_mode > 0 && strm.next_out > 0)
            {
                this.onData(strm.output.subarray(0, strm.next_out));
                strm.avail_out = 0;
                continue;
            }

            if (strm.avail_in === 0) break;
        }

        return true;
    }

    /**
     * Deflate#onData(chunk) -> Void
     * - chunk (Uint8Array): output data.
     *
     * By default, stores data blocks in `chunks[]` property and glue
     * those in `onEnd`. Override this handler, if you need another behaviour.
     **/
    public onData(chunk: Uint8Array)
    {
        this.chunks.push(chunk);
    }

    /**
     * Deflate#onEnd(status) -> Void
     * - status (Number): deflate status. 0 (Z_OK) on success,
     *   other if not.
     *
     * Called once after you tell deflate that the input stream is
     * complete (Z_FINISH). By default - join collected chunks,
     * free memory and fill `results` / `err` properties.
     **/
    public onEnd(status: number)
    {
        // On success - join
        if (status === PAKO.zlib_constants.Z_OK)
        {
            this.result = UTILS.utils_common.flattenChunks(this.chunks);
        }
        this.chunks = [];
        this.err = status;
        this.msg = this.strm.msg;
    };

    /**
     * deflate(data[, options]) -> Uint8Array
     * - data (Uint8Array|String): input data to compress.
     * - options (Object): zlib deflate options.
     *
     * Compress `data` with deflate algorithm and `options`.
     *
     * Supported options are:
     *
     * - level
     * - windowBits
     * - memLevel
     * - strategy
     * - dictionary
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information on these.
     *
     * Sugar (options):
     *
     * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
     *   negative windowBits implicitly.
     *
     * ##### Example:
     *
     * ```javascript
     * const pako = require('pako')
     * const data = new Uint8Array([1,2,3,4,5,6,7,8,9]);
     *
     * console.log(pako.deflate(data));
     * ```
     **/
    public static deflate(input: Uint8Array | String, options: {}): Uint8Array | null
    {
        const deflator = new Pako_Deflate(options);

        deflator.push(input, true);

        // That will never happens, if you don't cheat with options :)
        if (deflator.err)
        { throw deflator.msg || PAKO.zlib_messages.msg(deflator.err); }

        return deflator.result;
    }

    /**
     * deflateRaw(data[, options]) -> Uint8Array
     * - data (Uint8Array|String): input data to compress.
     * - options (Object): zlib deflate options.
     *
     * The same as [[deflate]], but creates raw data, without wrapper
     * (header and adler32 crc).
     **/
    public static deflateRaw(input: Uint8Array | String, options: {}): Uint8Array | null
    {

        (<any>options).raw = true;
        return Pako_Deflate.deflate(input, options);
    }

    /**
     * gzip(data[, options]) -> Uint8Array
     * - data (Uint8Array|String): input data to compress.
     * - options (Object): zlib deflate options.
     *
     * The same as [[deflate]], but create gzip wrapper instead of
     * deflate one.
     **/
    public static gzip(input: Uint8Array | String, options?: {}): Uint8Array | null
    {
        if (typeof options == "undefined") options = {};
        (<any>options).gzip = true;
        return Pako_Deflate.deflate(input, options);
    }
}

/*
module.exports.Deflate = Deflate;
module.exports.deflate = deflate;
module.exports.deflateRaw = deflateRaw;
module.exports.gzip = gzip;
module.exports.constants = require('./zlib/constants');
*/