//
//  To compile it on Mac OS X
//    gcc -c yapi/*.c
//    g++ -o example example.cpp yocto_api.cpp yocto_pwminput.cpp *.o `pkg-config fluidsynth --libs` -framework IOKit -framework CoreFoundation
//

#include <fluidsynth.h>
#include <stdlib.h>
#include <iostream>
#include "yocto_api.h"
#include "yocto_pwminput.h"

using namespace std;

#define SF2_FILE    "FluidR3_GM.sf2"
#define N_SENSORS   10
#define CHORD_SIZE  8
#define HIST_SIZE   3

typedef struct {
    int     k[CHORD_SIZE];
} Chord;

Chord chords[2*N_SENSORS] = { 
    // Upper keyboard: melody (two different keys available)
    { 55+12, 55+12 },  // G   G
    { 57+12, 57+12 },  // A   A
    { 59+12, 58+12 },  // B   Bb
    { 61+12, 60+12 },  // C#  C
    { 62+12, 62+12 },  // D   D
    { 64+12, 64+12 },  // E   E
    { 66+12, 65+12 },  // F#  F
    { 67+12, 67+12 },  // G   G
    { 69+12, 69+12 },  // A   A
    { 71+12, 70+12 },  // B   Bb
    // Lower keyboard: bontempi-style arpeggiated chords
    { 45, 50, 54, 50, 47, 50, 54, 50 }, // A D F# D  B D F# D
    { 47, 52, 55, 52, 47, 52, 55, 52 }, // B E G E   B E G E
    { 45, 52, 55, 49, 45, 52, 55, 49 }, // A E G C#  A E G C#
    { 0 }, 
    // Note: this cord switches the melody keyboard to 2nd key
    { 46, 50, 53, 57, 0,  0,  0,  0 },  // Bb D F A
    { 0 }, 
    { 0 }, 
    { 0 }, 
    { 0 }, 
    { 0 }, 
};

typedef struct {
    int     key;
    int     histPos;
    int     history[HIST_SIZE];
    int     prevChord;
    int     nextChord;
} NoteState;

NoteState noteStates[N_SENSORS];
fluid_synth_t       *synth = NULL;
int                 subBeat = 0;
int                 idle = 0;
int                 shift = 0;

// Play the sound for a given "note"
// 
void playChord(NoteState *noteState)
{
    Chord   *notes;
    int     chord, channel, volume, i;

    // Stop previous sound for this note, if any
    if(noteState->prevChord != -1) {
        chord = noteState->prevChord;
        channel = chord/N_SENSORS;
        notes = &chords[chord];
        for(i = 0; i < CHORD_SIZE; i++) {
            if(!notes->k[i]) break;
            fluid_synth_noteoff(synth, channel, notes->k[i]);
        }
        noteState->prevChord = -1;
    }
    // Start the new sound
    if(noteState->nextChord != -1) {
        chord = noteState->nextChord;
        channel = chord/N_SENSORS;
        notes = &chords[chord];
        if(channel == 0) {
            // Channel 0: melody
            volume = 127;
            i = shift;
        } else {
            // Channel 1: chords (arpegio)
            volume = (!(subBeat & 3) ? 127 : 60);
            i = subBeat;
        }
        if(notes->k[i]) {
            fluid_synth_noteon(synth, channel, notes->k[i], volume);
        }        
        noteState->prevChord = chord;
    }
}

// Callback invoked each time a sensor measures a new distance
//
void lidarCallback(YPwmInput *lidar, const string &value)
{
    NoteState *noteState = (NoteState *)(lidar->get_userData());
    int dist = (int)(100*atof(value.c_str()));
    int pos = noteState->histPos;
    int count = 0, minDist = 0;
    int chord;

    // Discard anything above 2m3 (bad reading)
    if(dist > 260) return;

    // Store current measure in a rotating history buffer
    // (for de-bouncing)
    noteState->history[pos++] = dist;
    if(pos >= HIST_SIZE) pos=0;
    noteState->histPos = pos;

    // Use a majority filter on the history to read the
    // effective measurement (de-bouncing)
    minDist = 999;
    for(pos = 0; pos < HIST_SIZE; pos++) {
        dist = noteState->history[pos];
        if(dist > 30 && dist < 160) {
            if(minDist > dist) minDist = dist;
            count++;
        }
    }
    if(count > HIST_SIZE/2) {
        dist = minDist;
        chord = noteState->key + (dist > 100 ? 0 : N_SENSORS);
    } else {
        dist = 0;
        chord = -1;
    }
    if(noteState->prevChord >= N_SENSORS && chord == noteState->prevChord - N_SENSORS) {
        chord += N_SENSORS;
    }

    // switch between alternate melody keys if needed
    if(chord >= N_SENSORS) {
        shift = (noteState->key > 3 ? 1 : 0);
    }    
    noteState->nextChord = chord;

    if(chord != noteState->prevChord) {
        // play melody sound immediately; arpegio chords are play
        // based on timing in the main loop
        if(chord < N_SENSORS) {
            playChord(noteState);
        }
    }
}

int main(int argc, char** argv)
{
    fluid_settings_t* settings;
    fluid_audio_driver_t* adriver;
    int sfont_id;
    int i, key;
    int instrument1 = 53, instrument2 = 24; // 53=voice, 24=guitar
    string errmsg;
    char lidarName[8];

    // optional argument: instrument number to use
    if(argc > 1) instrument1 = atoi(argv[1]);
    if(argc > 2) instrument2 = atoi(argv[2]);

    // setup the MIDI synthetiser
    settings = new_fluid_settings();
    synth = new_fluid_synth(settings);
    adriver = new_fluid_audio_driver(settings, synth);
    sfont_id = fluid_synth_sfload(synth, SF2_FILE, 1);
    fluid_synth_program_select_by_sfont_name(synth, 0, SF2_FILE, 0, instrument1);
    fluid_synth_program_select_by_sfont_name(synth, 1, SF2_FILE, 0, instrument2);

    // connect to the laser organ over IP
    if(YAPI::RegisterHub("http://192.168.1.151", errmsg) != YAPI_SUCCESS) {
        cerr << "RegisterHub failed: " << errmsg << endl;
        return 1;
    }
    // enumerate LIDAR-lite sensors available
    for(i = 0; i < N_SENSORS; i++) {
        strcpy(lidarName, "Lidar");
        lidarName[5] = '0'+i;
        lidarName[6] = 0;
        cout << "Using " << lidarName << endl;
        YPwmInput *lidar = YPwmInput::FindPwmInput(lidarName);
        memset(&noteStates[i], 0, sizeof(NoteState));
        noteStates[i].key = i;
        noteStates[i].prevChord = -1;
        lidar->set_userData(&noteStates[i]);
        lidar->set_pwmReportMode(Y_PWMREPORTMODE_PWM_PULSEDURATION);
        lidar->registerValueCallback(lidarCallback);
    }

    // Loop forever
    while(1) {
        // Handle measure events during 500ms
        YAPI::Sleep(500, errmsg);

        // Play next arpeggio note (lower keyboard)
        int isIdle = 1;
        for(i = 0; i < N_SENSORS; i++) {
            if(noteStates[i].nextChord >= N_SENSORS) {
                playChord(&noteStates[i]);
                isIdle = 0;
                idle = 0;
            }
        }
        // prepare to switch to next arpeggio note
        subBeat = (subBeat + 1) & (CHORD_SIZE-1);
        if(isIdle) {
            idle++;
            if(idle > 4) subBeat = 0;
        }
    }

    // Cleanup, never reached
    delete_fluid_audio_driver(adriver);
    delete_fluid_synth(synth);
    delete_fluid_settings(settings);
    
    return 0;
}
