﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Net;
using System.Xml;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Reflection;
using System.Globalization;
using System.Threading;
using System.Threading.Tasks;
using ExcelDna.Integration;
using ExcelDna.Integration.Rtd;

public class AsyncTestAddIn : IExcelAddIn
{
    public void AutoOpen()
    {
        ExcelIntegration.RegisterUnhandledExceptionHandler(ex => "!!! EXCEPTION: " + ex.ToString());
        RtdManager.ResetUpdateInterval();
    }

    public void AutoClose()
    {
    }
}

public static class RtdManager
{
    public static void SetUpdateInterval(int interval)
    {
        object app = ExcelDnaUtil.Application;
        object rtd = app.GetType().InvokeMember("RTD", BindingFlags.GetProperty, null, app, null);
        rtd.GetType().InvokeMember("ThrottleInterval", BindingFlags.SetProperty, null, rtd, new object[] { (int)interval });
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Set real-time updates rate to 10 ms")]
    public static void SetFastUpdateInterval()
    {
        SetUpdateInterval(10);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Set real-time updates rate to 100 ms")]
    public static void SetMediumUpdateInterval()
    {
        SetUpdateInterval(100);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Reset real-time updates rate to 1 second")]
    public static void ResetUpdateInterval()
    {
        SetUpdateInterval(1000);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Show Log Window")]
    public static void ShowLog()
    {
        ExcelDna.Logging.LogDisplay.Show();
    } 
}

public static class YoctoSensorFunctions
{
    [ExcelFunction(Name = "YTimeStamp", Category = "Yoctopuce Functions",
                   Description = "Retrieve the UNIX timestamp (including millisecods) of the last sensor update")]
    public static object YTimeStamp()
    {
        return XlCall.RTD("YoctoRTDServer.SensorServer", null, "NOW", "timestamp");
    }

    [ExcelFunction(Name = "YSensorValue", Category = "Yoctopuce Functions",
                   Description = "Retrieve a live sensor value from Yoctopuce Sensor RTD server")]
    public static object YSensorValue(string sensorId)
    {
        return XlCall.RTD("YoctoRTDServer.SensorServer", null, "NOW", sensorId);
    }
}

namespace YoctoRTDServer
{
    [ComVisible(true)]
    public class SensorServer : IRtdServer
    {
        private IRTDUpdateEvent _callback;
        private Dictionary<int, string> _topics;
        private Timer _timer = null;
        private int _timestampTopic = -1;
        private bool _notifyUpdates = false;
        private bool _freezeUpdates = false;

        private static string _status = "not started";
        private static void log(string msg)
        {
            ExcelDna.Logging.LogDisplay.WriteLine("{0}", msg);
            _status = msg;
        }

        #region IRtdServer Members

        public int ServerStart(IRTDUpdateEvent CallbackObject)
        {
            Console.Beep();
            log("Starting RTD Server");
            _callback = CallbackObject;
            _topics = new Dictionary<int, string>();
            if (_timer != null) return 1;
            string errmsg = "";
            YAPI.RegisterLogFunction(log);
            if (YAPI.RegisterHub("usb", ref errmsg) == YAPI.SUCCESS)
            {
                Console.Beep();
                log("USB ready");
                _notifyUpdates = true;
                _timer = new Timer(delegate
                {
                    string dummy = "";
                    YAPI.HandleEvents(ref dummy);
                }, null, 0, 10);
            }
            else log(errmsg);
            return 1;
        }

        public int Heartbeat()
        {
            return 1;
        }

        public void ServerTerminate()
        {
            if (_timer != null)
            {
                Console.Beep();
                _timer.Change(Timeout.Infinite, Timeout.Infinite);
                _timer = null;
                _notifyUpdates = false;
                YAPI.UnregisterHub("usb");
                YAPI.FreeAPI();
                log("USB closed");
            }
        }

        public object ConnectData(int topicId, ref Array Strings, ref bool GetNewValues)
        {
            string sensorId = (string)Strings.GetValue(1);
            YSensor sensor;
            // Exit immediately if USB could not be started
            if (_timer == null) return _status;

            if (sensorId == "timestamp")
            {
                _timestampTopic = topicId;
                return YAPI.GetTickCount() / 1000.0;
            }
            if (sensorId == "any" || sensorId == "")
            {
                sensor = YSensor.FirstSensor();
                if (sensor == null)
                {
                    log("No device found, check USB cable");
                    return _status;
                }
                log("Using " + sensor.get_hardwareId());
            }
            else
            {
                log("Connecting to [" + sensorId + "]");
                sensor = YSensor.FindSensor(sensorId);
                if (!sensor.isOnline())
                {
                    log("Sensor " + sensorId + " is not online, check USB cable");
                }
            }
            sensor.set_userData(new object[] { this, (int)topicId });
            sensor.registerValueCallback(sensorValueChangeCallBack);
            _topics.Add(topicId, "");
            if(sensor.isOnline()) {
                double res = sensor.get_currentValue();
                log("Current value: " + YAPI._floatToStr(res));
                return res;
            }
            return _status;
        }

        public void DisconnectData(int topicId)
        {
            if (_timestampTopic == topicId)
            {
                _timestampTopic = -1;
            }
            else
            {
                _topics.Remove(topicId);
            }
        }

        public Array RefreshData(ref int topicCount)
        {
            // Count changed values
            _freezeUpdates = true;
            topicCount = 0;
            foreach(var value in _topics.Values) {
                if (value != "") topicCount++;
            }
            if (_timestampTopic != -1) topicCount++;
            // Create an array of changes to report
            object[,] results = new object[2, topicCount];
            int idx = 0;
            double floatVal = 0;
            foreach (KeyValuePair<int, string> pair in _topics)
            {
                if (pair.Value != "")
                {
                    results[0, idx] = pair.Key;
                    if (double.TryParse(pair.Value, NumberStyles.Any, CultureInfo.InvariantCulture, out floatVal))
                    {
                        results[1, idx] = floatVal;
                    } else {
                        results[1, idx] = pair.Value;
                    }
                    idx++;
                }
            }
            // Clear reported values from array
            while (idx > 0) _topics[--idx] = "";
            if (_timestampTopic != -1)
            {
                results[0, topicCount - 1] = _timestampTopic;
                results[1, topicCount - 1] = YAPI.GetTickCount() / 1000.0;
            }
            // Enable further updates now that this one is processed
            _freezeUpdates = false;
            _notifyUpdates = true;

            return results;
        }

        #endregion

        static void sensorValueChangeCallBack(YSensor fct, string value)
        {
            object[] info = (object [])fct.get_userData();
            SensorServer srv = (SensorServer)info[0];
            int topicId = (int)info[1];

            if (srv._topics.ContainsKey(topicId) && !srv._freezeUpdates)
            {
                srv._topics[topicId] = value;
                if (srv._notifyUpdates)
                {
                    srv._notifyUpdates = false;
                    srv._callback.UpdateNotify();
                }
            }
        }
    }

}