﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Net;
using System.Xml;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Reflection;
using System.Globalization;
using System.Threading;
using System.Threading.Tasks;
using ExcelDna.Integration;
using ExcelDna.Integration.Rtd;

public class AsyncTestAddIn : IExcelAddIn
{
    public void AutoOpen()
    {
        ExcelIntegration.RegisterUnhandledExceptionHandler(ex => "!!! EXCEPTION: " + ex.ToString());
        RtdManager.ResetUpdateInterval();
    }

    public void AutoClose()
    {
    }
}

public static class RtdManager
{
    public static void SetUpdateInterval(int interval)
    {
        object app = ExcelDnaUtil.Application;
        object rtd = app.GetType().InvokeMember("RTD", BindingFlags.GetProperty, null, app, null);
        rtd.GetType().InvokeMember("ThrottleInterval", BindingFlags.SetProperty, null, rtd, new object[] { (int)interval });
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Set real-time updates rate to 10 ms")]
    public static void SetFastUpdateInterval()
    {
        SetUpdateInterval(10);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Set real-time updates rate to 30 ms")]
    public static void SetMediumFastUpdateInterval()
    {
        SetUpdateInterval(30);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Set real-time updates rate to 100 ms")]
    public static void SetMediumUpdateInterval()
    {
        SetUpdateInterval(100);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Set real-time updates rate to 300 ms")]
    public static void SetMediumSlowUpdateInterval()
    {
        SetUpdateInterval(300);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Reset real-time updates rate to 1 second")]
    public static void ResetUpdateInterval()
    {
        SetUpdateInterval(1000);
    }

    [ExcelCommand(MenuName = "Yoctopuce Sensor RTD Server", MenuText = "Show Log Window")]
    public static void ShowLog()
    {
        ExcelDna.Logging.LogDisplay.Show();
    } 
}

public static class YoctoSensorFunctions
{
    [ExcelFunction(Name = "YTimeStamp", Category = "Yoctopuce Functions",
                   Description = "Retrieve the UNIX timestamp (including millisecods) of the last sensor update")]
    public static object YTimeStamp()
    {
        return XlCall.RTD("YoctoRTDServer.SensorServer", null, "NOW", "timestamp");
    }

    [ExcelFunction(Name = "YSensorValue", Category = "Yoctopuce Functions",
                   Description = "Retrieve a live sensor value from Yoctopuce Sensor RTD server")]
    public static object YSensorValue(string sensorId)
    {
        return XlCall.RTD("YoctoRTDServer.SensorServer", null, "NOW", sensorId);
    }

    [ExcelFunction(Name = "YLastMessage", Category = "Yoctopuce Functions",
                   Description = "Retrieve the last log message returned by the API")]
    public static object YLastMessage()
    {
        return XlCall.RTD("YoctoRTDServer.SensorServer", null, "NOW", "lastmessage");
    }

    [ExcelFunction(Name = "YRelayState", Category = "Yoctopuce Functions",
                   Description = "Get and optionally update the state of a relay")]
    public static string YRelayState(string relayId, object newState)
    {       
        YRelay relay;
        string newVal = "";

        if (relayId == "any")
        {
            relay = YRelay.FirstRelay();
        }
        else
        {
            relay = YRelay.FindRelay(relayId);
        }
        if (relay == null || !relay.isOnline())
        {
            string errmsg = "";
            YAPI.UpdateDeviceList(ref errmsg);
            if (relayId == "any")
            {
                relay = YRelay.FirstRelay();
            }
            else
            {
                relay = YRelay.FindRelay(relayId);
            }
            if (relay == null || !relay.isOnline())
            {
                return "not connected";
            }
        }

        if (newState is ExcelMissing)
        {
            newVal = "";
        }
        else
        {
            newVal = newState.ToString();
        }
        if (newVal == "A" || newVal == "0")
        {            
            relay.set_state(YRelay.STATE_A);
        }
        else if (newVal == "B" || newVal == "1")
        {
            relay.set_state(YRelay.STATE_B);
        }
        else if (newVal != "")
        {
            return "invalid new state";
        }
        return (relay.get_state() == YRelay.STATE_A ? "A" : "B");
    }

    [ExcelFunction(Name = "YColorLedRGB", Category = "Yoctopuce Functions",
                   Description = "Get and optionally update the RGB color of an RGB LED")]
    public static string YColorLedRGB(string colorLedId, object newColor, object newG, object newB)
    {
        YColorLed colorled;
        int newVal = -1;

        if (colorLedId == "any")
        {
            colorled = YColorLed.FirstColorLed();
        }
        else
        {
            colorled = YColorLed.FindColorLed(colorLedId);
        }
        if (colorled == null || !colorled.isOnline())
        {
            string errmsg = "";
            YAPI.UpdateDeviceList(ref errmsg);
            if (colorLedId == "any")
            {
                colorled = YColorLed.FirstColorLed();
            }
            else
            {
                colorled = YColorLed.FindColorLed(colorLedId);
            }
            if (colorled == null || !colorled.isOnline())
            {
                return "not connected";
            }
        }

        if (newColor is ExcelMissing)
        {
            newColor = "";
        }
        else if (newColor is string)
        {
            string hexStr = newColor.ToString().Replace("#", "").Replace("$", "").Replace("0x", "");
            if(hexStr.Length == 6) 
            {
                newVal = Int32.Parse(hexStr, NumberStyles.HexNumber);
            }
        }
        else if (!(newG is ExcelMissing || newB is ExcelMissing))
        {
            int r, g, b;
            if(Int32.TryParse(newColor.ToString(), out r) &&
               Int32.TryParse(newG.ToString(), out g) &&
               Int32.TryParse(newB.ToString(), out b))
            {
                newVal = (r << 16) + (g << 8) + b;
            }
        }
        if (newVal == -1 && !((newColor is string) && newColor.ToString() == ""))
        {
            return "invalid new RGB color";
        }
        if (newVal != -1)
        {
            colorled.set_rgbColor(newVal);
        }
        return String.Format("#{0:X06}", colorled.get_rgbColor() & 0xffffff);
    }

    [ExcelFunction(Name = "YServoPosition", Category = "Yoctopuce Functions",
                   Description = "Get and optionally update the position of a Yocto-Servo")]
    public static object YServoPosition(string servoId, object newPos)
    {
        YServo servo;
        int newVal = 999999;

        if (servoId == "any")
        {
            servo = YServo.FirstServo();
        }
        else
        {
            servo = YServo.FindServo(servoId);
        }
        if (servo == null || !servo.isOnline())
        {
            string errmsg = "";
            YAPI.UpdateDeviceList(ref errmsg);
            if (servoId == "any")
            {
                servo = YServo.FirstServo();
            }
            else
            {
                servo = YServo.FindServo(servoId);
            }
            if (servo == null || !servo.isOnline())
            {
                return "not connected";
            }
        }

        if (newPos is ExcelMissing)
        {
            newPos = "";
        }
        else if (!Int32.TryParse(newPos.ToString(), out newVal))
        {
            newVal = 999999;
        }
        if (newVal == 999999 && !((newPos is string) && newPos.ToString() == ""))
        {
            return "invalid new position";
        }
        if (newVal != 999999)
        {
            servo.set_position(newVal);
        }
        return servo.get_position();
    }
}

namespace YoctoRTDServer
{
    [ComVisible(true)]
    public class SensorServer : IRtdServer
    {
        private IRTDUpdateEvent _callback;
        private Dictionary<int, string> _topics;
        private Timer _timer = null;
        private int _timestampTopic = -1;
        private int _lastmessageTopic = -1;
        private bool _notifyUpdates = false;
        private bool _freezeUpdates = false;

        private static bool _statusChanged = false;
        private static string _status = "not started";
        private static void log(string msg)
        {
            ExcelDna.Logging.LogDisplay.RecordLine("{0}", msg);
            _status = msg;
            _statusChanged = true;
        }

        #region IRtdServer Members

        public int ServerStart(IRTDUpdateEvent CallbackObject)
        {
            Console.Beep();
            log("Starting RTD Server");
            _callback = CallbackObject;
            _topics = new Dictionary<int, string>();
            if (_timer != null) return 1;
            string errmsg = "";
            YAPI.RegisterLogFunction(log);
            if (YAPI.RegisterHub("usb", ref errmsg) == YAPI.SUCCESS)
            {
                Console.Beep();
                log("USB ready");
                _notifyUpdates = true;
                _timer = new Timer(delegate
                {
                    string dummy = "";
                    YAPI.HandleEvents(ref dummy);
                }, null, 0, 10);
            }
            else log(errmsg);
            return 1;
        }

        public int Heartbeat()
        {
            return 1;
        }

        public void ServerTerminate()
        {
            if (_timer != null)
            {
                Console.Beep();
                _timer.Change(Timeout.Infinite, Timeout.Infinite);
                _timer = null;
                _notifyUpdates = false;
                YAPI.UnregisterHub("usb");
                YAPI.FreeAPI();
                log("USB closed");
            }
        }

        public object ConnectData(int topicId, ref Array Strings, ref bool GetNewValues)
        {
            string sensorId = (string)Strings.GetValue(1);

            // Exit immediately if USB could not be started
            if (_timer == null) return _status;

            if (sensorId == "timestamp")
            {
                _timestampTopic = topicId;
                return YAPI.GetTickCount() / 1000.0;
            }
            if (sensorId == "lastmessage")
            {
                _lastmessageTopic = topicId;
                _statusChanged = false;
                return _status;
            }
            if (sensorId == "any" || sensorId == "")
            {
                YSensor sensor = YSensor.FirstSensor();
                if (sensor == null)
                {
                    log("No device found, check USB cable");
                    return _status;
                }
                log("Using " + sensor.get_hardwareId());
            }
            else
            {
                log("Connecting to [" + sensorId + "]");
                YSensor sensor = YSensor.FindSensor(sensorId);
                if (sensor.isOnline())
                {
                    sensor.set_userData(new object[] { this, (int)topicId });
                    sensor.registerValueCallback(sensorValueChangeCallBack);
                    _topics.Add(topicId, "");
                    if (sensor.isOnline())
                    {
                        double res = sensor.get_currentValue();
                        log("Current value: " + YAPI._floatToStr(res));
                        return res;
                    }
                } else 
                {
                    YAnButton anbutton = YAnButton.FindAnButton(sensorId);
                    if (anbutton.isOnline())
                    {
                        anbutton.set_userData(new object[] { this, (int)topicId });
                        anbutton.registerValueCallback(anbuttonValueChangeCallBack);
                        _topics.Add(topicId, "");
                        if (anbutton.isOnline())
                        {
                            double res = anbutton.get_calibratedValue();
                            log("Current value: " + YAPI._floatToStr(res));
                            return res;
                        }
                    }
                    else
                    {
                        log("Sensor " + sensorId + " is not online, check USB cable");
                    }
                }
            }
            return _status;
        }

        public void DisconnectData(int topicId)
        {
            if (_timestampTopic == topicId)
            {
                _timestampTopic = -1;
            }
            else if (_lastmessageTopic == topicId)
            {
                _lastmessageTopic = -1;
            }
            else
            {
                _topics.Remove(topicId);
            }
        }

        public Array RefreshData(ref int topicCount)
        {
            // Count changed values
            _freezeUpdates = true;
            topicCount = 0;
            foreach(var value in _topics.Values) {
                if (value != "") topicCount++;
            }
            if (_timestampTopic != -1) topicCount++;
            if (_lastmessageTopic != -1 && _statusChanged) topicCount++;
            // Create an array of changes to report
            object[,] results = new object[2, topicCount];
            int idx = 0;
            double floatVal = 0;
            foreach (KeyValuePair<int, string> pair in _topics)
            {
                if (pair.Value != "")
                {
                    results[0, idx] = pair.Key;
                    if (double.TryParse(pair.Value, NumberStyles.Any, CultureInfo.InvariantCulture, out floatVal))
                    {
                        results[1, idx] = floatVal;
                    } else {
                        results[1, idx] = pair.Value;
                    }
                    idx++;
                }
            }
            // Clear reported values from array
            int i2 = idx;
            while (i2 > 0) _topics[--i2] = "";
            if (_timestampTopic != -1)
            {
                results[0, idx] = _timestampTopic;
                results[1, idx] = YAPI.GetTickCount() / 1000.0;
                idx++;
            }
            if (_lastmessageTopic != -1 && _statusChanged)
            {
                _statusChanged = false;
                results[0, idx] = _lastmessageTopic;
                results[1, idx] = _status;
                idx++;
            }
            // Enable further updates now that this one is processed
            _freezeUpdates = false;
            _notifyUpdates = true;

            return results;
        }

        #endregion

        static void sensorValueChangeCallBack(YSensor fct, string value)
        {
            object[] info = (object [])fct.get_userData();
            SensorServer srv = (SensorServer)info[0];
            int topicId = (int)info[1];

            if (srv._topics.ContainsKey(topicId) && !srv._freezeUpdates)
            {
                srv._topics[topicId] = value;
                if (srv._notifyUpdates)
                {
                    srv._notifyUpdates = false;
                    srv._callback.UpdateNotify();
                }
            }
        }

        static void anbuttonValueChangeCallBack(YAnButton fct, string value)
        {
            object[] info = (object[])fct.get_userData();
            SensorServer srv = (SensorServer)info[0];
            int topicId = (int)info[1];

            if (srv._topics.ContainsKey(topicId) && !srv._freezeUpdates)
            {
                srv._topics[topicId] = value;
                if (srv._notifyUpdates)
                {
                    srv._notifyUpdates = false;
                    srv._callback.UpdateNotify();
                }
            }
        }
    }

}