/*********************************************************************
 *
 *  $Id: yocto_network.ts 63327 2024-11-13 09:35:03Z seb $
 *
 *  Implements the high-level API for Network functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/
import { YAPIContext, YFunction } from './yocto_api.js';
/**
 * YNetwork Class: network interface control interface, available for instance in the
 * YoctoHub-Ethernet, the YoctoHub-GSM-4G, the YoctoHub-Wireless-SR or the YoctoHub-Wireless-n
 *
 * YNetwork objects provide access to TCP/IP parameters of Yoctopuce
 * devices that include a built-in network interface.
 */
export declare class YNetwork extends YFunction {
    _className: string;
    _readiness: YNetwork.READINESS;
    _macAddress: string;
    _ipAddress: string;
    _subnetMask: string;
    _router: string;
    _currentDNS: string;
    _ipConfig: string;
    _primaryDNS: string;
    _secondaryDNS: string;
    _ntpServer: string;
    _userPassword: string;
    _adminPassword: string;
    _httpPort: number;
    _httpsPort: number;
    _securityMode: YNetwork.SECURITYMODE;
    _defaultPage: string;
    _discoverable: YNetwork.DISCOVERABLE;
    _wwwWatchdogDelay: number;
    _callbackUrl: string;
    _callbackMethod: YNetwork.CALLBACKMETHOD;
    _callbackEncoding: YNetwork.CALLBACKENCODING;
    _callbackTemplate: YNetwork.CALLBACKTEMPLATE;
    _callbackCredentials: string;
    _callbackInitialDelay: number;
    _callbackSchedule: string;
    _callbackMinDelay: number;
    _callbackMaxDelay: number;
    _poeCurrent: number;
    _valueCallbackNetwork: YNetwork.ValueCallback | null;
    readonly READINESS_DOWN: YNetwork.READINESS;
    readonly READINESS_EXISTS: YNetwork.READINESS;
    readonly READINESS_LINKED: YNetwork.READINESS;
    readonly READINESS_LAN_OK: YNetwork.READINESS;
    readonly READINESS_WWW_OK: YNetwork.READINESS;
    readonly READINESS_INVALID: YNetwork.READINESS;
    readonly MACADDRESS_INVALID: string;
    readonly IPADDRESS_INVALID: string;
    readonly SUBNETMASK_INVALID: string;
    readonly ROUTER_INVALID: string;
    readonly CURRENTDNS_INVALID: string;
    readonly IPCONFIG_INVALID: string;
    readonly PRIMARYDNS_INVALID: string;
    readonly SECONDARYDNS_INVALID: string;
    readonly NTPSERVER_INVALID: string;
    readonly USERPASSWORD_INVALID: string;
    readonly ADMINPASSWORD_INVALID: string;
    readonly HTTPPORT_INVALID: number;
    readonly HTTPSPORT_INVALID: number;
    readonly SECURITYMODE_UNDEFINED: YNetwork.SECURITYMODE;
    readonly SECURITYMODE_LEGACY: YNetwork.SECURITYMODE;
    readonly SECURITYMODE_MIXED: YNetwork.SECURITYMODE;
    readonly SECURITYMODE_SECURE: YNetwork.SECURITYMODE;
    readonly SECURITYMODE_INVALID: YNetwork.SECURITYMODE;
    readonly DEFAULTPAGE_INVALID: string;
    readonly DISCOVERABLE_FALSE: YNetwork.DISCOVERABLE;
    readonly DISCOVERABLE_TRUE: YNetwork.DISCOVERABLE;
    readonly DISCOVERABLE_INVALID: YNetwork.DISCOVERABLE;
    readonly WWWWATCHDOGDELAY_INVALID: number;
    readonly CALLBACKURL_INVALID: string;
    readonly CALLBACKMETHOD_POST: YNetwork.CALLBACKMETHOD;
    readonly CALLBACKMETHOD_GET: YNetwork.CALLBACKMETHOD;
    readonly CALLBACKMETHOD_PUT: YNetwork.CALLBACKMETHOD;
    readonly CALLBACKMETHOD_INVALID: YNetwork.CALLBACKMETHOD;
    readonly CALLBACKENCODING_FORM: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_JSON: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_JSON_ARRAY: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_CSV: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_YOCTO_API: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_JSON_NUM: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_EMONCMS: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_AZURE: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_INFLUXDB: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_MQTT: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_YOCTO_API_JZON: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_PRTG: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_INFLUXDB_V2: YNetwork.CALLBACKENCODING;
    readonly CALLBACKENCODING_INVALID: YNetwork.CALLBACKENCODING;
    readonly CALLBACKTEMPLATE_OFF: YNetwork.CALLBACKTEMPLATE;
    readonly CALLBACKTEMPLATE_ON: YNetwork.CALLBACKTEMPLATE;
    readonly CALLBACKTEMPLATE_INVALID: YNetwork.CALLBACKTEMPLATE;
    readonly CALLBACKCREDENTIALS_INVALID: string;
    readonly CALLBACKINITIALDELAY_INVALID: number;
    readonly CALLBACKSCHEDULE_INVALID: string;
    readonly CALLBACKMINDELAY_INVALID: number;
    readonly CALLBACKMAXDELAY_INVALID: number;
    readonly POECURRENT_INVALID: number;
    static readonly READINESS_DOWN: YNetwork.READINESS;
    static readonly READINESS_EXISTS: YNetwork.READINESS;
    static readonly READINESS_LINKED: YNetwork.READINESS;
    static readonly READINESS_LAN_OK: YNetwork.READINESS;
    static readonly READINESS_WWW_OK: YNetwork.READINESS;
    static readonly READINESS_INVALID: YNetwork.READINESS;
    static readonly MACADDRESS_INVALID: string;
    static readonly IPADDRESS_INVALID: string;
    static readonly SUBNETMASK_INVALID: string;
    static readonly ROUTER_INVALID: string;
    static readonly CURRENTDNS_INVALID: string;
    static readonly IPCONFIG_INVALID: string;
    static readonly PRIMARYDNS_INVALID: string;
    static readonly SECONDARYDNS_INVALID: string;
    static readonly NTPSERVER_INVALID: string;
    static readonly USERPASSWORD_INVALID: string;
    static readonly ADMINPASSWORD_INVALID: string;
    static readonly HTTPPORT_INVALID: number;
    static readonly HTTPSPORT_INVALID: number;
    static readonly SECURITYMODE_UNDEFINED: YNetwork.SECURITYMODE;
    static readonly SECURITYMODE_LEGACY: YNetwork.SECURITYMODE;
    static readonly SECURITYMODE_MIXED: YNetwork.SECURITYMODE;
    static readonly SECURITYMODE_SECURE: YNetwork.SECURITYMODE;
    static readonly SECURITYMODE_INVALID: YNetwork.SECURITYMODE;
    static readonly DEFAULTPAGE_INVALID: string;
    static readonly DISCOVERABLE_FALSE: YNetwork.DISCOVERABLE;
    static readonly DISCOVERABLE_TRUE: YNetwork.DISCOVERABLE;
    static readonly DISCOVERABLE_INVALID: YNetwork.DISCOVERABLE;
    static readonly WWWWATCHDOGDELAY_INVALID: number;
    static readonly CALLBACKURL_INVALID: string;
    static readonly CALLBACKMETHOD_POST: YNetwork.CALLBACKMETHOD;
    static readonly CALLBACKMETHOD_GET: YNetwork.CALLBACKMETHOD;
    static readonly CALLBACKMETHOD_PUT: YNetwork.CALLBACKMETHOD;
    static readonly CALLBACKMETHOD_INVALID: YNetwork.CALLBACKMETHOD;
    static readonly CALLBACKENCODING_FORM: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_JSON: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_JSON_ARRAY: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_CSV: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_YOCTO_API: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_JSON_NUM: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_EMONCMS: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_AZURE: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_INFLUXDB: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_MQTT: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_YOCTO_API_JZON: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_PRTG: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_INFLUXDB_V2: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKENCODING_INVALID: YNetwork.CALLBACKENCODING;
    static readonly CALLBACKTEMPLATE_OFF: YNetwork.CALLBACKTEMPLATE;
    static readonly CALLBACKTEMPLATE_ON: YNetwork.CALLBACKTEMPLATE;
    static readonly CALLBACKTEMPLATE_INVALID: YNetwork.CALLBACKTEMPLATE;
    static readonly CALLBACKCREDENTIALS_INVALID: string;
    static readonly CALLBACKINITIALDELAY_INVALID: number;
    static readonly CALLBACKSCHEDULE_INVALID: string;
    static readonly CALLBACKMINDELAY_INVALID: number;
    static readonly CALLBACKMAXDELAY_INVALID: number;
    static readonly POECURRENT_INVALID: number;
    constructor(yapi: YAPIContext, func: string);
    imm_parseAttr(name: string, val: any): number;
    /**
     * Returns the current established working mode of the network interface.
     * Level zero (DOWN_0) means that no hardware link has been detected. Either there is no signal
     * on the network cable, or the selected wireless access point cannot be detected.
     * Level 1 (LIVE_1) is reached when the network is detected, but is not yet connected.
     * For a wireless network, this shows that the requested SSID is present.
     * Level 2 (LINK_2) is reached when the hardware connection is established.
     * For a wired network connection, level 2 means that the cable is attached at both ends.
     * For a connection to a wireless access point, it shows that the security parameters
     * are properly configured. For an ad-hoc wireless connection, it means that there is
     * at least one other device connected on the ad-hoc network.
     * Level 3 (DHCP_3) is reached when an IP address has been obtained using DHCP.
     * Level 4 (DNS_4) is reached when the DNS server is reachable on the network.
     * Level 5 (WWW_5) is reached when global connectivity is demonstrated by properly loading the
     * current time from an NTP server.
     *
     * @return a value among YNetwork.READINESS_DOWN, YNetwork.READINESS_EXISTS,
     * YNetwork.READINESS_LINKED, YNetwork.READINESS_LAN_OK and YNetwork.READINESS_WWW_OK corresponding to
     * the current established working mode of the network interface
     *
     * On failure, throws an exception or returns YNetwork.READINESS_INVALID.
     */
    get_readiness(): Promise<YNetwork.READINESS>;
    /**
     * Returns the MAC address of the network interface. The MAC address is also available on a sticker
     * on the module, in both numeric and barcode forms.
     *
     * @return a string corresponding to the MAC address of the network interface
     *
     * On failure, throws an exception or returns YNetwork.MACADDRESS_INVALID.
     */
    get_macAddress(): Promise<string>;
    /**
     * Returns the IP address currently in use by the device. The address may have been configured
     * statically, or provided by a DHCP server.
     *
     * @return a string corresponding to the IP address currently in use by the device
     *
     * On failure, throws an exception or returns YNetwork.IPADDRESS_INVALID.
     */
    get_ipAddress(): Promise<string>;
    /**
     * Returns the subnet mask currently used by the device.
     *
     * @return a string corresponding to the subnet mask currently used by the device
     *
     * On failure, throws an exception or returns YNetwork.SUBNETMASK_INVALID.
     */
    get_subnetMask(): Promise<string>;
    /**
     * Returns the IP address of the router on the device subnet (default gateway).
     *
     * @return a string corresponding to the IP address of the router on the device subnet (default gateway)
     *
     * On failure, throws an exception or returns YNetwork.ROUTER_INVALID.
     */
    get_router(): Promise<string>;
    /**
     * Returns the IP address of the DNS server currently used by the device.
     *
     * @return a string corresponding to the IP address of the DNS server currently used by the device
     *
     * On failure, throws an exception or returns YNetwork.CURRENTDNS_INVALID.
     */
    get_currentDNS(): Promise<string>;
    /**
     * Returns the IP configuration of the network interface.
     *
     * If the network interface is set up to use a static IP address, the string starts with "STATIC:" and
     * is followed by three
     * parameters, separated by "/". The first is the device IP address, followed by the subnet mask
     * length, and finally the
     * router IP address (default gateway). For instance: "STATIC:192.168.1.14/16/192.168.1.1"
     *
     * If the network interface is configured to receive its IP from a DHCP server, the string start with
     * "DHCP:" and is followed by
     * three parameters separated by "/". The first is the fallback IP address, then the fallback subnet
     * mask length and finally the
     * fallback router IP address. These three parameters are used when no DHCP reply is received.
     *
     * @return a string corresponding to the IP configuration of the network interface
     *
     * On failure, throws an exception or returns YNetwork.IPCONFIG_INVALID.
     */
    get_ipConfig(): Promise<string>;
    set_ipConfig(newval: string): Promise<number>;
    /**
     * Returns the IP address of the primary name server to be used by the module.
     *
     * @return a string corresponding to the IP address of the primary name server to be used by the module
     *
     * On failure, throws an exception or returns YNetwork.PRIMARYDNS_INVALID.
     */
    get_primaryDNS(): Promise<string>;
    /**
     * Changes the IP address of the primary name server to be used by the module.
     * When using DHCP, if a value is specified, it overrides the value received from the DHCP server.
     * Remember to call the saveToFlash() method and then to reboot the module to apply this setting.
     *
     * @param newval : a string corresponding to the IP address of the primary name server to be used by the module
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_primaryDNS(newval: string): Promise<number>;
    /**
     * Returns the IP address of the secondary name server to be used by the module.
     *
     * @return a string corresponding to the IP address of the secondary name server to be used by the module
     *
     * On failure, throws an exception or returns YNetwork.SECONDARYDNS_INVALID.
     */
    get_secondaryDNS(): Promise<string>;
    /**
     * Changes the IP address of the secondary name server to be used by the module.
     * When using DHCP, if a value is specified, it overrides the value received from the DHCP server.
     * Remember to call the saveToFlash() method and then to reboot the module to apply this setting.
     *
     * @param newval : a string corresponding to the IP address of the secondary name server to be used by the module
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_secondaryDNS(newval: string): Promise<number>;
    /**
     * Returns the IP address of the NTP server to be used by the device.
     *
     * @return a string corresponding to the IP address of the NTP server to be used by the device
     *
     * On failure, throws an exception or returns YNetwork.NTPSERVER_INVALID.
     */
    get_ntpServer(): Promise<string>;
    /**
     * Changes the IP address of the NTP server to be used by the module. Use an empty
     * string to restore the factory set  address.
     * Remember to call the saveToFlash() method and then to reboot the module to apply this setting.
     *
     * @param newval : a string corresponding to the IP address of the NTP server to be used by the module
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_ntpServer(newval: string): Promise<number>;
    /**
     * Returns a hash string if a password has been set for "user" user,
     * or an empty string otherwise.
     *
     * @return a string corresponding to a hash string if a password has been set for "user" user,
     *         or an empty string otherwise
     *
     * On failure, throws an exception or returns YNetwork.USERPASSWORD_INVALID.
     */
    get_userPassword(): Promise<string>;
    /**
     * Changes the password for the "user" user. This password becomes instantly required
     * to perform any use of the module. If the specified value is an
     * empty string, a password is not required anymore.
     * Remember to call the saveToFlash() method of the module if the
     * modification must be kept.
     *
     * @param newval : a string corresponding to the password for the "user" user
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_userPassword(newval: string): Promise<number>;
    /**
     * Returns a hash string if a password has been set for user "admin",
     * or an empty string otherwise.
     *
     * @return a string corresponding to a hash string if a password has been set for user "admin",
     *         or an empty string otherwise
     *
     * On failure, throws an exception or returns YNetwork.ADMINPASSWORD_INVALID.
     */
    get_adminPassword(): Promise<string>;
    /**
     * Changes the password for the "admin" user. This password becomes instantly required
     * to perform any change of the module state. If the specified value is an
     * empty string, a password is not required anymore.
     * Remember to call the saveToFlash() method of the module if the
     * modification must be kept.
     *
     * @param newval : a string corresponding to the password for the "admin" user
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_adminPassword(newval: string): Promise<number>;
    /**
     * Returns the TCP port used to serve the hub web UI.
     *
     * @return an integer corresponding to the TCP port used to serve the hub web UI
     *
     * On failure, throws an exception or returns YNetwork.HTTPPORT_INVALID.
     */
    get_httpPort(): Promise<number>;
    /**
     * Changes the the TCP port used to serve the hub web UI. The default value is port 80,
     * which is the default for all Web servers. Regardless of the value set here,
     * the hub will always reply on port 4444, which is used by default by Yoctopuce
     * API library. When you change this parameter, remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : an integer corresponding to the the TCP port used to serve the hub web UI
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_httpPort(newval: number): Promise<number>;
    /**
     * Returns the secure TCP port used to serve the hub web UI.
     *
     * @return an integer corresponding to the secure TCP port used to serve the hub web UI
     *
     * On failure, throws an exception or returns YNetwork.HTTPSPORT_INVALID.
     */
    get_httpsPort(): Promise<number>;
    /**
     * Changes the secure TCP port used to serve the hub web UI. The default value is port 4443,
     * which is the default for all Web servers. When you change this parameter, remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : an integer corresponding to the secure TCP port used to serve the hub web UI
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_httpsPort(newval: number): Promise<number>;
    /**
     * Returns the security level chosen to prevent unauthorized access to the server.
     *
     * @return a value among YNetwork.SECURITYMODE_UNDEFINED, YNetwork.SECURITYMODE_LEGACY,
     * YNetwork.SECURITYMODE_MIXED and YNetwork.SECURITYMODE_SECURE corresponding to the security level
     * chosen to prevent unauthorized access to the server
     *
     * On failure, throws an exception or returns YNetwork.SECURITYMODE_INVALID.
     */
    get_securityMode(): Promise<YNetwork.SECURITYMODE>;
    /**
     * Changes the security level used to prevent unauthorized access to the server.
     * The value UNDEFINED causes the security configuration wizard to be
     * displayed the next time you log on to the Web console.
     * The value LEGACY offers unencrypted HTTP access by default, and
     * is designed to provide compatibility with legacy applications that do not
     * handle password or do not support HTTPS. But it should
     * only be used when system security is guaranteed by other means, such as the
     * use of a firewall.
     * The value MIXED requires the configuration of passwords, and allows
     * access via both HTTP (unencrypted) and HTTPS (encrypted), while requiring
     * the Yoctopuce API to be tolerant of certificate characteristics.
     * The value SECURE requires the configuration of passwords and the
     * use of secure communications in all cases.
     * When you change this parameter, remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : a value among YNetwork.SECURITYMODE_UNDEFINED, YNetwork.SECURITYMODE_LEGACY,
     * YNetwork.SECURITYMODE_MIXED and YNetwork.SECURITYMODE_SECURE corresponding to the security level
     * used to prevent unauthorized access to the server
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_securityMode(newval: YNetwork.SECURITYMODE): Promise<number>;
    /**
     * Returns the HTML page to serve for the URL "/"" of the hub.
     *
     * @return a string corresponding to the HTML page to serve for the URL "/"" of the hub
     *
     * On failure, throws an exception or returns YNetwork.DEFAULTPAGE_INVALID.
     */
    get_defaultPage(): Promise<string>;
    /**
     * Changes the default HTML page returned by the hub. If not value are set the hub return
     * "index.html" which is the web interface of the hub. It is possible to change this page
     * for file that has been uploaded on the hub. The maximum filename size is 15 characters.
     * When you change this parameter, remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : a string corresponding to the default HTML page returned by the hub
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_defaultPage(newval: string): Promise<number>;
    /**
     * Returns the activation state of the multicast announce protocols to allow easy
     * discovery of the module in the network neighborhood (uPnP/Bonjour protocol).
     *
     * @return either YNetwork.DISCOVERABLE_FALSE or YNetwork.DISCOVERABLE_TRUE, according to the
     * activation state of the multicast announce protocols to allow easy
     *         discovery of the module in the network neighborhood (uPnP/Bonjour protocol)
     *
     * On failure, throws an exception or returns YNetwork.DISCOVERABLE_INVALID.
     */
    get_discoverable(): Promise<YNetwork.DISCOVERABLE>;
    /**
     * Changes the activation state of the multicast announce protocols to allow easy
     * discovery of the module in the network neighborhood (uPnP/Bonjour protocol).
     * Remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : either YNetwork.DISCOVERABLE_FALSE or YNetwork.DISCOVERABLE_TRUE, according to the
     * activation state of the multicast announce protocols to allow easy
     *         discovery of the module in the network neighborhood (uPnP/Bonjour protocol)
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_discoverable(newval: YNetwork.DISCOVERABLE): Promise<number>;
    /**
     * Returns the allowed downtime of the WWW link (in seconds) before triggering an automated
     * reboot to try to recover Internet connectivity. A zero value disables automated reboot
     * in case of Internet connectivity loss.
     *
     * @return an integer corresponding to the allowed downtime of the WWW link (in seconds) before
     * triggering an automated
     *         reboot to try to recover Internet connectivity
     *
     * On failure, throws an exception or returns YNetwork.WWWWATCHDOGDELAY_INVALID.
     */
    get_wwwWatchdogDelay(): Promise<number>;
    /**
     * Changes the allowed downtime of the WWW link (in seconds) before triggering an automated
     * reboot to try to recover Internet connectivity. A zero value disables automated reboot
     * in case of Internet connectivity loss. The smallest valid non-zero timeout is
     * 90 seconds. Remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : an integer corresponding to the allowed downtime of the WWW link (in seconds)
     * before triggering an automated
     *         reboot to try to recover Internet connectivity
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_wwwWatchdogDelay(newval: number): Promise<number>;
    /**
     * Returns the callback URL to notify of significant state changes.
     *
     * @return a string corresponding to the callback URL to notify of significant state changes
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKURL_INVALID.
     */
    get_callbackUrl(): Promise<string>;
    /**
     * Changes the callback URL to notify significant state changes. Remember to call the
     * saveToFlash() method of the module if the modification must be kept.
     *
     * @param newval : a string corresponding to the callback URL to notify significant state changes
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackUrl(newval: string): Promise<number>;
    /**
     * Returns the HTTP method used to notify callbacks for significant state changes.
     *
     * @return a value among YNetwork.CALLBACKMETHOD_POST, YNetwork.CALLBACKMETHOD_GET and
     * YNetwork.CALLBACKMETHOD_PUT corresponding to the HTTP method used to notify callbacks for
     * significant state changes
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKMETHOD_INVALID.
     */
    get_callbackMethod(): Promise<YNetwork.CALLBACKMETHOD>;
    /**
     * Changes the HTTP method used to notify callbacks for significant state changes.
     * Remember to call the saveToFlash() method of the module if the
     * modification must be kept.
     *
     * @param newval : a value among YNetwork.CALLBACKMETHOD_POST, YNetwork.CALLBACKMETHOD_GET and
     * YNetwork.CALLBACKMETHOD_PUT corresponding to the HTTP method used to notify callbacks for
     * significant state changes
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackMethod(newval: YNetwork.CALLBACKMETHOD): Promise<number>;
    /**
     * Returns the encoding standard to use for representing notification values.
     *
     * @return a value among YNetwork.CALLBACKENCODING_FORM, YNetwork.CALLBACKENCODING_JSON,
     * YNetwork.CALLBACKENCODING_JSON_ARRAY, YNetwork.CALLBACKENCODING_CSV,
     * YNetwork.CALLBACKENCODING_YOCTO_API, YNetwork.CALLBACKENCODING_JSON_NUM,
     * YNetwork.CALLBACKENCODING_EMONCMS, YNetwork.CALLBACKENCODING_AZURE,
     * YNetwork.CALLBACKENCODING_INFLUXDB, YNetwork.CALLBACKENCODING_MQTT,
     * YNetwork.CALLBACKENCODING_YOCTO_API_JZON, YNetwork.CALLBACKENCODING_PRTG and
     * YNetwork.CALLBACKENCODING_INFLUXDB_V2 corresponding to the encoding standard to use for
     * representing notification values
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKENCODING_INVALID.
     */
    get_callbackEncoding(): Promise<YNetwork.CALLBACKENCODING>;
    /**
     * Changes the encoding standard to use for representing notification values.
     * Remember to call the saveToFlash() method of the module if the
     * modification must be kept.
     *
     * @param newval : a value among YNetwork.CALLBACKENCODING_FORM, YNetwork.CALLBACKENCODING_JSON,
     * YNetwork.CALLBACKENCODING_JSON_ARRAY, YNetwork.CALLBACKENCODING_CSV,
     * YNetwork.CALLBACKENCODING_YOCTO_API, YNetwork.CALLBACKENCODING_JSON_NUM,
     * YNetwork.CALLBACKENCODING_EMONCMS, YNetwork.CALLBACKENCODING_AZURE,
     * YNetwork.CALLBACKENCODING_INFLUXDB, YNetwork.CALLBACKENCODING_MQTT,
     * YNetwork.CALLBACKENCODING_YOCTO_API_JZON, YNetwork.CALLBACKENCODING_PRTG and
     * YNetwork.CALLBACKENCODING_INFLUXDB_V2 corresponding to the encoding standard to use for
     * representing notification values
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackEncoding(newval: YNetwork.CALLBACKENCODING): Promise<number>;
    /**
     * Returns the activation state of the custom template file to customize callback
     * format. If the custom callback template is disabled, it will be ignored even
     * if present on the YoctoHub.
     *
     * @return either YNetwork.CALLBACKTEMPLATE_OFF or YNetwork.CALLBACKTEMPLATE_ON, according to the
     * activation state of the custom template file to customize callback
     *         format
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKTEMPLATE_INVALID.
     */
    get_callbackTemplate(): Promise<YNetwork.CALLBACKTEMPLATE>;
    /**
     * Enable the use of a template file to customize callbacks format.
     * When the custom callback template file is enabled, the template file
     * will be loaded for each callback in order to build the data to post to the
     * server. If template file does not exist on the YoctoHub, the callback will
     * fail with an error message indicating the name of the expected template file.
     * Remember to call the saveToFlash() method of the module if the
     * modification must be kept.
     *
     * @param newval : either YNetwork.CALLBACKTEMPLATE_OFF or YNetwork.CALLBACKTEMPLATE_ON
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackTemplate(newval: YNetwork.CALLBACKTEMPLATE): Promise<number>;
    /**
     * Returns a hashed version of the notification callback credentials if set,
     * or an empty string otherwise.
     *
     * @return a string corresponding to a hashed version of the notification callback credentials if set,
     *         or an empty string otherwise
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKCREDENTIALS_INVALID.
     */
    get_callbackCredentials(): Promise<string>;
    /**
     * Changes the credentials required to connect to the callback address. The credentials
     * must be provided as returned by function get_callbackCredentials,
     * in the form username:hash. The method used to compute the hash varies according
     * to the the authentication scheme implemented by the callback, For Basic authentication,
     * the hash is the MD5 of the string username:password. For Digest authentication,
     * the hash is the MD5 of the string username:realm:password. For a simpler
     * way to configure callback credentials, use function callbackLogin instead.
     * Remember to call the saveToFlash() method of the module if the
     * modification must be kept.
     *
     * @param newval : a string corresponding to the credentials required to connect to the callback address
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackCredentials(newval: string): Promise<number>;
    /**
     * Connects to the notification callback and saves the credentials required to
     * log into it. The password is not stored into the module, only a hashed
     * copy of the credentials are saved. Remember to call the
     * saveToFlash() method of the module if the modification must be kept.
     *
     * @param username : username required to log to the callback
     * @param password : password required to log to the callback
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    callbackLogin(username: string, password: string): Promise<number>;
    /**
     * Returns the initial waiting time before first callback notifications, in seconds.
     *
     * @return an integer corresponding to the initial waiting time before first callback notifications, in seconds
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKINITIALDELAY_INVALID.
     */
    get_callbackInitialDelay(): Promise<number>;
    /**
     * Changes the initial waiting time before first callback notifications, in seconds.
     * Remember to call the saveToFlash() method of the module if the modification must be kept.
     *
     * @param newval : an integer corresponding to the initial waiting time before first callback
     * notifications, in seconds
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackInitialDelay(newval: number): Promise<number>;
    /**
     * Returns the HTTP callback schedule strategy, as a text string.
     *
     * @return a string corresponding to the HTTP callback schedule strategy, as a text string
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKSCHEDULE_INVALID.
     */
    get_callbackSchedule(): Promise<string>;
    /**
     * Changes the HTTP callback schedule strategy, as a text string.
     * Remember to call the saveToFlash()
     * method of the module if the modification must be kept.
     *
     * @param newval : a string corresponding to the HTTP callback schedule strategy, as a text string
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackSchedule(newval: string): Promise<number>;
    /**
     * Returns the minimum waiting time between two HTTP callbacks, in seconds.
     *
     * @return an integer corresponding to the minimum waiting time between two HTTP callbacks, in seconds
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKMINDELAY_INVALID.
     */
    get_callbackMinDelay(): Promise<number>;
    /**
     * Changes the minimum waiting time between two HTTP callbacks, in seconds.
     * Remember to call the saveToFlash() method of the module if the modification must be kept.
     *
     * @param newval : an integer corresponding to the minimum waiting time between two HTTP callbacks, in seconds
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackMinDelay(newval: number): Promise<number>;
    /**
     * Returns the waiting time between two HTTP callbacks when there is nothing new.
     *
     * @return an integer corresponding to the waiting time between two HTTP callbacks when there is nothing new
     *
     * On failure, throws an exception or returns YNetwork.CALLBACKMAXDELAY_INVALID.
     */
    get_callbackMaxDelay(): Promise<number>;
    /**
     * Changes the waiting time between two HTTP callbacks when there is nothing new.
     * Remember to call the saveToFlash() method of the module if the modification must be kept.
     *
     * @param newval : an integer corresponding to the waiting time between two HTTP callbacks when there
     * is nothing new
     *
     * @return YAPI.SUCCESS if the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_callbackMaxDelay(newval: number): Promise<number>;
    /**
     * Returns the current consumed by the module from Power-over-Ethernet (PoE), in milliamps.
     * The current consumption is measured after converting PoE source to 5 Volt, and should
     * never exceed 1800 mA.
     *
     * @return an integer corresponding to the current consumed by the module from Power-over-Ethernet
     * (PoE), in milliamps
     *
     * On failure, throws an exception or returns YNetwork.POECURRENT_INVALID.
     */
    get_poeCurrent(): Promise<number>;
    /**
     * Retrieves a network interface for a given identifier.
     * The identifier can be specified using several formats:
     *
     * - FunctionLogicalName
     * - ModuleSerialNumber.FunctionIdentifier
     * - ModuleSerialNumber.FunctionLogicalName
     * - ModuleLogicalName.FunctionIdentifier
     * - ModuleLogicalName.FunctionLogicalName
     *
     *
     * This function does not require that the network interface is online at the time
     * it is invoked. The returned object is nevertheless valid.
     * Use the method YNetwork.isOnline() to test if the network interface is
     * indeed online at a given time. In case of ambiguity when looking for
     * a network interface by logical name, no error is notified: the first instance
     * found is returned. The search is performed first by hardware name,
     * then by logical name.
     *
     * If a call to this object's is_online() method returns FALSE although
     * you are certain that the matching device is plugged, make sure that you did
     * call registerHub() at application initialization time.
     *
     * @param func : a string that uniquely characterizes the network interface, for instance
     *         YHUBETH1.network.
     *
     * @return a YNetwork object allowing you to drive the network interface.
     */
    static FindNetwork(func: string): YNetwork;
    /**
     * Retrieves a network interface for a given identifier in a YAPI context.
     * The identifier can be specified using several formats:
     *
     * - FunctionLogicalName
     * - ModuleSerialNumber.FunctionIdentifier
     * - ModuleSerialNumber.FunctionLogicalName
     * - ModuleLogicalName.FunctionIdentifier
     * - ModuleLogicalName.FunctionLogicalName
     *
     *
     * This function does not require that the network interface is online at the time
     * it is invoked. The returned object is nevertheless valid.
     * Use the method YNetwork.isOnline() to test if the network interface is
     * indeed online at a given time. In case of ambiguity when looking for
     * a network interface by logical name, no error is notified: the first instance
     * found is returned. The search is performed first by hardware name,
     * then by logical name.
     *
     * @param yctx : a YAPI context
     * @param func : a string that uniquely characterizes the network interface, for instance
     *         YHUBETH1.network.
     *
     * @return a YNetwork object allowing you to drive the network interface.
     */
    static FindNetworkInContext(yctx: YAPIContext, func: string): YNetwork;
    /**
     * Registers the callback function that is invoked on every change of advertised value.
     * The callback is invoked only during the execution of ySleep or yHandleEvents.
     * This provides control over the time when the callback is triggered. For good responsiveness, remember to call
     * one of these two functions periodically. To unregister a callback, pass a null pointer as argument.
     *
     * @param callback : the callback function to call, or a null pointer. The callback function should take two
     *         arguments: the function object of which the value has changed, and the character string describing
     *         the new advertised value.
     * @noreturn
     */
    registerValueCallback(callback: YNetwork.ValueCallback | null): Promise<number>;
    _invokeValueCallback(value: string): Promise<number>;
    /**
     * Changes the configuration of the network interface to enable the use of an
     * IP address received from a DHCP server. Until an address is received from a DHCP
     * server, the module uses the IP parameters specified to this function.
     * Remember to call the saveToFlash() method and then to reboot the module to apply this setting.
     *
     * @param fallbackIpAddr : fallback IP address, to be used when no DHCP reply is received
     * @param fallbackSubnetMaskLen : fallback subnet mask length when no DHCP reply is received, as an
     *         integer (e.g. 24 means 255.255.255.0)
     * @param fallbackRouter : fallback router IP address, to be used when no DHCP reply is received
     *
     * @return YAPI.SUCCESS when the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    useDHCP(fallbackIpAddr: string, fallbackSubnetMaskLen: number, fallbackRouter: string): Promise<number>;
    /**
     * Changes the configuration of the network interface to enable the use of an
     * IP address received from a DHCP server. Until an address is received from a DHCP
     * server, the module uses an IP of the network 169.254.0.0/16 (APIPA).
     * Remember to call the saveToFlash() method and then to reboot the module to apply this setting.
     *
     * @return YAPI.SUCCESS when the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    useDHCPauto(): Promise<number>;
    /**
     * Changes the configuration of the network interface to use a static IP address.
     * Remember to call the saveToFlash() method and then to reboot the module to apply this setting.
     *
     * @param ipAddress : device IP address
     * @param subnetMaskLen : subnet mask length, as an integer (e.g. 24 means 255.255.255.0)
     * @param router : router IP address (default gateway)
     *
     * @return YAPI.SUCCESS when the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    useStaticIP(ipAddress: string, subnetMaskLen: number, router: string): Promise<number>;
    /**
     * Pings host to test the network connectivity. Sends four ICMP ECHO_REQUEST requests from the
     * module to the target host. This method returns a string with the result of the
     * 4 ICMP ECHO_REQUEST requests.
     *
     * @param host : the hostname or the IP address of the target
     *
     * @return a string with the result of the ping.
     */
    ping(host: string): Promise<string>;
    /**
     * Trigger an HTTP callback quickly. This function can even be called within
     * an HTTP callback, in which case the next callback will be triggered 5 seconds
     * after the end of the current callback, regardless if the minimum time between
     * callbacks configured in the device.
     *
     * @return YAPI.SUCCESS when the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    triggerCallback(): Promise<number>;
    /**
     * Set up periodic HTTP callbacks (simplified function).
     *
     * @param interval : a string representing the callback periodicity, expressed in
     *         seconds, minutes or hours, eg. "60s", "5m", "1h", "48h".
     * @param offset : an integer representing the time offset relative to the period
     *         when the callback should occur. For instance, if the periodicity is
     *         24h, an offset of 7 will make the callback occur each day at 7AM.
     *
     * @return YAPI.SUCCESS when the call succeeds.
     *
     * On failure, throws an exception or returns a negative error code.
     */
    set_periodicCallbackSchedule(interval: string, offset: number): Promise<number>;
    /**
     * Continues the enumeration of network interfaces started using yFirstNetwork().
     * Caution: You can't make any assumption about the returned network interfaces order.
     * If you want to find a specific a network interface, use Network.findNetwork()
     * and a hardwareID or a logical name.
     *
     * @return a pointer to a YNetwork object, corresponding to
     *         a network interface currently online, or a null pointer
     *         if there are no more network interfaces to enumerate.
     */
    nextNetwork(): YNetwork | null;
    /**
     * Starts the enumeration of network interfaces currently accessible.
     * Use the method YNetwork.nextNetwork() to iterate on
     * next network interfaces.
     *
     * @return a pointer to a YNetwork object, corresponding to
     *         the first network interface currently online, or a null pointer
     *         if there are none.
     */
    static FirstNetwork(): YNetwork | null;
    /**
     * Starts the enumeration of network interfaces currently accessible.
     * Use the method YNetwork.nextNetwork() to iterate on
     * next network interfaces.
     *
     * @param yctx : a YAPI context.
     *
     * @return a pointer to a YNetwork object, corresponding to
     *         the first network interface currently online, or a null pointer
     *         if there are none.
     */
    static FirstNetworkInContext(yctx: YAPIContext): YNetwork | null;
}
export declare namespace YNetwork {
    const enum READINESS {
        DOWN = 0,
        EXISTS = 1,
        LINKED = 2,
        LAN_OK = 3,
        WWW_OK = 4,
        INVALID = -1
    }
    const enum SECURITYMODE {
        UNDEFINED = 0,
        LEGACY = 1,
        MIXED = 2,
        SECURE = 3,
        INVALID = -1
    }
    const enum DISCOVERABLE {
        FALSE = 0,
        TRUE = 1,
        INVALID = -1
    }
    const enum CALLBACKMETHOD {
        POST = 0,
        GET = 1,
        PUT = 2,
        INVALID = -1
    }
    const enum CALLBACKENCODING {
        FORM = 0,
        JSON = 1,
        JSON_ARRAY = 2,
        CSV = 3,
        YOCTO_API = 4,
        JSON_NUM = 5,
        EMONCMS = 6,
        AZURE = 7,
        INFLUXDB = 8,
        MQTT = 9,
        YOCTO_API_JZON = 10,
        PRTG = 11,
        INFLUXDB_V2 = 12,
        INVALID = -1
    }
    const enum CALLBACKTEMPLATE {
        OFF = 0,
        ON = 1,
        INVALID = -1
    }
    interface ValueCallback {
        (func: YNetwork, value: string): void;
    }
}
