
/*
  This is the main entry point. Allowing the code to run both a regular program or as a service 


*/

#include <windows.h>
#include <tchar.h>
#include <strsafe.h>
#include <iostream>
#include "logs.h"
#include <algorithm>
#include "registry.h"
#include "YAudioOuputSwitcher.h"
#include <Shlwapi.h>

// inspired from
// https://docs.microsoft.com/en-us/windows/win32/services/the-complete-service-sample?redirectedfrom=MSDN

#pragma comment(lib, "advapi32.lib")

#define SVCNAME TEXT("YAudioOutputSwitcher")

SERVICE_STATUS          gSvcStatus;
SERVICE_STATUS_HANDLE   gSvcStatusHandle;
HANDLE                  ghSvcStopEvent = NULL;
BOOL                    pleaseStop = false;

VOID SvcInstall(void);
VOID  SvcDelete(void);
VOID  SvcStart(void);
VOID  SvcStop(void);
VOID WINAPI SvcCtrlHandler(DWORD);
VOID WINAPI SvcMain(DWORD, LPTSTR*);

VOID ReportSvcStatus(DWORD, DWORD, DWORD);
VOID SvcInit(DWORD, LPTSTR*);
VOID SvcReportEvent(LPTSTR);



void help(std::wstring  execname)
{
  std::cout  << "This program controls Windows default audio endpoint selection though a \n";
  std::cout << "physical device made of a YoctoHub, a Yocto-MaxiKnob and a Yocto-MiniDisplay.\n";
  std::cout << "\n";

  std::cout << "\n";
  std::wcout << L"usage: "<<execname<<L" [commands]\n";
  std::cout << "\n";
  std::cout << "available commands\n";
  std::cout << " -help : show help\n";
  std::cout << " -install : install program as a Windows service\n";
  std::cout << " -uninstall : remove service\n";
  std::cout << " -start : start service\n";
  std::cout << " -stop : stop service\n";
  std::cout << " -hub ipaddress : save Hub ip address in the registry (may require admin privileges)\n";

  std::cout << "\n";
  std::cout << "Make sure you use the -hub command at least once\n";
  std::cout << "\n";
  std::cout << "Examples\n";
  std::wcout << execname << " -hub 192.168.0.12 (save the hub address and run)\n";
  std::wcout << execname << "  (run with a previous saved address)\n";
  std::wcout << execname << " -install -start (install and run as a service) \n";
  std::wcout << execname << " -stop (stop the service)\n";
  std::wcout << execname << " -uninstall (remove the service)\n";



}

string  getIPaddress()
{
  string ipaddress = registryStorage::readIpAddress();
  if (ipaddress == "")
  {
    logs::log( "** ERROR ** No IP address set for the hub yet.");
    logs::log(L"            start the executable with  -hub  ipaddress command");
    logs::log(L"            start the executable with  -help  command for more info");

   
    return "";
  }
  logs::log("Hub ip address is set to "+ ipaddress);
  return ipaddress;
}



int __cdecl _tmain(int argc, TCHAR* argv[])
{
   
   // clear log file
   
    


    std::wstring execname = argv[0];
    execname = PathFindFileNameW(execname.c_str());
    


    std::wcout << L"Use " + execname + L" -help  for information about all options.\n";
    bool srvCmdfound = false;
    std::string ipaddress = "192.168.0.26";

    for (int i = 1; i < argc; i++)
    {
      std::wstring arg = argv[i];
      std::transform(arg.begin(), arg.end(), arg.begin(), ::toupper);
      if (arg == L"-HELP")
      {
        help(execname);
        return 0;
      }
      else
        if (arg == L"-INSTALL")
        {
          srvCmdfound = true;  SvcInstall();
        }
        else if (arg == L"-UNINSTALL")
        {
          srvCmdfound = true;  SvcDelete();;
        }
        else if (arg == L"-START")
        {
          srvCmdfound = true;   SvcStart();
        }
        else if (arg == L"-STOP")
        {
          srvCmdfound = true;   SvcStop();
        }
     else
      if  (arg == L"-HUB")
      {
        if (i >= argc - 1)
        {
          std::wcout << "\nno address after -hub command\n";
          return 0;
        }
        std::wstring address = argv[++i];
        std::string addr(address.length(), ' ');
        std::copy(address.begin(), address.end(), addr.begin());
        try
         {
          std::cout << "storing " << addr << " IP address \n";
          registryStorage::writeIpAddress(addr);
         }
        catch (std::invalid_argument& e)
        {
          std::cout << e.what() << std::endl;
        }
    }
      else
      {
        std::wcout << "\nUnknown command : " << arg << "\n";
        std::wcout << "use " << execname << " -help for available command list\n";
        return 0;

      }

    }
    if (srvCmdfound) return 0;
    logs::initLogs();

    logs::log("Running in command line mode.");
    ipaddress = getIPaddress();

    if (ipaddress == "") return 0;
   
     

    // TO_DO: Add any additional services for the process to this table.
    SERVICE_TABLE_ENTRY DispatchTable[] =
    {
        { (LPWSTR)SVCNAME, (LPSERVICE_MAIN_FUNCTION)SvcMain },
        { NULL, NULL }
    };

    // This call returns when the service has stopped. 
    // The process should simply terminate when the call returns.

    if (!StartServiceCtrlDispatcher(DispatchTable))
    {
        SvcReportEvent((LPWSTR)TEXT("StartServiceCtrlDispatcher"));
    }

    YAudioOuputSwitcher* it = new YAudioOuputSwitcher();
    it->init(ipaddress);
    logs::log("Starting...");

    while (1)  it->Run();

   

    return 0;
}

//
// Purpose: 
//   Installs a service in the SCM database
//
// Parameters:
//   None
// 
// Return value:
//   None
//
VOID SvcInstall()
{
    SC_HANDLE schSCManager;
    SC_HANDLE schService;
    TCHAR szPath[MAX_PATH];

    if (!GetModuleFileName(NULL, szPath, MAX_PATH))
    {
        logs::logErrorOnConsole("Cannot install service" );
        return;
    }

    // Get a handle to the SCM database. 

    schSCManager = OpenSCManager(
        NULL,                    // local computer
        NULL,                    // ServicesActive database 
        SC_MANAGER_ALL_ACCESS);  // full access rights 

    if (NULL == schSCManager)
    {
        logs::logErrorOnConsole( "OpenSCManager failed");
        return;
    }

    // Create the service

    schService = CreateService(
        schSCManager,              // SCM database 
        SVCNAME,                   // name of service 
        SVCNAME,                   // service name to display 
        SERVICE_ALL_ACCESS,        // desired access 
        SERVICE_WIN32_OWN_PROCESS, // service type 
        SERVICE_AUTO_START,        // start type 
        SERVICE_ERROR_NORMAL,      // error control type 
        szPath,                    // path to service's binary 
        NULL,                      // no load ordering group 
        NULL,                      // no tag identifier 
        NULL,                      // no dependencies 
        NULL,                      // LocalSystem account 
        NULL);                     // no password 

    if (schService == NULL)
    {
        logs::logErrorOnConsole("CreateService failed" );
        CloseServiceHandle(schSCManager);
        return;
    }
    else
    {
      std::wstring  desc = L"Control Windows default audio endpoint through a physical device based on Yoctopuce hardware.";
      SERVICE_DESCRIPTION description = { (LPWSTR) desc.c_str() };
      ChangeServiceConfig2(schService, SERVICE_CONFIG_DESCRIPTION, &description);

     
      
      logs::logOnConsole("Service installed successfully\n");
    }
    CloseServiceHandle(schService);
    CloseServiceHandle(schSCManager);
}

VOID  SvcDelete(void)
{
    SC_HANDLE schSCManager;
    SC_HANDLE schService;
   // SERVICE_STATUS ssStatus;

    // Get a handle to the SCM database. 

    schSCManager = OpenSCManager(
        NULL,                    // local computer
        NULL,                    // ServicesActive database 
        SC_MANAGER_ALL_ACCESS);  // full access rights 

    if (NULL == schSCManager)
    {
        logs::logErrorOnConsole("OpenSCManager failed ");
        return;
    }

    // Get a handle to the service.

    schService = OpenService(
        schSCManager,       // SCM database 
        SVCNAME,          // name of service 
        DELETE);            // need delete access 

    if (schService == NULL)
    {
        logs::logErrorOnConsole("OpenService failed");
        CloseServiceHandle(schSCManager);
        return;
    }

    // Delete the service.

    if (!DeleteService(schService))
    {
      logs::logErrorOnConsole("DeleteService failed");
    }
    else logs::logOnConsole("Service deleted successfully");

    CloseServiceHandle(schService);
    CloseServiceHandle(schSCManager);
}

//
// Purpose: 
//   Starts the service if possible.
//
// Parameters:
//   None
// 
// Return value:
//   None
//
VOID  SvcStart(void)
{
    SERVICE_STATUS_PROCESS ssStatus;
    DWORD dwOldCheckPoint;
    DWORD dwStartTickCount;
    DWORD dwWaitTime;
    DWORD dwBytesNeeded;
    SC_HANDLE schSCManager;
    SC_HANDLE schService;

    // Get a handle to the SCM database. 

    schSCManager = OpenSCManager(
        NULL,                    // local computer
        NULL,                    // servicesActive database 
        SC_MANAGER_ALL_ACCESS);  // full access rights 

    if (NULL == schSCManager)
    {
       logs::logErrorOnConsole("OpenSCManager failed");
        return;
    }

    // Get a handle to the service.

    schService = OpenService(
        schSCManager,         // SCM database 
        SVCNAME,            // name of service 
        SERVICE_ALL_ACCESS);  // full access 

    if (schService == NULL)
    {
      logs::logErrorOnConsole("OpenService failed");
        CloseServiceHandle(schSCManager);
        return;
    }

    // Check the status in case the service is not stopped. 

    if (!QueryServiceStatusEx(
        schService,                     // handle to service 
        SC_STATUS_PROCESS_INFO,         // information level
        (LPBYTE)&ssStatus,             // address of structure
        sizeof(SERVICE_STATUS_PROCESS), // size of structure
        &dwBytesNeeded))              // size needed if buffer is too small
    {
        logs::logErrorOnConsole("QueryServiceStatusEx failed\n");
        CloseServiceHandle(schService);
        CloseServiceHandle(schSCManager);
        return;
    }

    // Check if the service is already running. It would be possible 
    // to stop the service here, but for simplicity this example just returns. 

    if (ssStatus.dwCurrentState != SERVICE_STOPPED && ssStatus.dwCurrentState != SERVICE_STOP_PENDING)
    {
        logs::logOnConsole("Cannot start the service because it is already running");
        CloseServiceHandle(schService);
        CloseServiceHandle(schSCManager);
        return;
    }

    // Save the tick count and initial checkpoint.

    dwStartTickCount = GetTickCount();
    dwOldCheckPoint = ssStatus.dwCheckPoint;

    // Wait for the service to stop before attempting to start it.

    while (ssStatus.dwCurrentState == SERVICE_STOP_PENDING)
    {
        // Do not wait longer than the wait hint. A good interval is 
        // one-tenth of the wait hint but not less than 1 second  
        // and not more than 10 seconds. 

        dwWaitTime = ssStatus.dwWaitHint / 10;

        if (dwWaitTime < 1000)
            dwWaitTime = 1000;
        else if (dwWaitTime > 10000)
            dwWaitTime = 10000;

        Sleep(dwWaitTime);

        // Check the status until the service is no longer stop pending. 

        if (!QueryServiceStatusEx(
            schService,                     // handle to service 
            SC_STATUS_PROCESS_INFO,         // information level
            (LPBYTE)&ssStatus,             // address of structure
            sizeof(SERVICE_STATUS_PROCESS), // size of structure
            &dwBytesNeeded))              // size needed if buffer is too small
        {
            logs::logErrorOnConsole("QueryServiceStatusEx failed");
            CloseServiceHandle(schService);
            CloseServiceHandle(schSCManager);
            return;
        }

        if (ssStatus.dwCheckPoint > dwOldCheckPoint)
        {
            // Continue to wait and check.

            dwStartTickCount = GetTickCount();
            dwOldCheckPoint = ssStatus.dwCheckPoint;
        }
        else
        {
            if (GetTickCount() - dwStartTickCount > ssStatus.dwWaitHint)
            {
                logs::logOnConsole("Timeout waiting for service to stop");
                CloseServiceHandle(schService);
                CloseServiceHandle(schSCManager);
                return;
            }
        }
    }

    // Attempt to start the service.

    if (!StartService(
        schService,  // handle to service 
        0,           // number of arguments 
        NULL))      // no arguments 
    {
        logs::logErrorOnConsole("StartService failed");
        CloseServiceHandle(schService);
        CloseServiceHandle(schSCManager);
        return;
    }
    else logs::logOnConsole("Service start pending...");

    // Check the status until the service is no longer start pending. 

    if (!QueryServiceStatusEx(
        schService,                     // handle to service 
        SC_STATUS_PROCESS_INFO,         // info level
        (LPBYTE)&ssStatus,             // address of structure
        sizeof(SERVICE_STATUS_PROCESS), // size of structure
        &dwBytesNeeded))              // if buffer too small
    {
        logs::logErrorOnConsole("QueryServiceStatusEx failed");
        CloseServiceHandle(schService);
        CloseServiceHandle(schSCManager);
        return;
    }

    // Save the tick count and initial checkpoint.

    dwStartTickCount = GetTickCount();
    dwOldCheckPoint = ssStatus.dwCheckPoint;

    while (ssStatus.dwCurrentState == SERVICE_START_PENDING)
    {
        // Do not wait longer than the wait hint. A good interval is 
        // one-tenth the wait hint, but no less than 1 second and no 
        // more than 10 seconds. 

        dwWaitTime = ssStatus.dwWaitHint / 10;

        if (dwWaitTime < 1000)
            dwWaitTime = 1000;
        else if (dwWaitTime > 10000)
            dwWaitTime = 10000;

        Sleep(dwWaitTime);

        // Check the status again. 

        if (!QueryServiceStatusEx(
            schService,             // handle to service 
            SC_STATUS_PROCESS_INFO, // info level
            (LPBYTE)&ssStatus,             // address of structure
            sizeof(SERVICE_STATUS_PROCESS), // size of structure
            &dwBytesNeeded))              // if buffer too small
        {
            logs::logErrorOnConsole("QueryServiceStatusEx failed");
            break;
        }

        if (ssStatus.dwCheckPoint > dwOldCheckPoint)
        {
            // Continue to wait and check.

            dwStartTickCount = GetTickCount();
            dwOldCheckPoint = ssStatus.dwCheckPoint;
        }
        else
        {
            if (GetTickCount() - dwStartTickCount > ssStatus.dwWaitHint)
            {
                // No progress made within the wait hint.
                break;
            }
        }
    }

    // Determine whether the service is running.

    if (ssStatus.dwCurrentState == SERVICE_RUNNING)
    {
       logs::logOnConsole("Service started successfully.");
    }
    else
    {
      logs::logOnConsole("Service not started. ");
      logs::logOnConsole("  Current State:"+std::to_string(ssStatus.dwCurrentState));
      logs::logOnConsole("  Exit Code:" + std::to_string(ssStatus.dwWin32ExitCode));
      logs::logOnConsole("  Check Point:" + std::to_string(ssStatus.dwCheckPoint));
      logs::logOnConsole("  Wait Hint: " + std::to_string(ssStatus.dwWaitHint));
    }

    CloseServiceHandle(schService);
    CloseServiceHandle(schSCManager);
}

//
// Purpose: 
//   Stops the service.
//
// Parameters:
//   None
// 
// Return value:
//   None
//
VOID  SvcStop(void)
{
    SERVICE_STATUS_PROCESS ssp;
    DWORD dwStartTime = GetTickCount();
    DWORD dwBytesNeeded;
    DWORD dwTimeout = 30000; // 30-second time-out
    DWORD dwWaitTime;
    SC_HANDLE schSCManager;
    SC_HANDLE schService;

    // Get a handle to the SCM database. 

    schSCManager = OpenSCManager(
        NULL,                    // local computer
        NULL,                    // ServicesActive database 
        SC_MANAGER_ALL_ACCESS);  // full access rights 

    if (NULL == schSCManager)
    {
      logs::logErrorOnConsole("OpenSCManager failed");
        return;
    }

    // Get a handle to the service.

    schService = OpenService(
        schSCManager,         // SCM database 
        SVCNAME,              // name of service 
        SERVICE_STOP |
        SERVICE_QUERY_STATUS |
        SERVICE_ENUMERATE_DEPENDENTS);

    if (schService == NULL)
    {
      logs::logErrorOnConsole("OpenService failed");
        CloseServiceHandle(schSCManager);
        return;
    }

    // Make sure the service is not already stopped.

    if (!QueryServiceStatusEx(
        schService,
        SC_STATUS_PROCESS_INFO,
        (LPBYTE)&ssp,
        sizeof(SERVICE_STATUS_PROCESS),
        &dwBytesNeeded))
    {
        logs::logErrorOnConsole("QueryServiceStatusEx failed");
        goto stop_cleanup;
    }

    if (ssp.dwCurrentState == SERVICE_STOPPED)
    {
      logs::logOnConsole("Service is already stopped");
        goto stop_cleanup;
    }

    // If a stop is pending, wait for it.

    while (ssp.dwCurrentState == SERVICE_STOP_PENDING)
    {
        logs::logOnConsole("Service stop pending...");

        // Do not wait longer than the wait hint. A good interval is 
        // one-tenth of the wait hint but not less than 1 second  
        // and not more than 10 seconds. 

        dwWaitTime = ssp.dwWaitHint / 10;

        if (dwWaitTime < 1000)
            dwWaitTime = 1000;
        else if (dwWaitTime > 10000)
            dwWaitTime = 10000;

        Sleep(dwWaitTime);

        if (!QueryServiceStatusEx(
            schService,
            SC_STATUS_PROCESS_INFO,
            (LPBYTE)&ssp,
            sizeof(SERVICE_STATUS_PROCESS),
            &dwBytesNeeded))
        {
          logs::logErrorOnConsole("QueryServiceStatusEx failed");
            goto stop_cleanup;
        }

        if (ssp.dwCurrentState == SERVICE_STOPPED)
        {
            logs::logOnConsole("Service stopped successfully.");
            goto stop_cleanup;
        }

        if (GetTickCount() - dwStartTime > dwTimeout)
        {
           logs::logOnConsole("Service stop timed out.");
            goto stop_cleanup;
        }
    }

    // If the service is running, dependencies must be stopped first.

    //StopDependentServices();  NO NEED

    // Send a stop code to the service.

    if (!ControlService(
        schService,
        SERVICE_CONTROL_STOP,
        (LPSERVICE_STATUS)&ssp))
    {
      logs::logErrorOnConsole("ControlService failed");
        goto stop_cleanup;
    }

    // Wait for the service to stop.

    while (ssp.dwCurrentState != SERVICE_STOPPED)
    {
        Sleep(ssp.dwWaitHint);
        if (!QueryServiceStatusEx(
            schService,
            SC_STATUS_PROCESS_INFO,
            (LPBYTE)&ssp,
            sizeof(SERVICE_STATUS_PROCESS),
            &dwBytesNeeded))
        {
            logs::logErrorOnConsole("QueryServiceStatusEx failed");
            goto stop_cleanup;
        }

        if (ssp.dwCurrentState == SERVICE_STOPPED)
            break;

        if (GetTickCount() - dwStartTime > dwTimeout)
        {
            logs::logOnConsole("Wait timed out");
            goto stop_cleanup;
        }
    }
    logs::logOnConsole("Service stopped successfully");

stop_cleanup:
    CloseServiceHandle(schService);
    CloseServiceHandle(schSCManager);
}


//
// Purpose: 
//   Entry point for the service
//
// Parameters:
//   dwArgc - Number of arguments in the lpszArgv array
//   lpszArgv - Array of strings. The first string is the name of
//     the service and subsequent strings are passed by the process
//     that called the StartService function to start the service.
// 
// Return value:
//   None.
//
VOID WINAPI SvcMain(DWORD dwArgc, LPTSTR* lpszArgv)
{
    // Register the handler function for the service

    gSvcStatusHandle = RegisterServiceCtrlHandler(
        SVCNAME,
        SvcCtrlHandler);

    if (!gSvcStatusHandle)
    {
        SvcReportEvent((LPWSTR)TEXT("RegisterServiceCtrlHandler"));
        return;
    }

    // These SERVICE_STATUS members remain as set here

    gSvcStatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
    gSvcStatus.dwServiceSpecificExitCode = 0;

    // Report initial status to the SCM

    ReportSvcStatus(SERVICE_START_PENDING, NO_ERROR, 3000);

    // Perform service-specific initialization and work.

    SvcInit(dwArgc, lpszArgv);
}

//
// Purpose: 
//   The service code
//
// Parameters:
//   dwArgc - Number of arguments in the lpszArgv array
//   lpszArgv - Array of strings. The first string is the name of
//     the service and subsequent strings are passed by the process
//     that called the StartService function to start the service.
// 
// Return value:
//   None
//
VOID SvcInit(DWORD dwArgc, LPTSTR* lpszArgv)
{
    // TO_DO: Declare and set any required variables.
    //   Be sure to periodically call ReportSvcStatus() with 
    //   SERVICE_START_PENDING. If initialization fails, call
    //   ReportSvcStatus with SERVICE_STOPPED.

    // Create an event. The control handler function, SvcCtrlHandler,
    // signals this event when it receives the stop control code.

    ghSvcStopEvent = CreateEvent(
        NULL,    // default security attributes
        TRUE,    // manual reset event
        FALSE,   // not signaled
        NULL);   // no name

    if (ghSvcStopEvent == NULL)
    {
        ReportSvcStatus(SERVICE_STOPPED, GetLastError(), 0);
        return;
    }

    // Report running status when initialization is complete.

    ReportSvcStatus(SERVICE_RUNNING, NO_ERROR, 0);

    // TO_DO: Perform work until service stops.

    logs::initLogs();
    logs::log("Running in service mode.");
    string ipaddress = getIPaddress();

    if (ipaddress == "")
    {
      ReportSvcStatus(SERVICE_STOPPED, ERROR_INVALID_DATA, 0);
      return;
    }

    YAudioOuputSwitcher* it = new YAudioOuputSwitcher();
    it->init(ipaddress);
  
    DWORD state;
    do
    {
      it->Run();
       state = WaitForSingleObject(ghSvcStopEvent, 1);
    } while (state != WAIT_OBJECT_0);


    logs::log("Stopping execution");
    delete it;
    ReportSvcStatus(SERVICE_STOPPED, NO_ERROR, 0);
       
    
}

//
// Purpose: 
//   Sets the current service status and reports it to the SCM.
//
// Parameters:
//   dwCurrentState - The current state (see SERVICE_STATUS)
//   dwWin32ExitCode - The system error code
//   dwWaitHint - Estimated time for pending operation, 
//     in milliseconds
// 
// Return value:
//   None
//
VOID ReportSvcStatus(DWORD dwCurrentState,
    DWORD dwWin32ExitCode,
    DWORD dwWaitHint)
{
    static DWORD dwCheckPoint = 1;

    // Fill in the SERVICE_STATUS structure.

    gSvcStatus.dwCurrentState = dwCurrentState;
    gSvcStatus.dwWin32ExitCode = dwWin32ExitCode;
    gSvcStatus.dwWaitHint = dwWaitHint;

    if (dwCurrentState == SERVICE_START_PENDING)
        gSvcStatus.dwControlsAccepted = 0;
    else gSvcStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP;

    if ((dwCurrentState == SERVICE_RUNNING) ||
        (dwCurrentState == SERVICE_STOPPED))
        gSvcStatus.dwCheckPoint = 0;
    else gSvcStatus.dwCheckPoint = dwCheckPoint++;

    // Report the status of the service to the SCM.
    SetServiceStatus(gSvcStatusHandle, &gSvcStatus);
}

//
// Purpose: 
//   Called by SCM whenever a control code is sent to the service
//   using the ControlService function.
//
// Parameters:
//   dwCtrl - control code
// 
// Return value:
//   None
//
VOID WINAPI SvcCtrlHandler(DWORD dwCtrl)
{
    // Handle the requested control code. 

    switch (dwCtrl)
    {
    case SERVICE_CONTROL_STOP:
        ReportSvcStatus(SERVICE_STOP_PENDING, NO_ERROR, 0);

        // Signal the service to stop.
       
        SetEvent(ghSvcStopEvent);
        ReportSvcStatus(gSvcStatus.dwCurrentState, NO_ERROR, 0);

        return;

    case SERVICE_CONTROL_INTERROGATE:
        break;

    default:
        break;
    }

}

//
// Purpose: 
//   Logs messages to the event log
//
// Parameters:
//   szFunction - name of function that failed
// 
// Return value:
//   None
//
// Remarks:
//   The service must have an entry in the Application event log.
//
VOID SvcReportEvent(LPTSTR szFunction)
{
    HANDLE hEventSource;
    LPCTSTR lpszStrings[2];
    TCHAR Buffer[80];

    hEventSource = RegisterEventSource(NULL, SVCNAME);

    if (NULL != hEventSource)
    {
        StringCchPrintf(Buffer, 80, TEXT("%s failed with %d"), szFunction, GetLastError());

        lpszStrings[0] = SVCNAME;
        lpszStrings[1] = Buffer;
        /*

        ReportEvent(hEventSource,        // event log handle
            EVENTLOG_ERROR_TYPE, // event type
            0,                   // event category
            SVC_ERROR,           // event identifier
            NULL,                // no security identifier
            2,                   // size of lpszStrings array
            0,                   // no binary data
            lpszStrings,         // array of strings
            NULL);               // no binary data

        DeregisterEventSource(hEventSource);
        */
    }
}