/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle Threshold functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YThreshold.h"
#include "yocto_api.h"
#include "yocto_threshold.h"

using namespace std;

//--- (YThreshold definitions)

static const char *enumThresholdState[] = {
    "SAFE",
    "ALERT",
    NULL
};

//--- (end of YThreshold definitions)


//--- (YThreshold implementation)
bool   YThresholdCmdLine::isModule(void)      { return false; }
string YThresholdCmdLine::YFunctionName(void) { return "YThreshold"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of thresholdState is int (ThresholdState)
// type of targetSensor is string (Text)
// type of alertLevel is double (MeasureVal)
// type of safeLevel is double (MeasureVal)
/**
 * Returns the logical name of the threshold function.
 *
 * @return a string corresponding to the logical name of the threshold function
 *
 * On failure, throws an exception or returns YThreshold.LOGICALNAME_INVALID.
 */
class Threshold_get_logicalName : public YapiCommand /* arguments: */
{
public:
  Threshold_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the threshold function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the threshold function. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the threshold function
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Threshold_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  Threshold_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the threshold function.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the threshold function", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the threshold function.
 *
 * @return a string corresponding to a short string representing the current state of the threshold function
 *
 * On failure, throws an exception or returns YThreshold.ADVERTISEDVALUE_INVALID.
 */
class Threshold_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  Threshold_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the threshold function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns current state of the threshold function.
 *
 * @return either YThreshold.THRESHOLDSTATE_SAFE or YThreshold.THRESHOLDSTATE_ALERT, according to
 * current state of the threshold function
 *
 * On failure, throws an exception or returns YThreshold.THRESHOLDSTATE_INVALID.
 */
class Threshold_get_thresholdState : public YapiCommand /* arguments: */
{
public:
  Threshold_get_thresholdState(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_thresholdState";
  }

  string getDescription()
  {
    return "Returns current state of the threshold function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_thresholdState();
        string strval =  enumString(attrval,enumThresholdState);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of the sensor monitored by the threshold function.
 *
 * @return a string corresponding to the name of the sensor monitored by the threshold function
 *
 * On failure, throws an exception or returns YThreshold.TARGETSENSOR_INVALID.
 */
class Threshold_get_targetSensor : public YapiCommand /* arguments: */
{
public:
  Threshold_get_targetSensor(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_targetSensor";
  }

  string getDescription()
  {
    return "Returns the name of the sensor monitored by the threshold function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_targetSensor();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the sensor alert level triggering the threshold function.
 * Remember to call the matching module saveToFlash()
 * method if you want to preserve the setting after reboot.
 *
 * @param newval : a floating point number corresponding to the sensor alert level triggering the
 * threshold function
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Threshold_set_alertLevel : public YapiCommand /* arguments: newval */
{
public:
  Threshold_set_alertLevel(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_alertLevel";
  }

  string getDescription()
  {
    return "Changes the sensor alert level triggering the threshold function.";
  }

  string getMoreInfo()
  {
    return "Remember to call the matching module saveToFlash() method if you want to preserve the setting after reboot.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(DOUBLE_ARG, "newval", "a floating point number corresponding to the sensor alert level triggering the threshold function", "_DOUBLE", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_alertLevel(double((*args)[0]->get_doubleValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the sensor alert level, triggering the threshold function.
 *
 * @return a floating point number corresponding to the sensor alert level, triggering the threshold function
 *
 * On failure, throws an exception or returns YThreshold.ALERTLEVEL_INVALID.
 */
class Threshold_get_alertLevel : public YapiCommand /* arguments: */
{
public:
  Threshold_get_alertLevel(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_alertLevel";
  }

  string getDescription()
  {
    return "Returns the sensor alert level, triggering the threshold function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        double attrval = (*list)[i]->get_alertLevel();
        char buf[32]; SAFE_SPRINTF(buf,32,"%.3f",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the sensor acceptable level for disabling the threshold function.
 * Remember to call the matching module saveToFlash()
 * method if you want to preserve the setting after reboot.
 *
 * @param newval : a floating point number corresponding to the sensor acceptable level for disabling
 * the threshold function
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class Threshold_set_safeLevel : public YapiCommand /* arguments: newval */
{
public:
  Threshold_set_safeLevel(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_safeLevel";
  }

  string getDescription()
  {
    return "Changes the sensor acceptable level for disabling the threshold function.";
  }

  string getMoreInfo()
  {
    return "Remember to call the matching module saveToFlash() method if you want to preserve the setting after reboot.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(DOUBLE_ARG, "newval", "a floating point number corresponding to the sensor acceptable level for disabling the threshold function", "_DOUBLE", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_safeLevel(double((*args)[0]->get_doubleValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the sensor acceptable level for disabling the threshold function.
 *
 * @return a floating point number corresponding to the sensor acceptable level for disabling the
 * threshold function
 *
 * On failure, throws an exception or returns YThreshold.SAFELEVEL_INVALID.
 */
class Threshold_get_safeLevel : public YapiCommand /* arguments: */
{
public:
  Threshold_get_safeLevel(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_safeLevel";
  }

  string getDescription()
  {
    return "Returns the sensor acceptable level for disabling the threshold function.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        double attrval = (*list)[i]->get_safeLevel();
        char buf[32]; SAFE_SPRINTF(buf,32,"%.3f",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

// type of valueCallbackThreshold is YThresholdValueCallback* (YThresholdValueCallback)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_Threshold_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_Threshold_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_Threshold_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_Threshold_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_Threshold_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_Threshold_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_Threshold_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_Threshold_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YThreshold*>* list = enumerateTargets<YThreshold>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YThreshold implementation)

//--- (YThreshold functions)
void YThresholdCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YThreshold>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new Threshold_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_get_thresholdState(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_get_targetSensor(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_set_alertLevel(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_get_alertLevel(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_set_safeLevel(this)));
    cmdList->push_back((YapiCommand*) (new Threshold_get_safeLevel(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Threshold_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Threshold_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Threshold_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_Threshold_get_serialNumber(this)));
  }

//--- (end of YThreshold functions)
