/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle InputChain functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YInputChain.h"
#include "yocto_api.h"
#include "yocto_inputchain.h"

using namespace std;

//--- (YInputChain definitions)

static const char *enumOnOff[] = {
    "OFF",
    "ON",
    NULL
};

//--- (end of YInputChain definitions)


//--- (YInputChain implementation)
bool   YInputChainCmdLine::isModule(void)      { return false; }
string YInputChainCmdLine::YFunctionName(void) { return "YInputChain"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of expectedNodes is int (UInt31)
// type of detectedNodes is int (UInt31)
// type of loopbackTest is int (OnOff)
// type of refreshRate is int (UInt31)
// type of bitChain1 is string (Text)
// type of bitChain2 is string (Text)
// type of bitChain3 is string (Text)
// type of bitChain4 is string (Text)
// type of bitChain5 is string (Text)
// type of bitChain6 is string (Text)
// type of bitChain7 is string (Text)
// type of watchdogPeriod is int (UInt31)
// type of chainDiags is int (InputChainDiags)
/**
 * Returns the logical name of the digital input chain.
 *
 * @return a string corresponding to the logical name of the digital input chain
 *
 * On failure, throws an exception or returns YInputChain.LOGICALNAME_INVALID.
 */
class InputChain_get_logicalName : public YapiCommand /* arguments: */
{
public:
  InputChain_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the digital input chain.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the digital input chain. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the digital input chain
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class InputChain_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  InputChain_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the digital input chain.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the digital input chain", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the digital input chain.
 *
 * @return a string corresponding to a short string representing the current state of the digital input chain
 *
 * On failure, throws an exception or returns YInputChain.ADVERTISEDVALUE_INVALID.
 */
class InputChain_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  InputChain_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the digital input chain.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the number of nodes expected in the chain.
 *
 * @return an integer corresponding to the number of nodes expected in the chain
 *
 * On failure, throws an exception or returns YInputChain.EXPECTEDNODES_INVALID.
 */
class InputChain_get_expectedNodes : public YapiCommand /* arguments: */
{
public:
  InputChain_get_expectedNodes(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_expectedNodes";
  }

  string getDescription()
  {
    return "Returns the number of nodes expected in the chain.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_expectedNodes();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the number of nodes expected in the chain.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : an integer corresponding to the number of nodes expected in the chain
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class InputChain_set_expectedNodes : public YapiCommand /* arguments: newval */
{
public:
  InputChain_set_expectedNodes(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_expectedNodes";
  }

  string getDescription()
  {
    return "Changes the number of nodes expected in the chain.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the number of nodes expected in the chain", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_expectedNodes(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the number of nodes detected in the chain.
 *
 * @return an integer corresponding to the number of nodes detected in the chain
 *
 * On failure, throws an exception or returns YInputChain.DETECTEDNODES_INVALID.
 */
class InputChain_get_detectedNodes : public YapiCommand /* arguments: */
{
public:
  InputChain_get_detectedNodes(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_detectedNodes";
  }

  string getDescription()
  {
    return "Returns the number of nodes detected in the chain.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_detectedNodes();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the activation state of the exhaustive chain connectivity test.
 * The connectivity test requires a cable connecting the end of the chain
 * to the loopback test connector.
 *
 * @return either YInputChain.LOOPBACKTEST_OFF or YInputChain.LOOPBACKTEST_ON, according to the
 * activation state of the exhaustive chain connectivity test
 *
 * On failure, throws an exception or returns YInputChain.LOOPBACKTEST_INVALID.
 */
class InputChain_get_loopbackTest : public YapiCommand /* arguments: */
{
public:
  InputChain_get_loopbackTest(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_loopbackTest";
  }

  string getDescription()
  {
    return "Returns the activation state of the exhaustive chain connectivity test.";
  }

  string getMoreInfo()
  {
    return "The connectivity test requires a cable connecting the end of the chain to the loopback test connector.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_loopbackTest();
        string strval =  enumString(attrval,enumOnOff);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the activation state of the exhaustive chain connectivity test.
 * The connectivity test requires a cable connecting the end of the chain
 * to the loopback test connector.
 *
 * If you want the change to be kept after a device reboot,
 * make sure  to call the matching module saveToFlash().
 *
 * @param newval : either YInputChain.LOOPBACKTEST_OFF or YInputChain.LOOPBACKTEST_ON, according to
 * the activation state of the exhaustive chain connectivity test
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class InputChain_set_loopbackTest : public YapiCommand /* arguments: newval */
{
public:
  InputChain_set_loopbackTest(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_loopbackTest";
  }

  string getDescription()
  {
    return "Changes the activation state of the exhaustive chain connectivity test.";
  }

  string getMoreInfo()
  {
    return "The connectivity test requires a cable connecting the end of the chain to the loopback test connector.\nIf you want the change to be kept after a device reboot, make sure to call the matching module saveToFlash().";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either OFF or ON, according to the activation state of the exhaustive chain connectivity test", "OFF=0,ON=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_loopbackTest(Y_LOOPBACKTEST_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the desired refresh rate, measured in Hz.
 * The higher the refresh rate is set, the higher the
 * communication speed on the chain will be.
 *
 * @return an integer corresponding to the desired refresh rate, measured in Hz
 *
 * On failure, throws an exception or returns YInputChain.REFRESHRATE_INVALID.
 */
class InputChain_get_refreshRate : public YapiCommand /* arguments: */
{
public:
  InputChain_get_refreshRate(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_refreshRate";
  }

  string getDescription()
  {
    return "Returns the desired refresh rate, measured in Hz.";
  }

  string getMoreInfo()
  {
    return "The higher the refresh rate is set, the higher the communication speed on the chain will be.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_refreshRate();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the desired refresh rate, measured in Hz.
 * The higher the refresh rate is set, the higher the
 * communication speed on the chain will be.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : an integer corresponding to the desired refresh rate, measured in Hz
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class InputChain_set_refreshRate : public YapiCommand /* arguments: newval */
{
public:
  InputChain_set_refreshRate(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_refreshRate";
  }

  string getDescription()
  {
    return "Changes the desired refresh rate, measured in Hz.";
  }

  string getMoreInfo()
  {
    return "The higher the refresh rate is set, the higher the communication speed on the chain will be. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the desired refresh rate, measured in Hz", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_refreshRate(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the state of input 1 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 1 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN1_INVALID.
 */
class InputChain_get_bitChain1 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain1(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain1";
  }

  string getDescription()
  {
    return "Returns the state of input 1 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain1();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of input 2 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 2 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN2_INVALID.
 */
class InputChain_get_bitChain2 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain2(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain2";
  }

  string getDescription()
  {
    return "Returns the state of input 2 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain2();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of input 3 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 3 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN3_INVALID.
 */
class InputChain_get_bitChain3 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain3(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain3";
  }

  string getDescription()
  {
    return "Returns the state of input 3 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain3();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of input 4 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 4 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN4_INVALID.
 */
class InputChain_get_bitChain4 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain4(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain4";
  }

  string getDescription()
  {
    return "Returns the state of input 4 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain4();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of input 5 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 5 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN5_INVALID.
 */
class InputChain_get_bitChain5 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain5(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain5";
  }

  string getDescription()
  {
    return "Returns the state of input 5 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain5();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of input 6 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 6 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN6_INVALID.
 */
class InputChain_get_bitChain6 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain6(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain6";
  }

  string getDescription()
  {
    return "Returns the state of input 6 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain6();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the state of input 7 for all nodes of the input chain,
 * as a hexadecimal string. The node nearest to the controller
 * is the lowest bit of the result.
 *
 * @return a string corresponding to the state of input 7 for all nodes of the input chain,
 *         as a hexadecimal string
 *
 * On failure, throws an exception or returns YInputChain.BITCHAIN7_INVALID.
 */
class InputChain_get_bitChain7 : public YapiCommand /* arguments: */
{
public:
  InputChain_get_bitChain7(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_bitChain7";
  }

  string getDescription()
  {
    return "Returns the state of input 7 for all nodes of the input chain,as a hexadecimal string.";
  }

  string getMoreInfo()
  {
    return "The node nearest to the controller is the lowest bit of the result.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_bitChain7();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the wait time in seconds before triggering an inactivity
 * timeout error.
 *
 * @return an integer corresponding to the wait time in seconds before triggering an inactivity
 *         timeout error
 *
 * On failure, throws an exception or returns YInputChain.WATCHDOGPERIOD_INVALID.
 */
class InputChain_get_watchdogPeriod : public YapiCommand /* arguments: */
{
public:
  InputChain_get_watchdogPeriod(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_watchdogPeriod";
  }

  string getDescription()
  {
    return "Returns the wait time in seconds before triggering an inactivitytimeout error.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_watchdogPeriod();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the wait time in seconds before triggering an inactivity
 * timeout error. Remember to call the saveToFlash() method
 * of the module if the modification must be kept.
 *
 * @param newval : an integer corresponding to the wait time in seconds before triggering an inactivity
 *         timeout error
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class InputChain_set_watchdogPeriod : public YapiCommand /* arguments: newval */
{
public:
  InputChain_set_watchdogPeriod(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_watchdogPeriod";
  }

  string getDescription()
  {
    return "Changes the wait time in seconds before triggering an inactivitytimeout error.";
  }

  string getMoreInfo()
  {
    return "Remember to call the saveToFlash() method of the module if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the wait time in seconds before triggering an inactivity timeout error", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_watchdogPeriod(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the controller state diagnostics. Bit 0 indicates a chain length
 * error, bit 1 indicates an inactivity timeout and bit 2 indicates
 * a loopback test failure.
 *
 * @return an integer corresponding to the controller state diagnostics
 *
 * On failure, throws an exception or returns YInputChain.CHAINDIAGS_INVALID.
 */
class InputChain_get_chainDiags : public YapiCommand /* arguments: */
{
public:
  InputChain_get_chainDiags(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_chainDiags";
  }

  string getDescription()
  {
    return "Returns the controller state diagnostics.";
  }

  string getMoreInfo()
  {
    return "Bit 0 indicates a chain length error, bit 1 indicates an inactivity timeout and bit 2 indicates a loopback test failure.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_chainDiags();
        string strval =  bitString("NW5CDLRT", attrval);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

// type of valueCallbackInputChain is YInputChainValueCallback* (YInputChainValueCallback)
// type of eventCallback is YEventCallback* (YEventCallback)
// type of prevPos is int (int)
// type of eventPos is int (int)
// type of eventStamp is int (int)
// type of eventChains is vector<string> (strArr)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_InputChain_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_InputChain_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_InputChain_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_InputChain_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_InputChain_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_InputChain_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_InputChain_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_InputChain_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Resets the application watchdog countdown.
 * If you have setup a non-zero watchdogPeriod, you should
 * call this function on a regular basis to prevent the application
 * inactivity error to be triggered.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_InputChain_resetWatchdog : public YapiCommand /* arguments: */
{
public:
  apifun_InputChain_resetWatchdog(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "resetWatchdog";
  }

  string getDescription()
  {
    return "Resets the application watchdog countdown.";
  }

  string getMoreInfo()
  {
    return "If you have setup a non-zero watchdogPeriod, you should call this function on a regular basis to prevent the application inactivity error to be triggered.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->resetWatchdog();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns a string with last events observed on the digital input chain.
 * This method return only events that are still buffered in the device memory.
 *
 * @return a string with last events observed (one per line).
 *
 * On failure, throws an exception or returns  YAPI_INVALID_STRING.
 */
class apifun_InputChain_get_lastEvents : public YapiCommand /* arguments: */
{
public:
  apifun_InputChain_get_lastEvents(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastEvents";
  }

  string getDescription()
  {
    return "Returns a string with last events observed on the digital input chain.";
  }

  string getMoreInfo()
  {
    return "This method return only events that are still buffered in the device memory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YInputChain*>* list = enumerateTargets<YInputChain>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_lastEvents();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YInputChain implementation)

//--- (YInputChain functions)
void YInputChainCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YInputChain>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new InputChain_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_expectedNodes(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_set_expectedNodes(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_detectedNodes(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_loopbackTest(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_set_loopbackTest(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_refreshRate(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_set_refreshRate(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain1(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain2(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain3(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain4(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain5(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain6(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_bitChain7(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_watchdogPeriod(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_set_watchdogPeriod(this)));
    cmdList->push_back((YapiCommand*) (new InputChain_get_chainDiags(this)));
    cmdList->push_back((YapiCommand*) (new apifun_InputChain_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_InputChain_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_InputChain_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_InputChain_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_InputChain_resetWatchdog(this)));
    cmdList->push_back((YapiCommand*) (new apifun_InputChain_get_lastEvents(this)));
  }

//--- (end of YInputChain functions)
