/*********************************************************************
 *
 *  $Id: YI2cPort.cpp 53689 2023-03-22 11:17:15Z mvuilleu $
 *
 *  Implements commands to handle I2cPort functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YI2cPort.h"
#include "yocto_api.h"
#include "yocto_i2cport.h"

using namespace std;

//--- (generated code: YI2cPort definitions)

static const char *enumI2cVoltageLevel[] = {
    "OFF",
    "3V3",
    "1V8",
    NULL
};

//--- (end of generated code: YI2cPort definitions)


//--- (generated code: YI2cPort implementation)
bool   YI2cPortCmdLine::isModule(void)      { return false; }
string YI2cPortCmdLine::YFunctionName(void) { return "YI2cPort"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of rxCount is int (UInt31)
// type of txCount is int (UInt31)
// type of errCount is int (UInt31)
// type of rxMsgCount is int (UInt31)
// type of txMsgCount is int (UInt31)
// type of lastMsg is string (Text)
// type of currentJob is string (Text)
// type of startupJob is string (Text)
// type of jobMaxTask is int (UInt31)
// type of jobMaxSize is int (UInt31)
// type of command is string (Text)
// type of protocol is string (Protocol)
// type of i2cVoltageLevel is int (I2cVoltageLevel)
// type of i2cMode is string (I2cMode)
/**
 * Returns the logical name of the I2C port.
 *
 * @return a string corresponding to the logical name of the I2C port
 *
 * On failure, throws an exception or returns YI2cPort.LOGICALNAME_INVALID.
 */
class I2cPort_get_logicalName : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the I2C port.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the I2C port. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the I2C port
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class I2cPort_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  I2cPort_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the I2C port.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the I2C port", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the I2C port.
 *
 * @return a string corresponding to a short string representing the current state of the I2C port
 *
 * On failure, throws an exception or returns YI2cPort.ADVERTISEDVALUE_INVALID.
 */
class I2cPort_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the I2C port.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of bytes received since last reset.
 *
 * @return an integer corresponding to the total number of bytes received since last reset
 *
 * On failure, throws an exception or returns YI2cPort.RXCOUNT_INVALID.
 */
class I2cPort_get_rxCount : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_rxCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_rxCount";
  }

  string getDescription()
  {
    return "Returns the total number of bytes received since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_rxCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of bytes transmitted since last reset.
 *
 * @return an integer corresponding to the total number of bytes transmitted since last reset
 *
 * On failure, throws an exception or returns YI2cPort.TXCOUNT_INVALID.
 */
class I2cPort_get_txCount : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_txCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_txCount";
  }

  string getDescription()
  {
    return "Returns the total number of bytes transmitted since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_txCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of communication errors detected since last reset.
 *
 * @return an integer corresponding to the total number of communication errors detected since last reset
 *
 * On failure, throws an exception or returns YI2cPort.ERRCOUNT_INVALID.
 */
class I2cPort_get_errCount : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_errCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_errCount";
  }

  string getDescription()
  {
    return "Returns the total number of communication errors detected since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_errCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of messages received since last reset.
 *
 * @return an integer corresponding to the total number of messages received since last reset
 *
 * On failure, throws an exception or returns YI2cPort.RXMSGCOUNT_INVALID.
 */
class I2cPort_get_rxMsgCount : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_rxMsgCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_rxMsgCount";
  }

  string getDescription()
  {
    return "Returns the total number of messages received since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_rxMsgCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of messages send since last reset.
 *
 * @return an integer corresponding to the total number of messages send since last reset
 *
 * On failure, throws an exception or returns YI2cPort.TXMSGCOUNT_INVALID.
 */
class I2cPort_get_txMsgCount : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_txMsgCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_txMsgCount";
  }

  string getDescription()
  {
    return "Returns the total number of messages send since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_txMsgCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the latest message fully received (for Line and Frame protocols).
 *
 * @return a string corresponding to the latest message fully received (for Line and Frame protocols)
 *
 * On failure, throws an exception or returns YI2cPort.LASTMSG_INVALID.
 */
class I2cPort_get_lastMsg : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_lastMsg(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastMsg";
  }

  string getDescription()
  {
    return "Returns the latest message fully received (for Line and Frame protocols).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_lastMsg();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of the job file currently in use.
 *
 * @return a string corresponding to the name of the job file currently in use
 *
 * On failure, throws an exception or returns YI2cPort.CURRENTJOB_INVALID.
 */
class I2cPort_get_currentJob : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_currentJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_currentJob";
  }

  string getDescription()
  {
    return "Returns the name of the job file currently in use.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_currentJob();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Selects a job file to run immediately. If an empty string is
 * given as argument, stops running current job file.
 *
 * @param newval : a string
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class I2cPort_set_currentJob : public YapiCommand /* arguments: newval */
{
public:
  I2cPort_set_currentJob(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_currentJob";
  }

  string getDescription()
  {
    return "Selects a job file to run immediately.";
  }

  string getMoreInfo()
  {
    return "If an empty string is given as argument, stops running current job file.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_currentJob(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the job file to use when the device is powered on.
 *
 * @return a string corresponding to the job file to use when the device is powered on
 *
 * On failure, throws an exception or returns YI2cPort.STARTUPJOB_INVALID.
 */
class I2cPort_get_startupJob : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_startupJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_startupJob";
  }

  string getDescription()
  {
    return "Returns the job file to use when the device is powered on.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_startupJob();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the job to use when the device is powered on.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the job to use when the device is powered on
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class I2cPort_set_startupJob : public YapiCommand /* arguments: newval */
{
public:
  I2cPort_set_startupJob(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_startupJob";
  }

  string getDescription()
  {
    return "Changes the job to use when the device is powered on.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the job to use when the device is powered on", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_startupJob(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the maximum number of tasks in a job that the device can handle.
 *
 * @return an integer corresponding to the maximum number of tasks in a job that the device can handle
 *
 * On failure, throws an exception or returns YI2cPort.JOBMAXTASK_INVALID.
 */
class I2cPort_get_jobMaxTask : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_jobMaxTask(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_jobMaxTask";
  }

  string getDescription()
  {
    return "Returns the maximum number of tasks in a job that the device can handle.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_jobMaxTask();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns maximum size allowed for job files.
 *
 * @return an integer corresponding to maximum size allowed for job files
 *
 * On failure, throws an exception or returns YI2cPort.JOBMAXSIZE_INVALID.
 */
class I2cPort_get_jobMaxSize : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_jobMaxSize(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_jobMaxSize";
  }

  string getDescription()
  {
    return "Returns maximum size allowed for job files.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_jobMaxSize();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the type of protocol used to send I2C messages, as a string.
 * Possible values are
 * "Line" for messages separated by LF or
 * "Char" for continuous stream of codes.
 *
 * @return a string corresponding to the type of protocol used to send I2C messages, as a string
 *
 * On failure, throws an exception or returns YI2cPort.PROTOCOL_INVALID.
 */
class I2cPort_get_protocol : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_protocol(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_protocol";
  }

  string getDescription()
  {
    return "Returns the type of protocol used to send I2C messages, as a string.";
  }

  string getMoreInfo()
  {
    return "Possible values are \"Line\" for messages separated by LF or \"Char\" for continuous stream of codes.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_protocol();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the type of protocol used to send I2C messages.
 * Possible values are
 * "Line" for messages separated by LF or
 * "Char" for continuous stream of codes.
 * The suffix "/[wait]ms" can be added to reduce the transmit rate so that there
 * is always at lest the specified number of milliseconds between each message sent.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the type of protocol used to send I2C messages
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class I2cPort_set_protocol : public YapiCommand /* arguments: newval */
{
public:
  I2cPort_set_protocol(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_protocol";
  }

  string getDescription()
  {
    return "Changes the type of protocol used to send I2C messages.";
  }

  string getMoreInfo()
  {
    return "Possible values are \"Line\" for messages separated by LF or \"Char\" for continuous stream of codes. The suffix \"/[wait]ms\" can be added to reduce the transmit rate so that there is always at lest the specified number of milliseconds between each message sent. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the type of protocol used to send I2C messages", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_protocol(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the voltage level used on the I2C bus.
 *
 * @return a value among YI2cPort.I2CVOLTAGELEVEL_OFF, YI2cPort.I2CVOLTAGELEVEL_3V3 and
 * YI2cPort.I2CVOLTAGELEVEL_1V8 corresponding to the voltage level used on the I2C bus
 *
 * On failure, throws an exception or returns YI2cPort.I2CVOLTAGELEVEL_INVALID.
 */
class I2cPort_get_i2cVoltageLevel : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_i2cVoltageLevel(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_i2cVoltageLevel";
  }

  string getDescription()
  {
    return "Returns the voltage level used on the I2C bus.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_i2cVoltageLevel();
        string strval =  enumString(attrval,enumI2cVoltageLevel);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the voltage level used on the I2C bus.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a value among YI2cPort.I2CVOLTAGELEVEL_OFF, YI2cPort.I2CVOLTAGELEVEL_3V3 and
 * YI2cPort.I2CVOLTAGELEVEL_1V8 corresponding to the voltage level used on the I2C bus
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class I2cPort_set_i2cVoltageLevel : public YapiCommand /* arguments: newval */
{
public:
  I2cPort_set_i2cVoltageLevel(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_i2cVoltageLevel";
  }

  string getDescription()
  {
    return "Changes the voltage level used on the I2C bus.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "a value among OFF, 3V3 and 1V8 corresponding to the voltage level used on the I2C bus", "OFF=0,3V3=1,1V8=2", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_i2cVoltageLevel(Y_I2CVOLTAGELEVEL_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the I2C port communication parameters, as a string such as
 * "400kbps,2000ms,NoRestart". The string includes the baud rate, the
 * recovery delay after communications errors, and if needed the option
 * NoRestart to use a Stop/Start sequence instead of the
 * Restart state when performing read on the I2C bus.
 *
 * @return a string corresponding to the I2C port communication parameters, as a string such as
 *         "400kbps,2000ms,NoRestart"
 *
 * On failure, throws an exception or returns YI2cPort.I2CMODE_INVALID.
 */
class I2cPort_get_i2cMode : public YapiCommand /* arguments: */
{
public:
  I2cPort_get_i2cMode(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_i2cMode";
  }

  string getDescription()
  {
    return "Returns the I2C port communication parameters, as a string such as\"400kbps,2000ms,NoRestart\".";
  }

  string getMoreInfo()
  {
    return "The string includes the baud rate, the recovery delay after communications errors, and if needed the option NoRestart to use a Stop/Start sequence instead of the Restart state when performing read on the I2C bus.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_i2cMode();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the I2C port communication parameters, with a string such as
 * "400kbps,2000ms". The string includes the baud rate, the
 * recovery delay after communications errors, and if needed the option
 * NoRestart to use a Stop/Start sequence instead of the
 * Restart state when performing read on the I2C bus.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the I2C port communication parameters, with a string such as
 *         "400kbps,2000ms"
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class I2cPort_set_i2cMode : public YapiCommand /* arguments: newval */
{
public:
  I2cPort_set_i2cMode(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_i2cMode";
  }

  string getDescription()
  {
    return "Changes the I2C port communication parameters, with a string such as\"400kbps,2000ms\".";
  }

  string getMoreInfo()
  {
    return "The string includes the baud rate, the recovery delay after communications errors, and if needed the option NoRestart to use a Stop/Start sequence instead of the Restart state when performing read on the I2C bus. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the I2C port communication parameters, with a string such as \"400kbps,2000ms\"", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_i2cMode(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackI2cPort is YI2cPortValueCallback* (YI2cPortValueCallback)
// type of rxptr is int (int)
// type of rxbuff is bin (bin)
// type of rxbuffptr is int (int)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_I2cPort_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_I2cPort_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads a single line (or message) from the receive buffer, starting at current stream position.
 * This function is intended to be used when the serial port is configured for a message protocol,
 * such as 'Line' mode or frame protocols.
 *
 * If data at current stream position is not available anymore in the receive buffer,
 * the function returns the oldest available line and moves the stream position just after.
 * If no new full line is received, the function returns an empty line.
 *
 * @return a string with a single line of text
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_readLine : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_readLine(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readLine";
  }

  string getDescription()
  {
    return "Reads a single line (or message) from the receive buffer, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "This function is intended to be used when the serial port is configured for a message protocol, such as 'Line' mode or frame protocols.\nIf data at current stream position is not available anymore in the receive buffer, the function returns the oldest available line and moves the stream position just after. If no new full line is received, the function returns an empty line.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->readLine();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Searches for incoming messages in the serial port receive buffer matching a given pattern,
 * starting at current position. This function will only compare and return printable characters
 * in the message strings. Binary protocols are handled as hexadecimal strings.
 *
 * The search returns all messages matching the expression provided as argument in the buffer.
 * If no matching message is found, the search waits for one up to the specified maximum timeout
 * (in milliseconds).
 *
 * @param pattern : a limited regular expression describing the expected message format,
 *         or an empty string if all messages should be returned (no filtering).
 *         When using binary protocols, the format applies to the hexadecimal
 *         representation of the message.
 * @param maxWait : the maximum number of milliseconds to wait for a message if none is found
 *         in the receive buffer.
 *
 * @return an array of strings containing the messages found, if any.
 *         Binary messages are converted to hexadecimal representation.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_I2cPort_readMessages : public YapiCommand /* arguments: pattern maxWait */
{
public:
  apifun_I2cPort_readMessages(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readMessages";
  }

  string getDescription()
  {
    return "Searches for incoming messages in the serial port receive buffer matching a given pattern,starting at current position.";
  }

  string getMoreInfo()
  {
    return "This function will only compare and return printable characters in the message strings. Binary protocols are handled as hexadecimal strings.\nThe search returns all messages matching the expression provided as argument in the buffer. If no matching message is found, the search waits for one up to the specified maximum timeout (in milliseconds).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "pattern", "a limited regular expression describing the expected message format, or an empty string if all messages should be returned (no filtering). When using binary protocols, the format applies to the hexadecimal representation of the message.", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a message if none is found in the receive buffer.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<string> value = (*list)[i]->readMessages((*args)[0]->get_stringValue(),
                                                        (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Changes the current internal stream position to the specified value. This function
 * does not affect the device, it only changes the value stored in the API object
 * for the next read operations.
 *
 * @param absPos : the absolute position index for next read operations.
 *
 * @return nothing.
 */
class apifun_I2cPort_read_seek : public YapiCommand /* arguments: absPos */
{
public:
  apifun_I2cPort_read_seek(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "read_seek";
  }

  string getDescription()
  {
    return "Changes the current internal stream position to the specified value.";
  }

  string getMoreInfo()
  {
    return "This function does not affect the device, it only changes the value stored in the API object for the next read operations.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "absPos", "the absolute position index for next read operations.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->read_seek((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the current absolute stream position pointer of the API object.
 *
 * @return the absolute position index for next read operations.
 */
class apifun_I2cPort_read_tell : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_read_tell(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "read_tell";
  }

  string getDescription()
  {
    return "Returns the current absolute stream position pointer of the API object.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->read_tell();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the number of bytes available to read in the input buffer starting from the
 * current absolute stream position pointer of the API object.
 *
 * @return the number of bytes available to read
 */
class apifun_I2cPort_read_avail : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_read_avail(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "read_avail";
  }

  string getDescription()
  {
    return "Returns the number of bytes available to read in the input buffer starting from thecurrent absolute stream position pointer of the API object.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->read_avail();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a text line query to the serial port, and reads the reply, if any.
 * This function is intended to be used when the serial port is configured for 'Line' protocol.
 *
 * @param query : the line query to send (without CR/LF)
 * @param maxWait : the maximum number of milliseconds to wait for a reply.
 *
 * @return the next text line received after sending the text query, as a string.
 *         Additional lines can be obtained by calling readLine or readMessages.
 *
 * On failure, throws an exception or returns an empty string.
 */
class apifun_I2cPort_queryLine : public YapiCommand /* arguments: query maxWait */
{
public:
  apifun_I2cPort_queryLine(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "queryLine";
  }

  string getDescription()
  {
    return "Sends a text line query to the serial port, and reads the reply, if any.";
  }

  string getMoreInfo()
  {
    return "This function is intended to be used when the serial port is configured for 'Line' protocol.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "query", "the line query to send (without CR/LF)", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a reply.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->queryLine((*args)[0]->get_stringValue(),
                                             (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a binary message to the serial port, and reads the reply, if any.
 * This function is intended to be used when the serial port is configured for
 * Frame-based protocol.
 *
 * @param hexString : the message to send, coded in hexadecimal
 * @param maxWait : the maximum number of milliseconds to wait for a reply.
 *
 * @return the next frame received after sending the message, as a hex string.
 *         Additional frames can be obtained by calling readHex or readMessages.
 *
 * On failure, throws an exception or returns an empty string.
 */
class apifun_I2cPort_queryHex : public YapiCommand /* arguments: hexString maxWait */
{
public:
  apifun_I2cPort_queryHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "queryHex";
  }

  string getDescription()
  {
    return "Sends a binary message to the serial port, and reads the reply, if any.";
  }

  string getMoreInfo()
  {
    return "This function is intended to be used when the serial port is configured for Frame-based protocol.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "hexString", "the message to send, coded in hexadecimal", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a reply.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->queryHex((*args)[0]->get_stringValue(),
                                            (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Saves the job definition string (JSON data) into a job file.
 * The job file can be later enabled using selectJob().
 *
 * @param jobfile : name of the job file to save on the device filesystem
 * @param jsonDef : a string containing a JSON definition of the job
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_uploadJob : public YapiCommand /* arguments: jobfile jsonDef */
{
public:
  apifun_I2cPort_uploadJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "uploadJob";
  }

  string getDescription()
  {
    return "Saves the job definition string (JSON data) into a job file.";
  }

  string getMoreInfo()
  {
    return "The job file can be later enabled using selectJob().";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "jobfile", "name of the job file to save on the device filesystem", "_STRING", false));
    res->push_back(new ArgumentDesc(STRING_ARG, "jsonDef", "a string containing a JSON definition of the job", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->uploadJob((*args)[0]->get_stringValue(),
                              (*args)[1]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Load and start processing the specified job file. The file must have
 * been previously created using the user interface or uploaded on the
 * device filesystem using the uploadJob() function.
 *
 * @param jobfile : name of the job file (on the device filesystem)
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_selectJob : public YapiCommand /* arguments: jobfile */
{
public:
  apifun_I2cPort_selectJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "selectJob";
  }

  string getDescription()
  {
    return "Load and start processing the specified job file.";
  }

  string getMoreInfo()
  {
    return "The file must have been previously created using the user interface or uploaded on the device filesystem using the uploadJob() function.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "jobfile", "name of the job file (on the device filesystem)", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->selectJob((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Clears the serial port buffer and resets counters to zero.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_reset : public YapiCommand /* arguments: */
{
public:
  apifun_I2cPort_reset(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "reset";
  }

  string getDescription()
  {
    return "Clears the serial port buffer and resets counters to zero.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->reset();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a one-way message (provided as a a binary buffer) to a device on the I2C bus.
 * This function checks and reports communication errors on the I2C bus.
 *
 * @param slaveAddr : the 7-bit address of the slave device (without the direction bit)
 * @param buff : the binary buffer to be sent
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_i2cSendBin : public YapiCommand /* arguments: slaveAddr buff */
{
public:
  apifun_I2cPort_i2cSendBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "i2cSendBin";
  }

  string getDescription()
  {
    return "Sends a one-way message (provided as a a binary buffer) to a device on the I2C bus.";
  }

  string getMoreInfo()
  {
    return "This function checks and reports communication errors on the I2C bus.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveAddr", "the 7-bit address of the slave device (without the direction bit)", "_INT", false));
    res->push_back(new ArgumentDesc(BIN_ARG, "buff", "the binary buffer to be sent", "_BIN", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->i2cSendBin((int)(*args)[0]->get_intValue(),
                               (*args)[1]->get_binValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a one-way message (provided as a list of integer) to a device on the I2C bus.
 * This function checks and reports communication errors on the I2C bus.
 *
 * @param slaveAddr : the 7-bit address of the slave device (without the direction bit)
 * @param values : a list of data bytes to be sent
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_i2cSendArray : public YapiCommand /* arguments: slaveAddr values */
{
public:
  apifun_I2cPort_i2cSendArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "i2cSendArray";
  }

  string getDescription()
  {
    return "Sends a one-way message (provided as a list of integer) to a device on the I2C bus.";
  }

  string getMoreInfo()
  {
    return "This function checks and reports communication errors on the I2C bus.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveAddr", "the 7-bit address of the slave device (without the direction bit)", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "values", "a list of data bytes to be sent", "_INTARR", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->i2cSendArray((int)(*args)[0]->get_intValue(),
                                 (*args)[1]->get_intArrValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a one-way message (provided as a a binary buffer) to a device on the I2C bus,
 * then read back the specified number of bytes from device.
 * This function checks and reports communication errors on the I2C bus.
 *
 * @param slaveAddr : the 7-bit address of the slave device (without the direction bit)
 * @param buff : the binary buffer to be sent
 * @param rcvCount : the number of bytes to receive once the data bytes are sent
 *
 * @return a list of bytes with the data received from slave device.
 *
 * On failure, throws an exception or returns an empty binary buffer.
 */
class apifun_I2cPort_i2cSendAndReceiveBin : public YapiCommand /* arguments: slaveAddr buff rcvCount */
{
public:
  apifun_I2cPort_i2cSendAndReceiveBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "i2cSendAndReceiveBin";
  }

  string getDescription()
  {
    return "Sends a one-way message (provided as a a binary buffer) to a device on the I2C bus,then read back the specified number of bytes from device.";
  }

  string getMoreInfo()
  {
    return "This function checks and reports communication errors on the I2C bus.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveAddr", "the 7-bit address of the slave device (without the direction bit)", "_INT", false));
    res->push_back(new ArgumentDesc(BIN_ARG, "buff", "the binary buffer to be sent", "_BIN", false));
    res->push_back(new ArgumentDesc(INT_ARG, "rcvCount", "the number of bytes to receive once the data bytes are sent", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->i2cSendAndReceiveBin((int)(*args)[0]->get_intValue(),
                                                        (*args)[1]->get_binValue(),
                                                        (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a one-way message (provided as a list of integer) to a device on the I2C bus,
 * then read back the specified number of bytes from device.
 * This function checks and reports communication errors on the I2C bus.
 *
 * @param slaveAddr : the 7-bit address of the slave device (without the direction bit)
 * @param values : a list of data bytes to be sent
 * @param rcvCount : the number of bytes to receive once the data bytes are sent
 *
 * @return a list of bytes with the data received from slave device.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_I2cPort_i2cSendAndReceiveArray : public YapiCommand /* arguments: slaveAddr values rcvCount */
{
public:
  apifun_I2cPort_i2cSendAndReceiveArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "i2cSendAndReceiveArray";
  }

  string getDescription()
  {
    return "Sends a one-way message (provided as a list of integer) to a device on the I2C bus,then read back the specified number of bytes from device.";
  }

  string getMoreInfo()
  {
    return "This function checks and reports communication errors on the I2C bus.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveAddr", "the 7-bit address of the slave device (without the direction bit)", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "values", "a list of data bytes to be sent", "_INTARR", false));
    res->push_back(new ArgumentDesc(INT_ARG, "rcvCount", "the number of bytes to receive once the data bytes are sent", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->i2cSendAndReceiveArray((int)(*args)[0]->get_intValue(),
                                                               (*args)[1]->get_intArrValue(),
                                                               (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a text-encoded I2C code stream to the I2C bus, as is.
 * An I2C code stream is a string made of hexadecimal data bytes,
 * but that may also include the I2C state transitions code:
 * "{S}" to emit a start condition,
 * "{R}" for a repeated start condition,
 * "{P}" for a stop condition,
 * "xx" for receiving a data byte,
 * "{A}" to ack a data byte received and
 * "{N}" to nack a data byte received.
 * If a newline ("\n") is included in the stream, the message
 * will be terminated and a newline will also be added to the
 * receive stream.
 *
 * @param codes : the code stream to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_writeStr : public YapiCommand /* arguments: codes */
{
public:
  apifun_I2cPort_writeStr(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeStr";
  }

  string getDescription()
  {
    return "Sends a text-encoded I2C code stream to the I2C bus, as is.";
  }

  string getMoreInfo()
  {
    return "An I2C code stream is a string made of hexadecimal data bytes, but that may also include the I2C state transitions code: \"{S}\" to emit a start condition, \"{R}\" for a repeated start condition, \"{P}\" for a stop condition, \"xx\" for receiving a data byte, \"{A}\" to ack a data byte received and \"{N}\" to nack a data byte received. If a newline (\"\\n\") is included in the stream, the message will be terminated and a newline will also be added to the receive stream.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "codes", "the code stream to send", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeStr((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a text-encoded I2C code stream to the I2C bus, and terminate
 * the message en relâchant le bus.
 * An I2C code stream is a string made of hexadecimal data bytes,
 * but that may also include the I2C state transitions code:
 * "{S}" to emit a start condition,
 * "{R}" for a repeated start condition,
 * "{P}" for a stop condition,
 * "xx" for receiving a data byte,
 * "{A}" to ack a data byte received and
 * "{N}" to nack a data byte received.
 * At the end of the stream, a stop condition is added if missing
 * and a newline is added to the receive buffer as well.
 *
 * @param codes : the code stream to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_writeLine : public YapiCommand /* arguments: codes */
{
public:
  apifun_I2cPort_writeLine(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeLine";
  }

  string getDescription()
  {
    return "Sends a text-encoded I2C code stream to the I2C bus, and terminatethe message en rel\303\242chant le bus.";
  }

  string getMoreInfo()
  {
    return "An I2C code stream is a string made of hexadecimal data bytes, but that may also include the I2C state transitions code: \"{S}\" to emit a start condition, \"{R}\" for a repeated start condition, \"{P}\" for a stop condition, \"xx\" for receiving a data byte, \"{A}\" to ack a data byte received and \"{N}\" to nack a data byte received. At the end of the stream, a stop condition is added if missing and a newline is added to the receive buffer as well.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "codes", "the code stream to send", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeLine((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a single byte to the I2C bus. Depending on the I2C bus state, the byte
 * will be interpreted as an address byte or a data byte.
 *
 * @param code : the byte to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_writeByte : public YapiCommand /* arguments: code */
{
public:
  apifun_I2cPort_writeByte(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeByte";
  }

  string getDescription()
  {
    return "Sends a single byte to the I2C bus.";
  }

  string getMoreInfo()
  {
    return "Depending on the I2C bus state, the byte will be interpreted as an address byte or a data byte.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "code", "the byte to send", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeByte((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a byte sequence (provided as a hexadecimal string) to the I2C bus.
 * Depending on the I2C bus state, the first byte will be interpreted as an
 * address byte or a data byte.
 *
 * @param hexString : a string of hexadecimal byte codes
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_writeHex : public YapiCommand /* arguments: hexString */
{
public:
  apifun_I2cPort_writeHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeHex";
  }

  string getDescription()
  {
    return "Sends a byte sequence (provided as a hexadecimal string) to the I2C bus.";
  }

  string getMoreInfo()
  {
    return "Depending on the I2C bus state, the first byte will be interpreted as an address byte or a data byte.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "hexString", "a string of hexadecimal byte codes", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeHex((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a binary buffer to the I2C bus, as is.
 * Depending on the I2C bus state, the first byte will be interpreted
 * as an address byte or a data byte.
 *
 * @param buff : the binary buffer to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_writeBin : public YapiCommand /* arguments: buff */
{
public:
  apifun_I2cPort_writeBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeBin";
  }

  string getDescription()
  {
    return "Sends a binary buffer to the I2C bus, as is.";
  }

  string getMoreInfo()
  {
    return "Depending on the I2C bus state, the first byte will be interpreted as an address byte or a data byte.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(BIN_ARG, "buff", "the binary buffer to send", "_BIN", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeBin((*args)[0]->get_binValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a byte sequence (provided as a list of bytes) to the I2C bus.
 * Depending on the I2C bus state, the first byte will be interpreted as an
 * address byte or a data byte.
 *
 * @param byteList : a list of byte codes
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_I2cPort_writeArray : public YapiCommand /* arguments: byteList */
{
public:
  apifun_I2cPort_writeArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeArray";
  }

  string getDescription()
  {
    return "Sends a byte sequence (provided as a list of bytes) to the I2C bus.";
  }

  string getMoreInfo()
  {
    return "Depending on the I2C bus state, the first byte will be interpreted as an address byte or a data byte.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INTARR_ARG, "byteList", "a list of byte codes", "_INTARR", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeArray((*args)[0]->get_intArrValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Retrieves messages (both direction) in the I2C port buffer, starting at current position.
 *
 * If no message is found, the search waits for one up to the specified maximum timeout
 * (in milliseconds).
 *
 * @param maxWait : the maximum number of milliseconds to wait for a message if none is found
 *         in the receive buffer.
 *
 * @return an array of YI2cSnoopingRecord objects containing the messages found, if any.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_I2cPort_snoopMessages : public YapiCommand /* arguments: maxWait */
{
public:
  apifun_I2cPort_snoopMessages(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "snoopMessages";
  }

  string getDescription()
  {
    return "Retrieves messages (both direction) in the I2C port buffer, starting at current position.";
  }

  string getMoreInfo()
  {
    return "If no message is found, the search waits for one up to the specified maximum timeout (in milliseconds).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a message if none is found in the receive buffer.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YI2cPort*>* list = enumerateTargets<YI2cPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<YI2cSnoopingRecord> value = (*list)[i]->snoopMessages((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of generated code: YI2cPort implementation)

//--- (generated code: YI2cPort functions)
void YI2cPortCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YI2cPort>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new I2cPort_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_rxCount(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_txCount(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_errCount(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_rxMsgCount(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_txMsgCount(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_lastMsg(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_currentJob(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_set_currentJob(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_startupJob(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_set_startupJob(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_jobMaxTask(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_jobMaxSize(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_protocol(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_set_protocol(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_i2cVoltageLevel(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_set_i2cVoltageLevel(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_get_i2cMode(this)));
    cmdList->push_back((YapiCommand*) (new I2cPort_set_i2cMode(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_readLine(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_readMessages(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_read_seek(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_read_tell(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_read_avail(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_queryLine(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_queryHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_uploadJob(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_selectJob(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_reset(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_i2cSendBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_i2cSendArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_i2cSendAndReceiveBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_i2cSendAndReceiveArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_writeStr(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_writeLine(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_writeByte(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_writeHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_writeBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_writeArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_I2cPort_snoopMessages(this)));
  }

//--- (end of generated code: YI2cPort functions)
