/*********************************************************************
 *
 *  $Id: svn_id $
 *
 *  Implements commands to handle SpectralSensor functions
 *
 *  - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YSpectralSensor.h"
#include "yocto_api.h"
#include "yocto_spectralsensor.h"

using namespace std;

//--- (YSpectralSensor definitions)

static const char *enumEstimationModel[] = {
    "REFLECTION",
    "EMISSION",
    NULL
};

//--- (end of YSpectralSensor definitions)


//--- (YSpectralSensor implementation)
bool   YSpectralSensorCmdLine::isModule(void)      { return false; }
string YSpectralSensorCmdLine::YFunctionName(void) { return "YSpectralSensor"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of ledCurrent is int (Int)
// type of resolution is double (MeasureVal)
// type of integrationTime is int (Int)
// type of gain is int (Int)
// type of estimationModel is int (EstimationModel)
// type of saturation is int (SaturationBits)
// type of estimatedRGB is int (U24Color)
// type of estimatedHSL is int (U24Color)
// type of estimatedXYZ is string (ColorCoord)
// type of estimatedOkLab is string (ColorCoord)
// type of nearRAL1 is string (Text)
// type of nearRAL2 is string (Text)
// type of nearRAL3 is string (Text)
// type of nearHTMLColor is string (Text)
// type of nearSimpleColor is string (Text)
// type of ledCurrentAtPowerOn is int (Int)
// type of integrationTimeAtPowerOn is int (Int)
// type of gainAtPowerOn is int (Int)
/**
 * Returns the logical name of the spectral sensor.
 *
 * @return a string corresponding to the logical name of the spectral sensor
 *
 * On failure, throws an exception or returns YSpectralSensor.LOGICALNAME_INVALID.
 */
class SpectralSensor_get_logicalName : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the spectral sensor.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the spectral sensor. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the spectral sensor
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the spectral sensor.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the spectral sensor", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the spectral sensor.
 *
 * @return a string corresponding to a short string representing the current state of the spectral sensor
 *
 * On failure, throws an exception or returns YSpectralSensor.ADVERTISEDVALUE_INVALID.
 */
class SpectralSensor_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the spectral sensor.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the current value of the LED.
 * This method retrieves the current flowing through the LED
 * and returns it as an integer or an object.
 *
 * @return an integer corresponding to the current value of the LED
 *
 * On failure, throws an exception or returns YSpectralSensor.LEDCURRENT_INVALID.
 */
class SpectralSensor_get_ledCurrent : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_ledCurrent(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_ledCurrent";
  }

  string getDescription()
  {
    return "Returns the current value of the LED.";
  }

  string getMoreInfo()
  {
    return "This method retrieves the current flowing through the LED and returns it as an integer or an object.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_ledCurrent();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the luminosity of the module leds. The parameter is a
 * value between 0 and 254.
 *
 * @param newval : an integer corresponding to the luminosity of the module leds
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_ledCurrent : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_ledCurrent(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_ledCurrent";
  }

  string getDescription()
  {
    return "Changes the luminosity of the module leds.";
  }

  string getMoreInfo()
  {
    return "The parameter is a value between 0 and 254.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer corresponding to the luminosity of the module leds", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_ledCurrent(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Changes the resolution of the measured physical values. The resolution corresponds to the numerical precision
 * when displaying value. It does not change the precision of the measure itself.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : a floating point number corresponding to the resolution of the measured physical values
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_resolution : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_resolution(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_resolution";
  }

  string getDescription()
  {
    return "Changes the resolution of the measured physical values.";
  }

  string getMoreInfo()
  {
    return "The resolution corresponds to the numerical precision when displaying value. It does not change the precision of the measure itself. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(DOUBLE_ARG, "newval", "a floating point number corresponding to the resolution of the measured physical values", "_DOUBLE", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_resolution(double((*args)[0]->get_doubleValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the resolution of the measured values. The resolution corresponds to the numerical precision
 * of the measures, which is not always the same as the actual precision of the sensor.
 *
 * @return a floating point number corresponding to the resolution of the measured values
 *
 * On failure, throws an exception or returns YSpectralSensor.RESOLUTION_INVALID.
 */
class SpectralSensor_get_resolution : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_resolution(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_resolution";
  }

  string getDescription()
  {
    return "Returns the resolution of the measured values.";
  }

  string getMoreInfo()
  {
    return "The resolution corresponds to the numerical precision of the measures, which is not always the same as the actual precision of the sensor.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        double attrval = (*list)[i]->get_resolution();
        char buf[32]; SAFE_SPRINTF(buf,32,"%.3f",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the current integration time.
 * This method retrieves the integration time value
 * used for data processing and returns it as an integer or an object.
 *
 * @return an integer corresponding to the current integration time
 *
 * On failure, throws an exception or returns YSpectralSensor.INTEGRATIONTIME_INVALID.
 */
class SpectralSensor_get_integrationTime : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_integrationTime(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_integrationTime";
  }

  string getDescription()
  {
    return "Returns the current integration time.";
  }

  string getMoreInfo()
  {
    return "This method retrieves the integration time value used for data processing and returns it as an integer or an object.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_integrationTime();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Sets the integration time for data processing.
 * This method takes a parameter `val` and assigns it
 * as the new integration time. This affects the duration
 * for which data is integrated before being processed.
 *
 * @param newval : an integer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_integrationTime : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_integrationTime(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_integrationTime";
  }

  string getDescription()
  {
    return "Sets the integration time for data processing.";
  }

  string getMoreInfo()
  {
    return "This method takes a parameter `val` and assigns it as the new integration time. This affects the duration for which data is integrated before being processed.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_integrationTime(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Retrieves the current gain.
 * This method updates the gain value.
 *
 * @return an integer
 *
 * On failure, throws an exception or returns YSpectralSensor.GAIN_INVALID.
 */
class SpectralSensor_get_gain : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_gain(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_gain";
  }

  string getDescription()
  {
    return "Retrieves the current gain.";
  }

  string getMoreInfo()
  {
    return "This method updates the gain value.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_gain();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Sets the gain for signal processing.
 * This method takes a parameter `val` and assigns it
 * as the new gain. This affects the sensitivity and
 * intensity of the processed signal.
 *
 * @param newval : an integer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_gain : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_gain(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_gain";
  }

  string getDescription()
  {
    return "Sets the gain for signal processing.";
  }

  string getMoreInfo()
  {
    return "This method takes a parameter `val` and assigns it as the new gain. This affects the sensitivity and intensity of the processed signal.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_gain(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the model for color estimation.
 *
 * @return either YSpectralSensor.ESTIMATIONMODEL_REFLECTION or
 * YSpectralSensor.ESTIMATIONMODEL_EMISSION, according to the model for color estimation
 *
 * On failure, throws an exception or returns YSpectralSensor.ESTIMATIONMODEL_INVALID.
 */
class SpectralSensor_get_estimationModel : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_estimationModel(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimationModel";
  }

  string getDescription()
  {
    return "Returns the model for color estimation.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_estimationModel();
        string strval =  enumString(attrval,enumEstimationModel);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the model for color estimation.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : either YSpectralSensor.ESTIMATIONMODEL_REFLECTION or
 * YSpectralSensor.ESTIMATIONMODEL_EMISSION, according to the model for color estimation
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_estimationModel : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_estimationModel(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_estimationModel";
  }

  string getDescription()
  {
    return "Changes the model for color estimation.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "either REFLECTION or EMISSION, according to the model for color estimation", "REFLECTION=0,EMISSION=1", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_estimationModel(Y_ESTIMATIONMODEL_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the current saturation of the sensor.
 * This function updates the sensor's saturation value.
 *
 * @return an integer corresponding to the current saturation of the sensor
 *
 * On failure, throws an exception or returns YSpectralSensor.SATURATION_INVALID.
 */
class SpectralSensor_get_saturation : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_saturation(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_saturation";
  }

  string getDescription()
  {
    return "Returns the current saturation of the sensor.";
  }

  string getMoreInfo()
  {
    return "This function updates the sensor's saturation value.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_saturation();
        string strval =  bitString("AD", attrval);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color in RGB format (0xRRGGBB).
 * This method retrieves the estimated color values
 * and returns them as an RGB object or structure.
 *
 * @return an integer corresponding to the estimated color in RGB format (0xRRGGBB)
 *
 * On failure, throws an exception or returns YSpectralSensor.ESTIMATEDRGB_INVALID.
 */
class SpectralSensor_get_estimatedRGB : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_estimatedRGB(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedRGB";
  }

  string getDescription()
  {
    return "Returns the estimated color in RGB format (0xRRGGBB).";
  }

  string getMoreInfo()
  {
    return "This method retrieves the estimated color values and returns them as an RGB object or structure.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_estimatedRGB();
        char buf[32]; SAFE_SPRINTF(buf,32,"#%06x",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color in HSL (Hue, Saturation, Lightness) format.
 * This method retrieves the estimated color values
 * and returns them as an HSL object or structure.
 *
 * @return an integer corresponding to the estimated color in HSL (Hue, Saturation, Lightness) format
 *
 * On failure, throws an exception or returns YSpectralSensor.ESTIMATEDHSL_INVALID.
 */
class SpectralSensor_get_estimatedHSL : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_estimatedHSL(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedHSL";
  }

  string getDescription()
  {
    return "Returns the estimated color in HSL (Hue, Saturation, Lightness) format.";
  }

  string getMoreInfo()
  {
    return "This method retrieves the estimated color values and returns them as an HSL object or structure.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_estimatedHSL();
        char buf[32]; SAFE_SPRINTF(buf,32,"#%06x",attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color in XYZ format.
 * This method retrieves the estimated color values
 * and returns them as an XYZ object or structure.
 *
 * @return a string corresponding to the estimated color in XYZ format
 *
 * On failure, throws an exception or returns YSpectralSensor.ESTIMATEDXYZ_INVALID.
 */
class SpectralSensor_get_estimatedXYZ : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_estimatedXYZ(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedXYZ";
  }

  string getDescription()
  {
    return "Returns the estimated color in XYZ format.";
  }

  string getMoreInfo()
  {
    return "This method retrieves the estimated color values and returns them as an XYZ object or structure.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_estimatedXYZ();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color in OkLab format.
 * This method retrieves the estimated color values
 * and returns them as an OkLab object or structure.
 *
 * @return a string corresponding to the estimated color in OkLab format
 *
 * On failure, throws an exception or returns YSpectralSensor.ESTIMATEDOKLAB_INVALID.
 */
class SpectralSensor_get_estimatedOkLab : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_estimatedOkLab(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_estimatedOkLab";
  }

  string getDescription()
  {
    return "Returns the estimated color in OkLab format.";
  }

  string getMoreInfo()
  {
    return "This method retrieves the estimated color values and returns them as an OkLab object or structure.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_estimatedOkLab();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the estimated color.
 * This method retrieves the estimated color values
 * and returns them as the color name.
 *
 * @return a string corresponding to the estimated color
 *
 * On failure, throws an exception or returns YSpectralSensor.NEARSIMPLECOLOR_INVALID.
 */
class SpectralSensor_get_nearSimpleColor : public YapiCommand /* arguments: */
{
public:
  SpectralSensor_get_nearSimpleColor(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_nearSimpleColor";
  }

  string getDescription()
  {
    return "Returns the estimated color.";
  }

  string getMoreInfo()
  {
    return "This method retrieves the estimated color values and returns them as the color name.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_nearSimpleColor();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Sets the LED current at power-on.
 * This method takes a parameter `val` and assigns it to startupLumin, representing the LED current defined
 * at startup.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : an integer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_ledCurrentAtPowerOn : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_ledCurrentAtPowerOn(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_ledCurrentAtPowerOn";
  }

  string getDescription()
  {
    return "Sets the LED current at power-on.";
  }

  string getMoreInfo()
  {
    return "This method takes a parameter `val` and assigns it to startupLumin, representing the LED current defined at startup. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_ledCurrentAtPowerOn(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Sets the integration time at power-on.
 * This method takes a parameter `val` and assigns it to startupIntegTime, representing the integration time
 * defined at startup.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : an integer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_integrationTimeAtPowerOn : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_integrationTimeAtPowerOn(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_integrationTimeAtPowerOn";
  }

  string getDescription()
  {
    return "Sets the integration time at power-on.";
  }

  string getMoreInfo()
  {
    return "This method takes a parameter `val` and assigns it to startupIntegTime, representing the integration time defined at startup. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_integrationTimeAtPowerOn(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Sets the gain at power-on.
 * This method takes a parameter `val` and assigns it to startupGain, representing the gain defined at startup.
 * Remember to call the saveToFlash() method of the module if the modification must be kept.
 *
 * @param newval : an integer
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SpectralSensor_set_gainAtPowerOn : public YapiCommand /* arguments: newval */
{
public:
  SpectralSensor_set_gainAtPowerOn(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_gainAtPowerOn";
  }

  string getDescription()
  {
    return "Sets the gain at power-on.";
  }

  string getMoreInfo()
  {
    return "This method takes a parameter `val` and assigns it to startupGain, representing the gain defined at startup. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "an integer", "_INT", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_gainAtPowerOn(int((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackSpectralSensor is YSpectralSensorValueCallback* (YSpectralSensorValueCallback)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SpectralSensor_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_SpectralSensor_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SpectralSensor_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_SpectralSensor_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_SpectralSensor_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_SpectralSensor_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_SpectralSensor_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_SpectralSensor_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSpectralSensor*>* list = enumerateTargets<YSpectralSensor>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of YSpectralSensor implementation)

//--- (YSpectralSensor functions)
void YSpectralSensorCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YSpectralSensor>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_ledCurrent(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_ledCurrent(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_resolution(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_resolution(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_integrationTime(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_integrationTime(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_gain(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_gain(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_estimationModel(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_estimationModel(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_saturation(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_estimatedRGB(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_estimatedHSL(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_estimatedXYZ(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_estimatedOkLab(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_get_nearSimpleColor(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_ledCurrentAtPowerOn(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_integrationTimeAtPowerOn(this)));
    cmdList->push_back((YapiCommand*) (new SpectralSensor_set_gainAtPowerOn(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SpectralSensor_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SpectralSensor_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SpectralSensor_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SpectralSensor_get_serialNumber(this)));
  }

//--- (end of YSpectralSensor functions)
