/*********************************************************************
 *
 * $Id: YSerialPort.cpp 53689 2023-03-22 11:17:15Z mvuilleu $
 *
 * Implements commands to handle SerialPort functions
 *
 * - - - - - - - - - License information: - - - - - - - - -
 *
 *  Copyright (C) 2011 and beyond by Yoctopuce Sarl, Switzerland.
 *
 *  Yoctopuce Sarl (hereafter Licensor) grants to you a perpetual
 *  non-exclusive license to use, modify, copy and integrate this
 *  file into your software for the sole purpose of interfacing
 *  with Yoctopuce products.
 *
 *  You may reproduce and distribute copies of this file in
 *  source or object form, as long as the sole purpose of this
 *  code is to interface with Yoctopuce products. You must retain
 *  this notice in the distributed source file.
 *
 *  You should refer to Yoctopuce General Terms and Conditions
 *  for additional information regarding your rights and
 *  obligations.
 *
 *  THE SOFTWARE AND DOCUMENTATION ARE PROVIDED 'AS IS' WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS
 *  FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO
 *  EVENT SHALL LICENSOR BE LIABLE FOR ANY INCIDENTAL, SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA,
 *  COST OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR
 *  SERVICES, ANY CLAIMS BY THIRD PARTIES (INCLUDING BUT NOT
 *  LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS FOR INDEMNITY OR
 *  CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON THE
 *  BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF
 *  WARRANTY, OR OTHERWISE.
 *
 *********************************************************************/


#define _CRT_SECURE_NO_DEPRECATE
#include <stdio.h>
#include "YSerialPort.h"
#include "yocto_api.h"
#include "yocto_serialport.h"

using namespace std;

//--- (generated code: YSerialPort definitions)

static const char *enumSerialVoltageLevel[] = {
    "OFF",
    "TTL3V",
    "TTL3VR",
    "TTL5V",
    "TTL5VR",
    "RS232",
    "RS485",
    "TTL1V8",
    "SDI12",
    NULL
};

//--- (end of generated code: YSerialPort definitions)


//--- (generated code: YSerialPort implementation)
bool   YSerialPortCmdLine::isModule(void)      { return false; }
string YSerialPortCmdLine::YFunctionName(void) { return "YSerialPort"; }

// type of logicalName is string (Text)
// type of advertisedValue is string (PubText)
// type of rxCount is int (UInt31)
// type of txCount is int (UInt31)
// type of errCount is int (UInt31)
// type of rxMsgCount is int (UInt31)
// type of txMsgCount is int (UInt31)
// type of lastMsg is string (Text)
// type of currentJob is string (Text)
// type of startupJob is string (Text)
// type of jobMaxTask is int (UInt31)
// type of jobMaxSize is int (UInt31)
// type of command is string (Text)
// type of protocol is string (Protocol)
// type of voltageLevel is int (SerialVoltageLevel)
// type of serialMode is string (SerialMode)
/**
 * Returns the logical name of the serial port.
 *
 * @return a string corresponding to the logical name of the serial port
 *
 * On failure, throws an exception or returns YSerialPort.LOGICALNAME_INVALID.
 */
class SerialPort_get_logicalName : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_logicalName(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_logicalName";
  }

  string getDescription()
  {
    return "Returns the logical name of the serial port.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_logicalName();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the logical name of the serial port. You can use yCheckLogicalName()
 * prior to this call to make sure that your parameter is valid.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the logical name of the serial port
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SerialPort_set_logicalName : public YapiCommand /* arguments: newval */
{
public:
  SerialPort_set_logicalName(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_logicalName";
  }

  string getDescription()
  {
    return "Changes the logical name of the serial port.";
  }

  string getMoreInfo()
  {
    return "You can use yCheckLogicalName() prior to this call to make sure that your parameter is valid. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the logical name of the serial port", "_LOGICAL", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    if (!YAPI::CheckLogicalName((*args)[0]->get_stringValue())) {
        throw std::string("Invalid name :" + (*args)[0]->get_stringValue());
    }
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_logicalName(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns a short string representing the current state of the serial port.
 *
 * @return a string corresponding to a short string representing the current state of the serial port
 *
 * On failure, throws an exception or returns YSerialPort.ADVERTISEDVALUE_INVALID.
 */
class SerialPort_get_advertisedValue : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_advertisedValue(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_advertisedValue";
  }

  string getDescription()
  {
    return "Returns a short string representing the current state of the serial port.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_advertisedValue();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of bytes received since last reset.
 *
 * @return an integer corresponding to the total number of bytes received since last reset
 *
 * On failure, throws an exception or returns YSerialPort.RXCOUNT_INVALID.
 */
class SerialPort_get_rxCount : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_rxCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_rxCount";
  }

  string getDescription()
  {
    return "Returns the total number of bytes received since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_rxCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of bytes transmitted since last reset.
 *
 * @return an integer corresponding to the total number of bytes transmitted since last reset
 *
 * On failure, throws an exception or returns YSerialPort.TXCOUNT_INVALID.
 */
class SerialPort_get_txCount : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_txCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_txCount";
  }

  string getDescription()
  {
    return "Returns the total number of bytes transmitted since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_txCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of communication errors detected since last reset.
 *
 * @return an integer corresponding to the total number of communication errors detected since last reset
 *
 * On failure, throws an exception or returns YSerialPort.ERRCOUNT_INVALID.
 */
class SerialPort_get_errCount : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_errCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_errCount";
  }

  string getDescription()
  {
    return "Returns the total number of communication errors detected since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_errCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of messages received since last reset.
 *
 * @return an integer corresponding to the total number of messages received since last reset
 *
 * On failure, throws an exception or returns YSerialPort.RXMSGCOUNT_INVALID.
 */
class SerialPort_get_rxMsgCount : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_rxMsgCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_rxMsgCount";
  }

  string getDescription()
  {
    return "Returns the total number of messages received since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_rxMsgCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the total number of messages send since last reset.
 *
 * @return an integer corresponding to the total number of messages send since last reset
 *
 * On failure, throws an exception or returns YSerialPort.TXMSGCOUNT_INVALID.
 */
class SerialPort_get_txMsgCount : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_txMsgCount(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_txMsgCount";
  }

  string getDescription()
  {
    return "Returns the total number of messages send since last reset.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_txMsgCount();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the latest message fully received (for Line, Frame and Modbus protocols).
 *
 * @return a string corresponding to the latest message fully received (for Line, Frame and Modbus protocols)
 *
 * On failure, throws an exception or returns YSerialPort.LASTMSG_INVALID.
 */
class SerialPort_get_lastMsg : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_lastMsg(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_lastMsg";
  }

  string getDescription()
  {
    return "Returns the latest message fully received (for Line, Frame and Modbus protocols).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_lastMsg();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the name of the job file currently in use.
 *
 * @return a string corresponding to the name of the job file currently in use
 *
 * On failure, throws an exception or returns YSerialPort.CURRENTJOB_INVALID.
 */
class SerialPort_get_currentJob : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_currentJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_currentJob";
  }

  string getDescription()
  {
    return "Returns the name of the job file currently in use.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_currentJob();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Selects a job file to run immediately. If an empty string is
 * given as argument, stops running current job file.
 *
 * @param newval : a string
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SerialPort_set_currentJob : public YapiCommand /* arguments: newval */
{
public:
  SerialPort_set_currentJob(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_currentJob";
  }

  string getDescription()
  {
    return "Selects a job file to run immediately.";
  }

  string getMoreInfo()
  {
    return "If an empty string is given as argument, stops running current job file.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_currentJob(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the job file to use when the device is powered on.
 *
 * @return a string corresponding to the job file to use when the device is powered on
 *
 * On failure, throws an exception or returns YSerialPort.STARTUPJOB_INVALID.
 */
class SerialPort_get_startupJob : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_startupJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_startupJob";
  }

  string getDescription()
  {
    return "Returns the job file to use when the device is powered on.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_startupJob();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the job to use when the device is powered on.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the job to use when the device is powered on
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SerialPort_set_startupJob : public YapiCommand /* arguments: newval */
{
public:
  SerialPort_set_startupJob(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_startupJob";
  }

  string getDescription()
  {
    return "Changes the job to use when the device is powered on.";
  }

  string getMoreInfo()
  {
    return "Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the job to use when the device is powered on", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_startupJob(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the maximum number of tasks in a job that the device can handle.
 *
 * @return an integer corresponding to the maximum number of tasks in a job that the device can handle
 *
 * On failure, throws an exception or returns YSerialPort.JOBMAXTASK_INVALID.
 */
class SerialPort_get_jobMaxTask : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_jobMaxTask(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_jobMaxTask";
  }

  string getDescription()
  {
    return "Returns the maximum number of tasks in a job that the device can handle.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_jobMaxTask();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns maximum size allowed for job files.
 *
 * @return an integer corresponding to maximum size allowed for job files
 *
 * On failure, throws an exception or returns YSerialPort.JOBMAXSIZE_INVALID.
 */
class SerialPort_get_jobMaxSize : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_jobMaxSize(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_jobMaxSize";
  }

  string getDescription()
  {
    return "Returns maximum size allowed for job files.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_jobMaxSize();
        char buf[32]; SAFE_SPRINTF(buf, 32, "%d", attrval); string strval =  string(buf);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Returns the type of protocol used over the serial line, as a string.
 * Possible values are "Line" for ASCII messages separated by CR and/or LF,
 * "StxEtx" for ASCII messages delimited by STX/ETX codes,
 * "Frame:[timeout]ms" for binary messages separated by a delay time,
 * "Modbus-ASCII" for MODBUS messages in ASCII mode,
 * "Modbus-RTU" for MODBUS messages in RTU mode,
 * "Wiegand-ASCII" for Wiegand messages in ASCII mode,
 * "Wiegand-26","Wiegand-34", etc for Wiegand messages in byte mode,
 * "Char" for a continuous ASCII stream or
 * "Byte" for a continuous binary stream.
 *
 * @return a string corresponding to the type of protocol used over the serial line, as a string
 *
 * On failure, throws an exception or returns YSerialPort.PROTOCOL_INVALID.
 */
class SerialPort_get_protocol : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_protocol(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_protocol";
  }

  string getDescription()
  {
    return "Returns the type of protocol used over the serial line, as a string.";
  }

  string getMoreInfo()
  {
    return "Possible values are \"Line\" for ASCII messages separated by CR and/or LF, \"StxEtx\" for ASCII messages delimited by STX/ETX codes, \"Frame:[timeout]ms\" for binary messages separated by a delay time, \"Modbus-ASCII\" for MODBUS messages in ASCII mode, \"Modbus-RTU\" for MODBUS messages in RTU mode, \"Wiegand-ASCII\" for Wiegand messages in ASCII mode, \"Wiegand-26\",\"Wiegand-34\", etc for Wiegand messages in byte mode, \"Char\" for a continuous ASCII stream or \"Byte\" for a continuous binary stream.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_protocol();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the type of protocol used over the serial line.
 * Possible values are "Line" for ASCII messages separated by CR and/or LF,
 * "StxEtx" for ASCII messages delimited by STX/ETX codes,
 * "Frame:[timeout]ms" for binary messages separated by a delay time,
 * "Modbus-ASCII" for MODBUS messages in ASCII mode,
 * "Modbus-RTU" for MODBUS messages in RTU mode,
 * "Wiegand-ASCII" for Wiegand messages in ASCII mode,
 * "Wiegand-26","Wiegand-34", etc for Wiegand messages in byte mode,
 * "Char" for a continuous ASCII stream or
 * "Byte" for a continuous binary stream.
 * The suffix "/[wait]ms" can be added to reduce the transmit rate so that there
 * is always at lest the specified number of milliseconds between each bytes sent.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the type of protocol used over the serial line
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SerialPort_set_protocol : public YapiCommand /* arguments: newval */
{
public:
  SerialPort_set_protocol(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_protocol";
  }

  string getDescription()
  {
    return "Changes the type of protocol used over the serial line.";
  }

  string getMoreInfo()
  {
    return "Possible values are \"Line\" for ASCII messages separated by CR and/or LF, \"StxEtx\" for ASCII messages delimited by STX/ETX codes, \"Frame:[timeout]ms\" for binary messages separated by a delay time, \"Modbus-ASCII\" for MODBUS messages in ASCII mode, \"Modbus-RTU\" for MODBUS messages in RTU mode, \"Wiegand-ASCII\" for Wiegand messages in ASCII mode, \"Wiegand-26\",\"Wiegand-34\", etc for Wiegand messages in byte mode, \"Char\" for a continuous ASCII stream or \"Byte\" for a continuous binary stream. The suffix \"/[wait]ms\" can be added to reduce the transmit rate so that there is always at lest the specified number of milliseconds between each bytes sent. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the type of protocol used over the serial line", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_protocol(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the voltage level used on the serial line.
 *
 * @return a value among YSerialPort.VOLTAGELEVEL_OFF, YSerialPort.VOLTAGELEVEL_TTL3V,
 * YSerialPort.VOLTAGELEVEL_TTL3VR, YSerialPort.VOLTAGELEVEL_TTL5V, YSerialPort.VOLTAGELEVEL_TTL5VR,
 * YSerialPort.VOLTAGELEVEL_RS232, YSerialPort.VOLTAGELEVEL_RS485, YSerialPort.VOLTAGELEVEL_TTL1V8 and
 * YSerialPort.VOLTAGELEVEL_SDI12 corresponding to the voltage level used on the serial line
 *
 * On failure, throws an exception or returns YSerialPort.VOLTAGELEVEL_INVALID.
 */
class SerialPort_get_voltageLevel : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_voltageLevel(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_voltageLevel";
  }

  string getDescription()
  {
    return "Returns the voltage level used on the serial line.";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int attrval = (*list)[i]->get_voltageLevel();
        string strval =  enumString(attrval,enumSerialVoltageLevel);
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the voltage type used on the serial line. Valid
 * values  will depend on the Yoctopuce device model featuring
 * the serial port feature.  Check your device documentation
 * to find out which values are valid for that specific model.
 * Trying to set an invalid value will have no effect.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a value among YSerialPort.VOLTAGELEVEL_OFF, YSerialPort.VOLTAGELEVEL_TTL3V,
 * YSerialPort.VOLTAGELEVEL_TTL3VR, YSerialPort.VOLTAGELEVEL_TTL5V, YSerialPort.VOLTAGELEVEL_TTL5VR,
 * YSerialPort.VOLTAGELEVEL_RS232, YSerialPort.VOLTAGELEVEL_RS485, YSerialPort.VOLTAGELEVEL_TTL1V8 and
 * YSerialPort.VOLTAGELEVEL_SDI12 corresponding to the voltage type used on the serial line
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SerialPort_set_voltageLevel : public YapiCommand /* arguments: newval */
{
public:
  SerialPort_set_voltageLevel(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_voltageLevel";
  }

  string getDescription()
  {
    return "Changes the voltage type used on the serial line.";
  }

  string getMoreInfo()
  {
    return "Valid values will depend on the Yoctopuce device model featuring the serial port feature. Check your device documentation to find out which values are valid for that specific model. Trying to set an invalid value will have no effect. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "newval", "a value among OFF, TTL3V, TTL3VR, TTL5V, TTL5VR, RS232, RS485, TTL1V8 and SDI12 corresponding to the voltage type used on the serial line", "OFF=0,TTL3V=1,TTL3VR=2,TTL5V=3,TTL5VR=4,RS232=5,RS485=6,TTL1V8=7,SDI12=8", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_voltageLevel(Y_VOLTAGELEVEL_enum((*args)[0]->get_intValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

/**
 * Returns the serial port communication parameters, as a string such as
 * "9600,8N1". The string includes the baud rate, the number of data bits,
 * the parity, and the number of stop bits. An optional suffix is included
 * if flow control is active: "CtsRts" for hardware handshake, "XOnXOff"
 * for logical flow control and "Simplex" for acquiring a shared bus using
 * the RTS line (as used by some RS485 adapters for instance).
 *
 * @return a string corresponding to the serial port communication parameters, as a string such as
 *         "9600,8N1"
 *
 * On failure, throws an exception or returns YSerialPort.SERIALMODE_INVALID.
 */
class SerialPort_get_serialMode : public YapiCommand /* arguments: */
{
public:
  SerialPort_get_serialMode(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialMode";
  }

  string getDescription()
  {
    return "Returns the serial port communication parameters, as a string such as\"9600,8N1\".";
  }

  string getMoreInfo()
  {
    return "The string includes the baud rate, the number of data bits, the parity, and the number of stop bits. An optional suffix is included if flow control is active: \"CtsRts\" for hardware handshake, \"XOnXOff\" for logical flow control and \"Simplex\" for acquiring a shared bus using the RTS line (as used by some RS485 adapters for instance).";
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches )
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string attrval = (*list)[i]->get_serialMode();
        string strval =  attrval;
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), strval, true);
      }
  }
};

/**
 * Changes the serial port communication parameters, with a string such as
 * "9600,8N1". The string includes the baud rate, the number of data bits,
 * the parity, and the number of stop bits. An optional suffix can be added
 * to enable flow control: "CtsRts" for hardware handshake, "XOnXOff"
 * for logical flow control and "Simplex" for acquiring a shared bus using
 * the RTS line (as used by some RS485 adapters for instance).
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @param newval : a string corresponding to the serial port communication parameters, with a string such as
 *         "9600,8N1"
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class SerialPort_set_serialMode : public YapiCommand /* arguments: newval */
{
public:
  SerialPort_set_serialMode(YFunctionCmdLine *function):YapiCommand(function){}
  string getName()
  {
    return "set_serialMode";
  }

  string getDescription()
  {
    return "Changes the serial port communication parameters, with a string such as\"9600,8N1\".";
  }

  string getMoreInfo()
  {
    return "The string includes the baud rate, the number of data bits, the parity, and the number of stop bits. An optional suffix can be added to enable flow control: \"CtsRts\" for hardware handshake, \"XOnXOff\" for logical flow control and \"Simplex\" for acquiring a shared bus using the RTS line (as used by some RS485 adapters for instance). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "newval", "a string corresponding to the serial port communication parameters, with a string such as \"9600,8N1\"", "_STRING", false));
    return res;
  }

  void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++) {
        (*list)[i]->set_serialMode(string((*args)[0]->get_stringValue()));
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
    }
  }
};

// type of valueCallbackSerialPort is YSerialPortValueCallback* (YSerialPortValueCallback)
// type of rxptr is int (int)
// type of rxbuff is bin (bin)
// type of rxbuffptr is int (int)
// type of eventPos is int (int)
// type of eventCallback is YSnoopingCallback* (YSnoopingCallback)
/**
 * Disables the propagation of every new advertised value to the parent hub.
 * You can use this function to save bandwidth and CPU on computers with limited
 * resources, or to prevent unwanted invocations of the HTTP callback.
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_muteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_muteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "muteValueCallbacks";
  }

  string getDescription()
  {
    return "Disables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "You can use this function to save bandwidth and CPU on computers with limited resources, or to prevent unwanted invocations of the HTTP callback. Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->muteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Re-enables the propagation of every new advertised value to the parent hub.
 * This function reverts the effect of a previous call to muteValueCallbacks().
 * Remember to call the saveToFlash() method of the module if the
 * modification must be kept.
 *
 * @return YAPI.SUCCESS when the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_unmuteValueCallbacks : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_unmuteValueCallbacks(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "unmuteValueCallbacks";
  }

  string getDescription()
  {
    return "Re-enables the propagation of every new advertised value to the parent hub.";
  }

  string getMoreInfo()
  {
    return "This function reverts the effect of a previous call to muteValueCallbacks(). Remember to call \"YModule {target} saveToFlash\" if the modification must be kept.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->unmuteValueCallbacks();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Indicates whether changes to the function are prohibited or allowed.
 * Returns true if the function is blocked by an admin password
 * or if the function is not available.
 *
 * @return true if the function is write-protected or not online.
 */
class apifun_SerialPort_isReadOnly : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_isReadOnly(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "isReadOnly";
  }

  string getDescription()
  {
    return "Indicates whether changes to the function are prohibited or allowed.";
  }

  string getMoreInfo()
  {
    return "Returns true if the function is blocked by an admin password or if the function is not available.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        bool value = (*list)[i]->isReadOnly();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the serial number of the module, as set by the factory.
 *
 * @return a string corresponding to the serial number of the module, as set by the factory.
 *
 * On failure, throws an exception or returns YFunction.SERIALNUMBER_INVALID.
 */
class apifun_SerialPort_get_serialNumber : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_get_serialNumber(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_serialNumber";
  }

  string getDescription()
  {
    return "Returns the serial number of the module, as set by the factory.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->get_serialNumber();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads a single line (or message) from the receive buffer, starting at current stream position.
 * This function is intended to be used when the serial port is configured for a message protocol,
 * such as 'Line' mode or frame protocols.
 *
 * If data at current stream position is not available anymore in the receive buffer,
 * the function returns the oldest available line and moves the stream position just after.
 * If no new full line is received, the function returns an empty line.
 *
 * @return a string with a single line of text
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_readLine : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_readLine(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readLine";
  }

  string getDescription()
  {
    return "Reads a single line (or message) from the receive buffer, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "This function is intended to be used when the serial port is configured for a message protocol, such as 'Line' mode or frame protocols.\nIf data at current stream position is not available anymore in the receive buffer, the function returns the oldest available line and moves the stream position just after. If no new full line is received, the function returns an empty line.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->readLine();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Searches for incoming messages in the serial port receive buffer matching a given pattern,
 * starting at current position. This function will only compare and return printable characters
 * in the message strings. Binary protocols are handled as hexadecimal strings.
 *
 * The search returns all messages matching the expression provided as argument in the buffer.
 * If no matching message is found, the search waits for one up to the specified maximum timeout
 * (in milliseconds).
 *
 * @param pattern : a limited regular expression describing the expected message format,
 *         or an empty string if all messages should be returned (no filtering).
 *         When using binary protocols, the format applies to the hexadecimal
 *         representation of the message.
 * @param maxWait : the maximum number of milliseconds to wait for a message if none is found
 *         in the receive buffer.
 *
 * @return an array of strings containing the messages found, if any.
 *         Binary messages are converted to hexadecimal representation.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_readMessages : public YapiCommand /* arguments: pattern maxWait */
{
public:
  apifun_SerialPort_readMessages(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readMessages";
  }

  string getDescription()
  {
    return "Searches for incoming messages in the serial port receive buffer matching a given pattern,starting at current position.";
  }

  string getMoreInfo()
  {
    return "This function will only compare and return printable characters in the message strings. Binary protocols are handled as hexadecimal strings.\nThe search returns all messages matching the expression provided as argument in the buffer. If no matching message is found, the search waits for one up to the specified maximum timeout (in milliseconds).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "pattern", "a limited regular expression describing the expected message format, or an empty string if all messages should be returned (no filtering). When using binary protocols, the format applies to the hexadecimal representation of the message.", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a message if none is found in the receive buffer.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<string> value = (*list)[i]->readMessages((*args)[0]->get_stringValue(),
                                                        (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Changes the current internal stream position to the specified value. This function
 * does not affect the device, it only changes the value stored in the API object
 * for the next read operations.
 *
 * @param absPos : the absolute position index for next read operations.
 *
 * @return nothing.
 */
class apifun_SerialPort_read_seek : public YapiCommand /* arguments: absPos */
{
public:
  apifun_SerialPort_read_seek(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "read_seek";
  }

  string getDescription()
  {
    return "Changes the current internal stream position to the specified value.";
  }

  string getMoreInfo()
  {
    return "This function does not affect the device, it only changes the value stored in the API object for the next read operations.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "absPos", "the absolute position index for next read operations.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->read_seek((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the current absolute stream position pointer of the API object.
 *
 * @return the absolute position index for next read operations.
 */
class apifun_SerialPort_read_tell : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_read_tell(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "read_tell";
  }

  string getDescription()
  {
    return "Returns the current absolute stream position pointer of the API object.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->read_tell();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Returns the number of bytes available to read in the input buffer starting from the
 * current absolute stream position pointer of the API object.
 *
 * @return the number of bytes available to read
 */
class apifun_SerialPort_read_avail : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_read_avail(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "read_avail";
  }

  string getDescription()
  {
    return "Returns the number of bytes available to read in the input buffer starting from thecurrent absolute stream position pointer of the API object.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->read_avail();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a text line query to the serial port, and reads the reply, if any.
 * This function is intended to be used when the serial port is configured for 'Line' protocol.
 *
 * @param query : the line query to send (without CR/LF)
 * @param maxWait : the maximum number of milliseconds to wait for a reply.
 *
 * @return the next text line received after sending the text query, as a string.
 *         Additional lines can be obtained by calling readLine or readMessages.
 *
 * On failure, throws an exception or returns an empty string.
 */
class apifun_SerialPort_queryLine : public YapiCommand /* arguments: query maxWait */
{
public:
  apifun_SerialPort_queryLine(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "queryLine";
  }

  string getDescription()
  {
    return "Sends a text line query to the serial port, and reads the reply, if any.";
  }

  string getMoreInfo()
  {
    return "This function is intended to be used when the serial port is configured for 'Line' protocol.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "query", "the line query to send (without CR/LF)", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a reply.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->queryLine((*args)[0]->get_stringValue(),
                                             (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a binary message to the serial port, and reads the reply, if any.
 * This function is intended to be used when the serial port is configured for
 * Frame-based protocol.
 *
 * @param hexString : the message to send, coded in hexadecimal
 * @param maxWait : the maximum number of milliseconds to wait for a reply.
 *
 * @return the next frame received after sending the message, as a hex string.
 *         Additional frames can be obtained by calling readHex or readMessages.
 *
 * On failure, throws an exception or returns an empty string.
 */
class apifun_SerialPort_queryHex : public YapiCommand /* arguments: hexString maxWait */
{
public:
  apifun_SerialPort_queryHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "queryHex";
  }

  string getDescription()
  {
    return "Sends a binary message to the serial port, and reads the reply, if any.";
  }

  string getMoreInfo()
  {
    return "This function is intended to be used when the serial port is configured for Frame-based protocol.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "hexString", "the message to send, coded in hexadecimal", "_STRING", false));
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a reply.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->queryHex((*args)[0]->get_stringValue(),
                                            (int)(*args)[1]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Saves the job definition string (JSON data) into a job file.
 * The job file can be later enabled using selectJob().
 *
 * @param jobfile : name of the job file to save on the device filesystem
 * @param jsonDef : a string containing a JSON definition of the job
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_uploadJob : public YapiCommand /* arguments: jobfile jsonDef */
{
public:
  apifun_SerialPort_uploadJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "uploadJob";
  }

  string getDescription()
  {
    return "Saves the job definition string (JSON data) into a job file.";
  }

  string getMoreInfo()
  {
    return "The job file can be later enabled using selectJob().";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "jobfile", "name of the job file to save on the device filesystem", "_STRING", false));
    res->push_back(new ArgumentDesc(STRING_ARG, "jsonDef", "a string containing a JSON definition of the job", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->uploadJob((*args)[0]->get_stringValue(),
                              (*args)[1]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Load and start processing the specified job file. The file must have
 * been previously created using the user interface or uploaded on the
 * device filesystem using the uploadJob() function.
 *
 * @param jobfile : name of the job file (on the device filesystem)
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_selectJob : public YapiCommand /* arguments: jobfile */
{
public:
  apifun_SerialPort_selectJob(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "selectJob";
  }

  string getDescription()
  {
    return "Load and start processing the specified job file.";
  }

  string getMoreInfo()
  {
    return "The file must have been previously created using the user interface or uploaded on the device filesystem using the uploadJob() function.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "jobfile", "name of the job file (on the device filesystem)", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->selectJob((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Clears the serial port buffer and resets counters to zero.
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_reset : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_reset(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "reset";
  }

  string getDescription()
  {
    return "Clears the serial port buffer and resets counters to zero.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->reset();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a single byte to the serial port.
 *
 * @param code : the byte to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeByte : public YapiCommand /* arguments: code */
{
public:
  apifun_SerialPort_writeByte(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeByte";
  }

  string getDescription()
  {
    return "Sends a single byte to the serial port.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "code", "the byte to send", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeByte((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends an ASCII string to the serial port, as is.
 *
 * @param text : the text string to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeStr : public YapiCommand /* arguments: text */
{
public:
  apifun_SerialPort_writeStr(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeStr";
  }

  string getDescription()
  {
    return "Sends an ASCII string to the serial port, as is.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "text", "the text string to send", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeStr((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a binary buffer to the serial port, as is.
 *
 * @param buff : the binary buffer to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeBin : public YapiCommand /* arguments: buff */
{
public:
  apifun_SerialPort_writeBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeBin";
  }

  string getDescription()
  {
    return "Sends a binary buffer to the serial port, as is.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(BIN_ARG, "buff", "the binary buffer to send", "_BIN", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeBin((*args)[0]->get_binValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a byte sequence (provided as a list of bytes) to the serial port.
 *
 * @param byteList : a list of byte codes
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeArray : public YapiCommand /* arguments: byteList */
{
public:
  apifun_SerialPort_writeArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeArray";
  }

  string getDescription()
  {
    return "Sends a byte sequence (provided as a list of bytes) to the serial port.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INTARR_ARG, "byteList", "a list of byte codes", "_INTARR", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeArray((*args)[0]->get_intArrValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a byte sequence (provided as a hexadecimal string) to the serial port.
 *
 * @param hexString : a string of hexadecimal byte codes
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeHex : public YapiCommand /* arguments: hexString */
{
public:
  apifun_SerialPort_writeHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeHex";
  }

  string getDescription()
  {
    return "Sends a byte sequence (provided as a hexadecimal string) to the serial port.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "hexString", "a string of hexadecimal byte codes", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeHex((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends an ASCII string to the serial port, followed by a line break (CR LF).
 *
 * @param text : the text string to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeLine : public YapiCommand /* arguments: text */
{
public:
  apifun_SerialPort_writeLine(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeLine";
  }

  string getDescription()
  {
    return "Sends an ASCII string to the serial port, followed by a line break (CR LF).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "text", "the text string to send", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeLine((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads one byte from the receive buffer, starting at current stream position.
 * If data at current stream position is not available anymore in the receive buffer,
 * or if there is no data available yet, the function returns YAPI_NO_MORE_DATA.
 *
 * @return the next byte
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_readByte : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_readByte(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readByte";
  }

  string getDescription()
  {
    return "Reads one byte from the receive buffer, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "If data at current stream position is not available anymore in the receive buffer, or if there is no data available yet, the function returns YAPI_NO_MORE_DATA.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->readByte();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads data from the receive buffer as a string, starting at current stream position.
 * If data at current stream position is not available anymore in the receive buffer, the
 * function performs a short read.
 *
 * @param nChars : the maximum number of characters to read
 *
 * @return a string with receive buffer contents
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_readStr : public YapiCommand /* arguments: nChars */
{
public:
  apifun_SerialPort_readStr(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readStr";
  }

  string getDescription()
  {
    return "Reads data from the receive buffer as a string, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "If data at current stream position is not available anymore in the receive buffer, the function performs a short read.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "nChars", "the maximum number of characters to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->readStr((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads data from the receive buffer as a binary buffer, starting at current stream position.
 * If data at current stream position is not available anymore in the receive buffer, the
 * function performs a short read.
 *
 * @param nChars : the maximum number of bytes to read
 *
 * @return a binary object with receive buffer contents
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_readBin : public YapiCommand /* arguments: nChars */
{
public:
  apifun_SerialPort_readBin(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readBin";
  }

  string getDescription()
  {
    return "Reads data from the receive buffer as a binary buffer, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "If data at current stream position is not available anymore in the receive buffer, the function performs a short read.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "nChars", "the maximum number of bytes to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->readBin((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads data from the receive buffer as a list of bytes, starting at current stream position.
 * If data at current stream position is not available anymore in the receive buffer, the
 * function performs a short read.
 *
 * @param nChars : the maximum number of bytes to read
 *
 * @return a sequence of bytes with receive buffer contents
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_readArray : public YapiCommand /* arguments: nChars */
{
public:
  apifun_SerialPort_readArray(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readArray";
  }

  string getDescription()
  {
    return "Reads data from the receive buffer as a list of bytes, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "If data at current stream position is not available anymore in the receive buffer, the function performs a short read.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "nChars", "the maximum number of bytes to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->readArray((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads data from the receive buffer as a hexadecimal string, starting at current stream position.
 * If data at current stream position is not available anymore in the receive buffer, the
 * function performs a short read.
 *
 * @param nBytes : the maximum number of bytes to read
 *
 * @return a string with receive buffer contents, encoded in hexadecimal
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_readHex : public YapiCommand /* arguments: nBytes */
{
public:
  apifun_SerialPort_readHex(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "readHex";
  }

  string getDescription()
  {
    return "Reads data from the receive buffer as a hexadecimal string, starting at current stream position.";
  }

  string getMoreInfo()
  {
    return "If data at current stream position is not available anymore in the receive buffer, the function performs a short read.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "nBytes", "the maximum number of bytes to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        string value = (*list)[i]->readHex((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Emits a BREAK condition on the serial interface. When the specified
 * duration is 0, the BREAK signal will be exactly one character wide.
 * When the duration is between 1 and 100, the BREAK condition will
 * be hold for the specified number of milliseconds.
 *
 * @param duration : 0 for a standard BREAK, or duration between 1 and 100 ms
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_sendBreak : public YapiCommand /* arguments: duration */
{
public:
  apifun_SerialPort_sendBreak(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "sendBreak";
  }

  string getDescription()
  {
    return "Emits a BREAK condition on the serial interface.";
  }

  string getMoreInfo()
  {
    return "When the specified duration is 0, the BREAK signal will be exactly one character wide. When the duration is between 1 and 100, the BREAK condition will be hold for the specified number of milliseconds.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "duration", "0 for a standard BREAK, or duration between 1 and 100 ms", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->sendBreak((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Manually sets the state of the RTS line. This function has no effect when
 * hardware handshake is enabled, as the RTS line is driven automatically.
 *
 * @param val : 1 to turn RTS on, 0 to turn RTS off
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_set_RTS : public YapiCommand /* arguments: val */
{
public:
  apifun_SerialPort_set_RTS(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "set_RTS";
  }

  string getDescription()
  {
    return "Manually sets the state of the RTS line.";
  }

  string getMoreInfo()
  {
    return "This function has no effect when hardware handshake is enabled, as the RTS line is driven automatically.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "val", "1 to turn RTS on, 0 to turn RTS off", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->set_RTS((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads the level of the CTS line. The CTS line is usually driven by
 * the RTS signal of the connected serial device.
 *
 * @return 1 if the CTS line is high, 0 if the CTS line is low.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_get_CTS : public YapiCommand /* arguments: */
{
public:
  apifun_SerialPort_get_CTS(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "get_CTS";
  }

  string getDescription()
  {
    return "Reads the level of the CTS line.";
  }

  string getMoreInfo()
  {
    return "The CTS line is usually driven by the RTS signal of the connected serial device.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        int value = (*list)[i]->get_CTS();
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Retrieves messages (both direction) in the serial port buffer, starting at current position.
 * This function will only compare and return printable characters in the message strings.
 * Binary protocols are handled as hexadecimal strings.
 *
 * If no message is found, the search waits for one up to the specified maximum timeout
 * (in milliseconds).
 *
 * @param maxWait : the maximum number of milliseconds to wait for a message if none is found
 *         in the receive buffer.
 *
 * @return an array of YSnoopingRecord objects containing the messages found, if any.
 *         Binary messages are converted to hexadecimal representation.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_snoopMessages : public YapiCommand /* arguments: maxWait */
{
public:
  apifun_SerialPort_snoopMessages(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "snoopMessages";
  }

  string getDescription()
  {
    return "Retrieves messages (both direction) in the serial port buffer, starting at current position.";
  }

  string getMoreInfo()
  {
    return "This function will only compare and return printable characters in the message strings. Binary protocols are handled as hexadecimal strings.\nIf no message is found, the search waits for one up to the specified maximum timeout (in milliseconds).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "maxWait", "the maximum number of milliseconds to wait for a message if none is found in the receive buffer.", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<YSnoopingRecord> value = (*list)[i]->snoopMessages((int)(*args)[0]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends an ASCII string to the serial port, preceeded with an STX code and
 * followed by an ETX code.
 *
 * @param text : the text string to send
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeStxEtx : public YapiCommand /* arguments: text */
{
public:
  apifun_SerialPort_writeStxEtx(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeStxEtx";
  }

  string getDescription()
  {
    return "Sends an ASCII string to the serial port, preceeded with an STX code andfollowed by an ETX code.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "text", "the text string to send", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeStxEtx((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a MODBUS message (provided as a hexadecimal string) to the serial port.
 * The message must start with the slave address. The MODBUS CRC/LRC is
 * automatically added by the function. This function does not wait for a reply.
 *
 * @param hexString : a hexadecimal message string, including device address but no CRC/LRC
 *
 * @return YAPI.SUCCESS if the call succeeds.
 *
 * On failure, throws an exception or returns a negative error code.
 */
class apifun_SerialPort_writeMODBUS : public YapiCommand /* arguments: hexString */
{
public:
  apifun_SerialPort_writeMODBUS(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "writeMODBUS";
  }

  string getDescription()
  {
    return "Sends a MODBUS message (provided as a hexadecimal string) to the serial port.";
  }

  string getMoreInfo()
  {
    return "The message must start with the slave address. The MODBUS CRC/LRC is automatically added by the function. This function does not wait for a reply.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(STRING_ARG, "hexString", "a hexadecimal message string, including device address but no CRC/LRC", "_STRING", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->writeMODBUS((*args)[0]->get_stringValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sends a message to a specified MODBUS slave connected to the serial port, and reads the
 * reply, if any. The message is the PDU, provided as a vector of bytes.
 *
 * @param slaveNo : the address of the slave MODBUS device to query
 * @param pduBytes : the message to send (PDU), as a vector of bytes. The first byte of the
 *         PDU is the MODBUS function code.
 *
 * @return the received reply, as a vector of bytes.
 *
 * On failure, throws an exception or returns an empty array (or a MODBUS error reply).
 */
class apifun_SerialPort_queryMODBUS : public YapiCommand /* arguments: slaveNo pduBytes */
{
public:
  apifun_SerialPort_queryMODBUS(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "queryMODBUS";
  }

  string getDescription()
  {
    return "Sends a message to a specified MODBUS slave connected to the serial port, and reads thereply, if any.";
  }

  string getMoreInfo()
  {
    return "The message is the PDU, provided as a vector of bytes.";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to query", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "pduBytes", "the message to send (PDU), as a vector of bytes. The first byte of the PDU is the MODBUS function code.", "_INTARR", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->queryMODBUS((int)(*args)[0]->get_intValue(),
                                                    (*args)[1]->get_intArrValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads one or more contiguous internal bits (or coil status) from a MODBUS serial device.
 * This method uses the MODBUS function code 0x01 (Read Coils).
 *
 * @param slaveNo : the address of the slave MODBUS device to query
 * @param pduAddr : the relative address of the first bit/coil to read (zero-based)
 * @param nBits : the number of bits/coils to read
 *
 * @return a vector of integers, each corresponding to one bit.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_modbusReadBits : public YapiCommand /* arguments: slaveNo pduAddr nBits */
{
public:
  apifun_SerialPort_modbusReadBits(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusReadBits";
  }

  string getDescription()
  {
    return "Reads one or more contiguous internal bits (or coil status) from a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x01 (Read Coils).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to query", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the first bit/coil to read (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBits", "the number of bits/coils to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->modbusReadBits((int)(*args)[0]->get_intValue(),
                                                       (int)(*args)[1]->get_intValue(),
                                                       (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads one or more contiguous input bits (or discrete inputs) from a MODBUS serial device.
 * This method uses the MODBUS function code 0x02 (Read Discrete Inputs).
 *
 * @param slaveNo : the address of the slave MODBUS device to query
 * @param pduAddr : the relative address of the first bit/input to read (zero-based)
 * @param nBits : the number of bits/inputs to read
 *
 * @return a vector of integers, each corresponding to one bit.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_modbusReadInputBits : public YapiCommand /* arguments: slaveNo pduAddr nBits */
{
public:
  apifun_SerialPort_modbusReadInputBits(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusReadInputBits";
  }

  string getDescription()
  {
    return "Reads one or more contiguous input bits (or discrete inputs) from a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x02 (Read Discrete Inputs).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to query", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the first bit/input to read (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nBits", "the number of bits/inputs to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->modbusReadInputBits((int)(*args)[0]->get_intValue(),
                                                            (int)(*args)[1]->get_intValue(),
                                                            (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads one or more contiguous internal registers (holding registers) from a MODBUS serial device.
 * This method uses the MODBUS function code 0x03 (Read Holding Registers).
 *
 * @param slaveNo : the address of the slave MODBUS device to query
 * @param pduAddr : the relative address of the first holding register to read (zero-based)
 * @param nWords : the number of holding registers to read
 *
 * @return a vector of integers, each corresponding to one 16-bit register value.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_modbusReadRegisters : public YapiCommand /* arguments: slaveNo pduAddr nWords */
{
public:
  apifun_SerialPort_modbusReadRegisters(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusReadRegisters";
  }

  string getDescription()
  {
    return "Reads one or more contiguous internal registers (holding registers) from a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x03 (Read Holding Registers).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to query", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the first holding register to read (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nWords", "the number of holding registers to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->modbusReadRegisters((int)(*args)[0]->get_intValue(),
                                                            (int)(*args)[1]->get_intValue(),
                                                            (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Reads one or more contiguous input registers (read-only registers) from a MODBUS serial device.
 * This method uses the MODBUS function code 0x04 (Read Input Registers).
 *
 * @param slaveNo : the address of the slave MODBUS device to query
 * @param pduAddr : the relative address of the first input register to read (zero-based)
 * @param nWords : the number of input registers to read
 *
 * @return a vector of integers, each corresponding to one 16-bit input value.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_modbusReadInputRegisters : public YapiCommand /* arguments: slaveNo pduAddr nWords */
{
public:
  apifun_SerialPort_modbusReadInputRegisters(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusReadInputRegisters";
  }

  string getDescription()
  {
    return "Reads one or more contiguous input registers (read-only registers) from a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x04 (Read Input Registers).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to query", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the first input register to read (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nWords", "the number of input registers to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->modbusReadInputRegisters((int)(*args)[0]->get_intValue(),
                                                                 (int)(*args)[1]->get_intValue(),
                                                                 (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sets a single internal bit (or coil) on a MODBUS serial device.
 * This method uses the MODBUS function code 0x05 (Write Single Coil).
 *
 * @param slaveNo : the address of the slave MODBUS device to drive
 * @param pduAddr : the relative address of the bit/coil to set (zero-based)
 * @param value : the value to set (0 for OFF state, non-zero for ON state)
 *
 * @return the number of bits/coils affected on the device (1)
 *
 * On failure, throws an exception or returns zero.
 */
class apifun_SerialPort_modbusWriteBit : public YapiCommand /* arguments: slaveNo pduAddr value */
{
public:
  apifun_SerialPort_modbusWriteBit(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusWriteBit";
  }

  string getDescription()
  {
    return "Sets a single internal bit (or coil) on a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x05 (Write Single Coil).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to drive", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the bit/coil to set (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "value", "the value to set (0 for OFF state, non-zero for ON state)", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->modbusWriteBit((int)(*args)[0]->get_intValue(),
                                   (int)(*args)[1]->get_intValue(),
                                   (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sets several contiguous internal bits (or coils) on a MODBUS serial device.
 * This method uses the MODBUS function code 0x0f (Write Multiple Coils).
 *
 * @param slaveNo : the address of the slave MODBUS device to drive
 * @param pduAddr : the relative address of the first bit/coil to set (zero-based)
 * @param bits : the vector of bits to be set (one integer per bit)
 *
 * @return the number of bits/coils affected on the device
 *
 * On failure, throws an exception or returns zero.
 */
class apifun_SerialPort_modbusWriteBits : public YapiCommand /* arguments: slaveNo pduAddr bits */
{
public:
  apifun_SerialPort_modbusWriteBits(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusWriteBits";
  }

  string getDescription()
  {
    return "Sets several contiguous internal bits (or coils) on a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x0f (Write Multiple Coils).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to drive", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the first bit/coil to set (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "bits", "the vector of bits to be set (one integer per bit)", "_INTARR", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->modbusWriteBits((int)(*args)[0]->get_intValue(),
                                    (int)(*args)[1]->get_intValue(),
                                    (*args)[2]->get_intArrValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sets a single internal register (or holding register) on a MODBUS serial device.
 * This method uses the MODBUS function code 0x06 (Write Single Register).
 *
 * @param slaveNo : the address of the slave MODBUS device to drive
 * @param pduAddr : the relative address of the register to set (zero-based)
 * @param value : the 16 bit value to set
 *
 * @return the number of registers affected on the device (1)
 *
 * On failure, throws an exception or returns zero.
 */
class apifun_SerialPort_modbusWriteRegister : public YapiCommand /* arguments: slaveNo pduAddr value */
{
public:
  apifun_SerialPort_modbusWriteRegister(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusWriteRegister";
  }

  string getDescription()
  {
    return "Sets a single internal register (or holding register) on a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x06 (Write Single Register).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to drive", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the register to set (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "value", "the 16 bit value to set", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->modbusWriteRegister((int)(*args)[0]->get_intValue(),
                                        (int)(*args)[1]->get_intValue(),
                                        (int)(*args)[2]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sets several contiguous internal registers (or holding registers) on a MODBUS serial device.
 * This method uses the MODBUS function code 0x10 (Write Multiple Registers).
 *
 * @param slaveNo : the address of the slave MODBUS device to drive
 * @param pduAddr : the relative address of the first internal register to set (zero-based)
 * @param values : the vector of 16 bit values to set
 *
 * @return the number of registers affected on the device
 *
 * On failure, throws an exception or returns zero.
 */
class apifun_SerialPort_modbusWriteRegisters : public YapiCommand /* arguments: slaveNo pduAddr values */
{
public:
  apifun_SerialPort_modbusWriteRegisters(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusWriteRegisters";
  }

  string getDescription()
  {
    return "Sets several contiguous internal registers (or holding registers) on a MODBUS serial device.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x10 (Write Multiple Registers).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to drive", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduAddr", "the relative address of the first internal register to set (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "values", "the vector of 16 bit values to set", "_INTARR", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    string value = "";
    value = value +  (*args)[0]->to_str();
    value = value + " "+ (*args)[1]->to_str();
    value = value + " "+ (*args)[2]->to_str();
    for (i = 0; i < list->size(); i++)
      {
        (*list)[i]->modbusWriteRegisters((int)(*args)[0]->get_intValue(),
                                         (int)(*args)[1]->get_intValue(),
                                         (*args)[2]->get_intArrValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

/**
 * Sets several contiguous internal registers (holding registers) on a MODBUS serial device,
 * then performs a contiguous read of a set of (possibly different) internal registers.
 * This method uses the MODBUS function code 0x17 (Read/Write Multiple Registers).
 *
 * @param slaveNo : the address of the slave MODBUS device to drive
 * @param pduWriteAddr : the relative address of the first internal register to set (zero-based)
 * @param values : the vector of 16 bit values to set
 * @param pduReadAddr : the relative address of the first internal register to read (zero-based)
 * @param nReadWords : the number of 16 bit values to read
 *
 * @return a vector of integers, each corresponding to one 16-bit register value read.
 *
 * On failure, throws an exception or returns an empty array.
 */
class apifun_SerialPort_modbusWriteAndReadRegisters : public YapiCommand /* arguments: slaveNo pduWriteAddr values pduReadAddr nReadWords */
{
public:
  apifun_SerialPort_modbusWriteAndReadRegisters(YFunctionCmdLine *function):YapiCommand(function){}

  string getName()
  {
    return "modbusWriteAndReadRegisters";
  }

  string getDescription()
  {
    return "Sets several contiguous internal registers (holding registers) on a MODBUS serial device,then performs a contiguous read of a set of (possibly different) internal registers.";
  }

  string getMoreInfo()
  {
    return "This method uses the MODBUS function code 0x17 (Read/Write Multiple Registers).";
  }

  vector<ArgumentDesc*>* getArgumentDesc()
  {
    vector<ArgumentDesc*>* res = new vector<ArgumentDesc*>();
    res->push_back(new ArgumentDesc(INT_ARG, "slaveNo", "the address of the slave MODBUS device to drive", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduWriteAddr", "the relative address of the first internal register to set (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INTARR_ARG, "values", "the vector of 16 bit values to set", "_INTARR", false));
    res->push_back(new ArgumentDesc(INT_ARG, "pduReadAddr", "the relative address of the first internal register to read (zero-based)", "_INT", false));
    res->push_back(new ArgumentDesc(INT_ARG, "nReadWords", "the number of 16 bit values to read", "_INT", false));
    return res;
  }

  virtual void execute(string target, vector<YModule*> *modulelist, string resultformat, vector<ArgumentDesc*>* args, vector<SwitchDesc*>* switches)
  {
    vector<YSerialPort*>* list = enumerateTargets<YSerialPort>(_function, target, modulelist);
    unsigned int i;
    for (i = 0; i < list->size(); i++)
      {
        vector<int> value = (*list)[i]->modbusWriteAndReadRegisters((int)(*args)[0]->get_intValue(),
                                                                    (int)(*args)[1]->get_intValue(),
                                                                    (*args)[2]->get_intArrValue(),
                                                                    (int)(*args)[3]->get_intValue(),
                                                                    (int)(*args)[4]->get_intValue());
        PrintResult(resultformat, this->getName(),YFunctionInfoCache((*list)[i]), value, true);
      }
  }
};

//--- (end of generated code: YSerialPort implementation)

//--- (generated code: YSerialPort functions)
void YSerialPortCmdLine::RegisterCommands(vector<YapiCommand*>* cmdList)
  {
    RegisterGenericFunctions<YSerialPort>(this, cmdList);
    cmdList->push_back((YapiCommand*) (new SerialPort_get_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_set_logicalName(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_advertisedValue(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_rxCount(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_txCount(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_errCount(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_rxMsgCount(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_txMsgCount(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_lastMsg(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_currentJob(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_set_currentJob(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_startupJob(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_set_startupJob(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_jobMaxTask(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_jobMaxSize(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_protocol(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_set_protocol(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_voltageLevel(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_set_voltageLevel(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_get_serialMode(this)));
    cmdList->push_back((YapiCommand*) (new SerialPort_set_serialMode(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_muteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_unmuteValueCallbacks(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_isReadOnly(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_get_serialNumber(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readLine(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readMessages(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_read_seek(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_read_tell(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_read_avail(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_queryLine(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_queryHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_uploadJob(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_selectJob(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_reset(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeByte(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeStr(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeLine(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readByte(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readStr(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readBin(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readArray(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_readHex(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_sendBreak(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_set_RTS(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_get_CTS(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_snoopMessages(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeStxEtx(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_writeMODBUS(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_queryMODBUS(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusReadBits(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusReadInputBits(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusReadRegisters(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusReadInputRegisters(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusWriteBit(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusWriteBits(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusWriteRegister(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusWriteRegisters(this)));
    cmdList->push_back((YapiCommand*) (new apifun_SerialPort_modbusWriteAndReadRegisters(this)));
  }

//--- (end of generated code: YSerialPort functions)
