// Yoctopuce hardware handling

#include "YAudioOuputSwitcher.h"

#include "logs.h"

#define notLinkedColor  0x400000
#define linkedColor     0x004000
#define ledsResetDelay  15000




std::string WstringTostring(std::wstring wStr)
{
  std::string str(wStr.length(), ' ');
  std::copy(wStr.begin(), wStr.end(), str.begin());
  return str;
}

// global callbacks allowing to call YAudioOuputSwitcher's methods

void  BTN_muteChange_global(YAnButton* fct, const string& value)
{
  YAudioOuputSwitcher* as = (YAudioOuputSwitcher*)fct->get_userData();
  as->BTN_muteChange(fct, value);
}
void  QD_out_change_global(YQuadratureDecoder* fct, const string& value)
{
  YAudioOuputSwitcher* as = (YAudioOuputSwitcher*)fct->get_userData();
  as->QD_out_change(fct, value);
}
void QD_Vol_change_global(YQuadratureDecoder* fct, const string& value)
{
  YAudioOuputSwitcher* as = (YAudioOuputSwitcher*)fct->get_userData();
  as->QD_Vol_change(fct, value);
}
void change_global(AudioOutputManager* source, ChangeCause cause)
{
  YAudioOuputSwitcher* as = (YAudioOuputSwitcher*)source->get_userData();
  as->change(source, cause);
}

YAudioOuputSwitcher* switcher = NULL;

static void deviceArrival_global(YModule* m)
{ if (switcher != NULL) switcher->deviceArrival(m);
}


// called by the AudioOutputmanager when volume knob is pressed
void  YAudioOuputSwitcher::BTN_muteChange(YAnButton* fct, const string& value)
{ if (value == "15")
  { this->manager->set_mute(!manager->get_mute());
  }
}

// called by the AudioOutputmanager when ouput knob is turned
void  YAudioOuputSwitcher::QD_out_change(YQuadratureDecoder* fct, const string& value)
{ int offset = stoi(value);
  if (offset != 0)
  { fct->set_currentValue(0);
    this->manager->set_defaultOutputRel(offset);
  }
}

// called by the AudioOutputmanager when volume knob is turned

void YAudioOuputSwitcher::QD_Vol_change(YQuadratureDecoder* fct, const string& value)
{ int offset = stoi(value);
  if (offset != 0)
  {
    fct->set_currentValue(0);
    this->manager->set_volume((float)(offset * 0.05));
  }
}


// called by the AudioOutputmanager when an AudioEnpoint  change is detected 
void YAudioOuputSwitcher::change(AudioOutputManager* source, ChangeCause cause)
{
  RefreshNeeded = true;
  if (cause == VOLUMECHANGE)  lastVolumeChange = GetTickCount64();
}


// constructor
YAudioOuputSwitcher::YAudioOuputSwitcher()
{

  this->initdone = "Lagaffe";
  this->state = ALLOCATED;
  this->manager = NULL;
  switcher = this;
}


// init the API and try to establish the connection
// with the hub
void YAudioOuputSwitcher::init(std::string ipaddress)
{
  string       errmsg;
  YAPI::InitAPI(YAPI::DETECT_NONE, errmsg);
  YAPI::RegisterDeviceArrivalCallback(deviceArrival_global);
  YAPI::PreregisterHub(ipaddress, errmsg);
  this->state = INIT_DONE;
  logs::log("init done");

}

// config the animation used to turn the led green when
// the connection is active and red when it is lost (watchog)
void YAudioOuputSwitcher::configure_ledWatchdog()
{
  try
  {
    leds->resetBlinkSeq(0);
    leds->addRgbMoveToBlinkSeq(0, linkedColor, 00);
    leds->addRgbMoveToBlinkSeq(0, linkedColor, ledsResetDelay);
    leds->addJumpToBlinkSeq(0, 1);
    leds->startBlinkSeq(0);

    leds->resetBlinkSeq(1);
    leds->addRgbMoveToBlinkSeq(1, notLinkedColor, 1000);
    leds->startBlinkSeq(1);
    lastLedsReset = GetTickCount64();
    leds->linkLedToBlinkSeq(1, 1, 0, 0);
  }
  catch (...) {}
}

// reset the LED watchdog
void YAudioOuputSwitcher::manageLeds()
{
  if (this->state < READY) return;
  if (GetTickCount64() - lastLedsReset >= (ledsResetDelay - 1500))
  {
    logs::logdebug("[L]");
    try
    {
      leds->resetBlinkSeq(0);
      leds->addRgbMoveToBlinkSeq(0, linkedColor, 00);
      leds->addRgbMoveToBlinkSeq(0, linkedColor, ledsResetDelay);
      leds->addJumpToBlinkSeq(0, 1);
      leds->startBlinkSeq(0);
      leds->linkLedToBlinkSeq(1, 1, 0, 0);
    }
    catch (...) {}
    lastLedsReset = GetTickCount64();

  }
}



// called eachtime a device is conencted. 
// will wait for a device with a network fonction,
// than makesure a Yocto-miniDisplay and a
// Yocto-knob is connected to the device hosting
// the network function
void YAudioOuputSwitcher::deviceArrival(YModule* m)
{
  if (this->state >= READY)  // lamost everything is already configred from a previous connection 
  {
    try { layer2->hide(); }
    catch (...) {}
    this->configure_ledWatchdog();
    lastLedsReset = 0;
    lastVolumeChange = 0;
    lastRefesh = 0;
    RefreshNeeded = TRUE;
    return;
  }

  int fctcount = m->functionCount();
  bool netFound = false;

  for (int i = 0; i < fctcount; i++)
  {  if (m->functionType(i) == "Network")  netFound = true;
  }

  if (!netFound) return;

  string serial = m->get_serialNumber();
  YHubPort* ports[3];
  try
  {
    ports[0] = YHubPort::FindHubPort(serial + ".hubPort1");
    ports[1] = YHubPort::FindHubPort(serial + ".hubPort2");
    ports[2] = YHubPort::FindHubPort(serial + ".hubPort3");
  }
  catch (...)
  {
    logs::log("No hub port found on " + m->get_friendlyName());
    return;
  }
  logs::log("found hub" + m->get_friendlyName());

  string DisplaySerial = "";
  string MaxiKnobSerial = "";
  for (int i = 0; i < 3; i++)
  {
    string portlLogicalname = ports[i]->get_logicalName();
    if (portlLogicalname.substr(0, 8) == "YMXBTN01")   MaxiKnobSerial = portlLogicalname;
    if (portlLogicalname.substr(0, 8) == "YD096X16")   DisplaySerial = portlLogicalname;
  }

  if (MaxiKnobSerial == "")
  {
    logs::log("No Yocto-MiniDisplay found on hub " + m->get_friendlyName());
    return;
  }

  if (DisplaySerial == "")
  {
    logs::log("No Yocto-MaxiKnob found on hub " + m->get_friendlyName());
    return;
  }

  logs::log("found Yocto-miniDisplay : " + DisplaySerial);
  logs::log("found Yocto-MaxiKnob : " + MaxiKnobSerial);

  string explanations = ", device must be configured with 2 quadrature decoders and 8 anbuttons";

  YQuadratureDecoder* OutQDdecoder;
  try
  {
    OutQDdecoder = YQuadratureDecoder::FindQuadratureDecoder(MaxiKnobSerial + ".quadratureDecoder1");
  }
  catch (...)
  {
    logs::log("No quadratureDecoder found on  " + MaxiKnobSerial + explanations);
    return;
  }
  if (!OutQDdecoder->isOnline())
  {
    logs::log("No  quadratureDecoder1 found on  " + MaxiKnobSerial + explanations);
    return;
  }

  YQuadratureDecoder* VolQDdecoder = YQuadratureDecoder::FindQuadratureDecoder(MaxiKnobSerial + ".quadratureDecoder2");
  if (!VolQDdecoder->isOnline())
  {
    logs::log("No  quadratureDecoder2 found on  " + MaxiKnobSerial + explanations);
    return;
  }

  OutQDdecoder->setCurrentValue(0);
  VolQDdecoder->setCurrentValue(0);

  YAnButton* MuteBtn;
  try
  {
    MuteBtn = YAnButton::FindAnButton(MaxiKnobSerial + ".anButton8");

  }
  catch (...)
  {
    logs::log("No anButton found on  " + MaxiKnobSerial + explanations);
    return;
  }
  if (!MuteBtn->isOnline())
  {
    logs::log("No  anButton8 found on  " + MaxiKnobSerial + explanations);
    return;
  }

  display = YDisplay::FindDisplay(DisplaySerial + ".display");
  if (!display->isOnline())
  { logs::log("strange, display is offline although is was detected a tad before");
    return;
  }

  display->resetAll();
  layer1 = display->get_displayLayer(1);
  layer2 = display->get_displayLayer(2);
  layer2->hide();

  leds = YColorLedCluster::FindColorLedCluster(MaxiKnobSerial + ".colorLedCluster");
  leds->set_activeLedCount(2);
  leds->set_ledType(YColorLedCluster::LEDTYPE_WS2811);

  vector<int>  configuredStartColor = leds->get_rgbColorArrayAtPowerOn(1, 1);
  if (configuredStartColor[0] != notLinkedColor)
  {
    leds->set_rgbColorAtPowerOn(1, 1, notLinkedColor);
    leds->get_module()->saveToFlash();
  }

  this->configure_ledWatchdog();

  this->manager = new AudioOutputManager(&change_global, this);

  OutQDdecoder->setUserData(this);
  VolQDdecoder->setUserData(this);
  MuteBtn->setUserData(this);
  OutQDdecoder->registerValueCallback(QD_out_change_global);
  VolQDdecoder->registerValueCallback(QD_Vol_change_global);
  MuteBtn->registerValueCallback(BTN_muteChange_global);

  logs::log("Audio Endpoints list:");
  int count = manager->get_count();
  for (int i = 0; i < count; i++)
    logs::log(" " + std::to_string(i + 1) + "-" + manager->get_FriendlyName(i));

  this->lastRefesh = 0;
  RefreshNeeded = TRUE;
  logs::log("All set, now running...");
  this->state = READY;

  return;

}

// refresh the display
void YAudioOuputSwitcher::refresh(AudioOutputManager* manager)
{
  if (GetTickCount64() - lastRefesh < 100) return;

  unsigned defaultAudioOutputIndex = manager->get_defaultOutput();
  if (defaultAudioOutputIndex < 0) return;
  bool mute = manager->get_mute();

  if ((lastVolumeChange != 0) && (GetTickCount64() - lastVolumeChange > 5000) && (!mute))
  { lastVolumeChange = 0;
    RefreshNeeded = TRUE;
  }

  if (!RefreshNeeded) return;
  logs::logdebug("[R]");

  try
  {

    string line1 = manager->get_FriendlyName(defaultAudioOutputIndex);
    string line2 = "";
    int p = (int)line1.find('(');
    if (p > 0)
    { line2 = line1.substr(p);
      line1 = line1.substr(0, p);
    }
    layer2->clear();

    layer2->drawText(0, 8, YDisplayLayer::ALIGN_BOTTOM_LEFT, line1);

    if ((lastVolumeChange == 0) || (mute))
    { layer2->drawText(0, 16, YDisplayLayer::ALIGN_BOTTOM_LEFT, line2);
    }

    else
    { float volume = manager->get_volume();
      layer2->drawRect(23, 10, 73, 15);
      layer2->drawBar(23, 10, (int)(23 + 50 * volume), 15);
    }

    if (mute)
    { lastVolumeChange = GetTickCount64();   
      layer2->selectColorPen(0x000000);
      layer2->drawBar(80, 0, 95, 15);
      layer2->selectColorPen(0xFFFFFF);
      layer2->moveTo(84, 5);
      layer2->lineTo(87, 5);
      layer2->lineTo(90, 2);
      layer2->lineTo(90, 13);
      layer2->lineTo(87, 10);
      layer2->lineTo(84, 10);
      layer2->lineTo(84, 5);
      layer2->moveTo(80, 0);
      layer2->lineTo(95, 15);
    }

    display->swapLayerContent(1, 2);
  }
  catch (...) {}

  lastRefesh = GetTickCount64();
  RefreshNeeded = FALSE;

}

// destructor

YAudioOuputSwitcher::~YAudioOuputSwitcher()
{
  if (this->manager != NULL) delete this->manager;
  this->manager = NULL;
  YAPI::FreeAPI();
}

// this is the business part, it must be called periodically
int YAudioOuputSwitcher::Run()
{

  string errmsg = "";
  logs::logdebug(".");
  YAPI::Sleep(100, errmsg);
  YAPI::UpdateDeviceList(errmsg);
  this->manageLeds();
  if (this->state >= READY)  refresh(manager);
  return 0;

}


